package io.nearpay.install.core.utils

import android.content.SharedPreferences
import io.nearpay.install.core.data.dto.ResponseInstallUrlDto
import io.nearpay.install.core.data.remote.ApiResponse
import io.nearpay.install.core.data.remote.ServiceApi
import io.nearpay.install.core.data.remote.safeApiCall
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import java.util.concurrent.TimeUnit

class UrlManager(
    private val sharedPreferences: SharedPreferences,
    private val serviceApi: ServiceApi,
    private val installUrlPath: String
) {

    companion object {
        private const val INSTALL_URL = "install_url"
        private const val INSTALL_FILE_NAME = "install_file_name"
        private const val INSTALL_TARGET_SDK = "install_target_sdk"
        private const val INSTALL_IS_FORCE_UPDATE = "install_is_force_update"
        private const val INSTALL_VERSION = "install_version"
        private const val INSTALL_URL_SAVE_TIME = "install_url_save_time"
    }

    private val _response = MutableStateFlow<ApiResponse<ResponseInstallUrlDto>?>(null)
    val response = _response.asStateFlow()

    suspend fun getInstallUrl(): ResponseInstallUrlDto? {
        return if (shouldFetchUrlFromServer()) {
            val fetchedUrlDto = fetchUrlFromServer(installUrlPath)
            if (fetchedUrlDto != null && isValidUrl(fetchedUrlDto.url)) {
                saveUrlToPrefs(fetchedUrlDto)
                fetchedUrlDto
            }
            null
        } else {
            val savedUrlDto = getUrlDtoFromPrefs()
            _response.value = ApiResponse.Success(savedUrlDto)
            savedUrlDto
        }
    }

    private fun saveUrlToPrefs(urlDto: ResponseInstallUrlDto?) {

        sharedPreferences.edit().apply {
            putString(INSTALL_URL, urlDto?.url)
            putString(INSTALL_FILE_NAME, urlDto?.file_name)
            putString(INSTALL_TARGET_SDK, urlDto?.target_sdk)
            urlDto?.is_force_update?.let { putBoolean(INSTALL_IS_FORCE_UPDATE, it) }
            urlDto?.version?.let { putLong(INSTALL_VERSION, it) }
            putLong(INSTALL_URL_SAVE_TIME, System.currentTimeMillis())
            apply()
        }
    }

    private fun getUrlDtoFromPrefs(): ResponseInstallUrlDto {
        val url = sharedPreferences.getString(INSTALL_URL, null) ?: ""
        val fileName = sharedPreferences.getString(INSTALL_FILE_NAME, null) ?: ""
        val targetSDK = sharedPreferences.getString(INSTALL_TARGET_SDK, null) ?: ""
        val isForceUpdate = sharedPreferences.getBoolean(INSTALL_IS_FORCE_UPDATE, false)
        val version = sharedPreferences.getLong(INSTALL_VERSION, 0)

        return ResponseInstallUrlDto(
            file_name = fileName,
            is_force_update = isForceUpdate,
            target_sdk = targetSDK,
            url = url,
            version = version
        )
    }

    private fun shouldFetchUrlFromServer(): Boolean {
        val savedTime = sharedPreferences.getLong(INSTALL_URL_SAVE_TIME, 0)
        if (savedTime == 0L) return true
        return System.currentTimeMillis() - savedTime > TimeUnit.HOURS.toMillis(1)
    }

    private suspend fun fetchUrlFromServer(installUrlPath: String): ResponseInstallUrlDto? {
        return safeApiCall {
            serviceApi.getInstallUrl(installUrlPath)
        }.let {
            _response.value = it

            when (it) {
                is ApiResponse.Error -> {
                    null
                }

                is ApiResponse.Success -> {
                    it.items
                }
            }
        }
    }
}