package io.nearpay.install.ui

import android.content.Context
import android.content.Intent
import android.content.pm.ActivityInfo
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.view.View
import android.view.WindowManager
import androidx.activity.OnBackPressedCallback
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.content.res.AppCompatResources
import androidx.core.content.ContextCompat
import androidx.core.view.WindowCompat
import androidx.core.view.isVisible
import androidx.navigation.fragment.NavHostFragment
import com.google.android.material.bottomsheet.BottomSheetBehavior
import dev.chrisbanes.insetter.applyInsetter
import io.nearpay.install.BuildConfig
import io.nearpay.install.R
import io.nearpay.install.core.data.dto.DataEnvironments
import io.nearpay.install.core.data.dto.HorizontalUI
import io.nearpay.install.core.data.dto.UIConfiguration
import io.nearpay.install.core.data.dto.VerticalUI
import io.nearpay.install.core.data.remote.ServiceApi
import io.nearpay.install.core.databinding.ActivityInstallPluginBinding
import io.nearpay.install.core.listener.InstallListener
import io.nearpay.install.core.listener.NearpayInstallListener
import io.nearpay.install.core.utils.WIDE_SCREENS
import io.nearpay.install.core.utils.getWidthDp
import io.nearpay.install.core.utils.textColor
import kotlinx.coroutines.runBlocking
import java.util.*

 class InstallActivity : AppCompatActivity(), NearpayInstallListener {

    companion object {

        private var listener: InstallListener? = null
        private var isSuccessfullyInstalled = false
        private var uiConfig: UIConfiguration? = null
        var environment: DataEnvironments? = null
        private var versionName: String = ""
        var installUrlPath: String = ""

        fun startActivity(
            context: Context?,
            environment: DataEnvironments,
            uiConfiguration: UIConfiguration,
            installListener: InstallListener,
            versionName: String,
            installUrlPath: String
        ) {
            listener = installListener
            isSuccessfullyInstalled = !installListener.isMustInstall()
            uiConfig = uiConfiguration
            Companion.versionName = versionName
            Companion.installUrlPath = installUrlPath
            Companion.environment = environment

            val intent = Intent(context, InstallActivity::class.java).apply {
                flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_NO_ANIMATION
            }
            context?.startActivity(intent)
        }
    }

    private lateinit var binding: ActivityInstallPluginBinding

    @RequiresApi(Build.VERSION_CODES.O)
    override fun onCreate(savedInstanceState: Bundle?) {
        requestedOrientation = if (Build.VERSION.SDK_INT != Build.VERSION_CODES.O) {
            ActivityInfo.SCREEN_ORIENTATION_LOCKED
        } else {
            ActivityInfo.SCREEN_ORIENTATION_NOSENSOR
        }
        super.onCreate(savedInstanceState)

        binding = ActivityInstallPluginBinding.inflate(layoutInflater)
        setContentView(binding.root)

        createUiConfigurations()

        binding.nearpayFooterLayout.nearpayFooterLayout.applyInsetter {
            type(navigationBars = true) { padding(bottom = true) }
        }

        binding.nearpayFooterLayout.poweredBy.text = getString(R.string.powered_by)
        binding.nearpayFooterLayout.nearpay.text = getString(R.string.nearpay_sdk)
        binding.nearpayFooterLayout.version.text = versionName

        binding.nearpayFooterLayout.installType.isVisible = true
        if (BuildConfig.DEBUG)
            binding.nearpayFooterLayout.installType.text = "NearPay Store - ${environment?.name}"
        else
            binding.nearpayFooterLayout.installType.text = "NearPay Store"


        if (environment?.name == DataEnvironments.SANDBOX.name || environment?.name == DataEnvironments.SANDBOX_V2.name ) {
            binding.nearpayFooterLayout.nearpayFooterLayout.setBackgroundColor(
                ContextCompat.getColor(
                    this,
                    R.color.error_red
                )
            )

            binding.nearpayFooterLayout.poweredBy.textColor(this, R.color.white)
            binding.nearpayFooterLayout.nearpay.textColor(this, R.color.white)
            binding.nearpayFooterLayout.version.textColor(this, R.color.white)
            binding.nearpayFooterLayout.installType.textColor(this, R.color.white)
        }

        WindowCompat.getInsetsController(window, window.decorView).isAppearanceLightStatusBars = true
        WindowCompat.setDecorFitsSystemWindows(window, false)
        window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)

        initBottomSheetBehavior()

        onBackPressedDispatcher.addCallback(
            this, // lifecycle owner
            object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    onBack()
                }
            })

        val navHostFragment =
            supportFragmentManager.findFragmentById(binding.fragmentContainer.id) as NavHostFragment
        val navController = navHostFragment.navController

        // setup graph with start destination
        val graph = navController.navInflater.inflate(R.navigation.install_nav_main)
        graph.setStartDestination(R.id.install_plugin_permission_fragment)
        navController.graph = graph
    }

    override fun attachBaseContext(newBase: Context) {
        val newLocale = listener?.getLocale()?: Locale.getDefault()
        Locale.setDefault(newLocale)
        val configuration = newBase.resources.configuration.apply {
            setLocale(newLocale)
            setLayoutDirection(newLocale)
        }
        super.attachBaseContext(newBase.applicationContext.createConfigurationContext(configuration))
    }

    override fun finish() {
        runBlocking {
            listener?.onInstallCompleted(isSuccessfullyInstalled)
        }
        overridePendingTransition(0, 0)
        super.finish()
    }

    private fun initBottomSheetBehavior() {
        val bottomSheetBehavior = BottomSheetBehavior.from(binding.activityContainer)

        // Expanded by default
        bottomSheetBehavior.state = BottomSheetBehavior.STATE_EXPANDED
        bottomSheetBehavior.skipCollapsed = true

        bottomSheetBehavior.addBottomSheetCallback(object :
            BottomSheetBehavior.BottomSheetCallback() {
            override fun onStateChanged(bottomSheet: View, newState: Int) {
                when (newState) {
                    BottomSheetBehavior.STATE_EXPANDED,
                    BottomSheetBehavior.STATE_COLLAPSED,
                    BottomSheetBehavior.STATE_DRAGGING,
                    BottomSheetBehavior.STATE_SETTLING,
                    BottomSheetBehavior.STATE_HALF_EXPANDED -> {
                    }

                    BottomSheetBehavior.STATE_HIDDEN -> {
                        finish()
                    }
                }
            }

            override fun onSlide(bottomSheet: View, slideOffset: Float) {}
        })
    }

    private fun onBack() {
        finish()
    }

    override fun getServiceApi(): ServiceApi? {
        return listener?.getServiceApi()
    }

    override fun updateValue(boolean: Boolean) {
        isSuccessfullyInstalled = boolean
    }

    private fun createUiConfigurations() {
        if (getWidthDp() >= WIDE_SCREENS) {

            val horizontalLayoutGravity = HorizontalUI.horizontalLayoutGravity(uiConfig?.horizontalUi?.id ?: HorizontalUI.DEFAULT.id)
            val verticalLayoutGravity = VerticalUI.verticalLayoutGravity(uiConfig?.verticalUi?.id ?: VerticalUI.DEFAULT.id)
            binding.activityContainer.gravity = horizontalLayoutGravity or verticalLayoutGravity

            when (uiConfig?.verticalUi?.id) {
                VerticalUI.TOP.id -> binding.mainLayout?.background =
                    AppCompatResources.getDrawable(this, R.drawable.round_corners_bottom)
                VerticalUI.BOTTOM.id -> binding.mainLayout?.background =
                    AppCompatResources.getDrawable(this, R.drawable.round_corners_top)
            }

            when (uiConfig?.horizontalUi?.id) {
                HorizontalUI.LEFT.id, HorizontalUI.START.id -> binding.mainLayout?.background =
                    AppCompatResources.getDrawable(this, R.drawable.round_corners_right)
                HorizontalUI.RIGHT.id, HorizontalUI.END.id -> binding.mainLayout?.background =
                    AppCompatResources.getDrawable(this, R.drawable.round_corners_left)
            }

        }
    }
}