package io.nearpay.install.ui

import android.annotation.SuppressLint
import android.content.Context
import android.content.SharedPreferences
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.graphics.Color
import android.os.Build
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.view.isVisible
import androidx.fragment.app.Fragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.lifecycleScope
import io.nearpay.install.R
import io.nearpay.install.core.InstallManager
import io.nearpay.install.core.data.dto.InstallationFailure
import io.nearpay.install.core.data.dto.NiceErrors
import io.nearpay.install.core.data.dto.ResponseInstallUrlDto
import io.nearpay.install.core.data.remote.ApiResponse
import io.nearpay.install.core.databinding.FragmentInstallPluginBinding
import io.nearpay.install.core.listener.InstallationListener
import io.nearpay.install.core.listener.NearpayInstallListener
import io.nearpay.install.core.utils.InstallState
import io.nearpay.install.core.utils.setOnSingleClickListener
import io.nearpay.install.viewModel.NearpayInstallViewModel
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

internal class InstallPluginFragment : Fragment(R.layout.fragment_install_plugin) {

    private lateinit var installManager: InstallManager

    private var _binding: FragmentInstallPluginBinding? = null
    private val binding get() = _binding!!

    private val viewModel: NearpayInstallViewModel by viewModels()

    private var activityListener: NearpayInstallListener? = null
            private val installationListener = object : InstallationListener {

        override fun onDownloadStart() {
            handleUi(InstallState.Downloading(0))
        }

        override fun onDownloadProgressChange(progress: Int) {
            handleUi(InstallState.Downloading(progress))
        }

        override fun onDownloadFailed() {
            handleUi(InstallState.Failed(NiceErrors.SDK_DOWNLOAD_FAILED.messageError))
        }

        override fun onInstallationStart() {
            handleUi(InstallState.Installing)
        }

        override fun onInstallationCompleted() {
            handleUi(InstallState.Completed)
        }

        override fun onInstallationFailed(installationError: InstallationFailure) {

            when (installationError) {
                is InstallationFailure.AlreadyInstalled ->
                    handleUi(InstallState.Failed(NiceErrors.SDK_ALREADY_INSTALLED.messageError))

                is InstallationFailure.DownloadedFileNotFound ->
                    handleUi(InstallState.Failed(NiceErrors.SDK_DOWNLOAD_FILE_NOT_FOUND.messageError))

                is InstallationFailure.InstallFailed ->
                    handleUi(InstallState.Failed(NiceErrors.SDK_MARKETPLACE_STORE_FAILED.messageError))

                is InstallationFailure.NetworkError ->
                    handleUi(InstallState.Failed(NiceErrors.NETWORK_ERROR.messageError))

                is InstallationFailure.UnsupportedSdkVersion ->
                    handleUi(InstallState.Failed(NiceErrors.SDK_UNSUPPORTED_ANDROID_SDK_VERSION.messageError))
            }
        }

    }

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View {
        _binding = FragmentInstallPluginBinding.inflate(inflater, container, false)
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        installManager = InstallManager()

        handleUi(InstallState.Ideal)
        startInstallProcess()
    }

    private fun startInstallProcess() {
        activityListener?.getServiceApi()?.run {
            viewModel.initializeUrlManager(getSharedPreferences(), this)
            viewModel.getUrl()
        } ?: run {
            requireActivity().finish()
        }

        lifecycleScope.launch {
            viewModel.response.collect { urlResponse ->
                onResponseReceived(urlResponse)
            }
        }
    }

    private fun restartInstallProcess() {
        lifecycleScope.launch {
            handleUi(InstallState.Ideal)
            delay(1000)
            startInstallProcess()
        }
    }

    private fun onResponseReceived(response: ApiResponse<ResponseInstallUrlDto>?) {
        when (response) {
            is ApiResponse.Success -> {

                //Update file name
                binding.pluginStateLayout.paymentPluginTv.text = response.items.file_name

                handleUi(
                    InstallState.Start(
                        startInstallationAction = {
                            this@InstallPluginFragment.installManager.start(
                                requireContext().applicationContext,
                                response.items.url,
                                response.items.file_name,
                                installationListener
                            )
                        },
                        version = response.items.version.toString()
                    )
                )
            }

            is ApiResponse.Error -> {
                handleUi(InstallState.Failed(response.requestException.serverErrorMessage))
            }

            null -> {
                handleUi(InstallState.Failed(NiceErrors.SDK_DOWNLOAD_FAILED.messageError))
            }
        }
    }

    @SuppressLint("SetTextI18n")
    private fun handleUi(state: InstallState) = lifecycleScope.launchWhenResumed {

        fun clearState() {
            binding.titleTextview.text = String()
            binding.actionBtn.isVisible = false
            binding.cancelButtonBtn.visibility = View.INVISIBLE

            val isVersionExist = binding.pluginStateLayout.versionStateTv.text.isNotBlank()
            binding.pluginStateLayout.versionStateTv.isVisible = isVersionExist

            binding.pluginStateLayout.downloadGroup.isVisible = false
            binding.pluginStateLayout.generalProgressBar.isVisible = false
        }

        fun showPaymentPluginInfo(visible: Boolean) {
            binding.cardView.isVisible = visible
            binding.pluginStateLayout.root.isVisible = visible
            binding.errorDialogLayout.isVisible = !visible

        }

        when (state) {
            is InstallState.Ideal -> {
                clearState()

                showPaymentPluginInfo(true)
                binding.cardView.visibility = View.VISIBLE
                binding.pluginStateLayout.root.visibility = View.VISIBLE
                binding.titleTextview.text = getString(R.string.loading)
                binding.pluginStateLayout.generalProgressBar.isVisible = true
            }

            is InstallState.Start -> {
                clearState()

                showPaymentPluginInfo(true)
                binding.titleTextview.text = getString(R.string.install)
                binding.pluginStateLayout.versionStateTv.isVisible = true
                binding.pluginStateLayout.versionStateTv.text = getString(R.string.version, state.version)
                binding.actionBtn.isVisible = true
                binding.actionBtn.text = getString(R.string.install)

                if (isPackageExist() && (installedPluginVersion() < state.version.toLong())) {
                    binding.message.text = getString(R.string.update_message)
                    binding.titleTextview.text = getString(R.string.update)
                    binding.actionBtn.text = getString(R.string.update)
                }

                binding.actionBtn.setOnSingleClickListener {
                    state.startInstallationAction()
                }
            }

            is InstallState.Downloading -> {
                clearState()
                binding.titleTextview.text = getString(R.string.downloading)

                showPaymentPluginInfo(true)
                binding.pluginStateLayout.downloadGroup.isVisible = true
                binding.pluginStateLayout.downloadProgressBar.progress = state.progress
                binding.pluginStateLayout.downloadPercentageTextView.text = "% ${state.progress}"
                binding.cancelButtonBtn.isVisible = true
                binding.actionBtn.visibility = View.INVISIBLE
                binding.cancelButtonBtn.text = getString(R.string.cancel)
                binding.cancelButtonBtn.setOnClickListener {
                    installManager.cancelDownload()
                    requireActivity().finish()
                }
            }

            is InstallState.Installing -> {
                clearState()

                showPaymentPluginInfo(true)

                binding.errorDialogLayout.visibility = View.GONE
                binding.titleTextview.text = getString(R.string.installing)
                binding.pluginStateLayout.generalProgressBar.isVisible = true
            }

            is InstallState.Completed -> {
                activityListener?.updateValue(true)
                clearState()

                binding.titleTextview.text = getString(R.string.completed)
                binding.errorDialogLayout.visibility = View.GONE

                binding.actionBtn.isVisible = true
                binding.actionBtn.text = getString(R.string.done)
                binding.actionBtn.setOnClickListener {
                    requireActivity().finish()
                }
            }

            is InstallState.Failed -> {
                clearState()

                binding.titleTextview.text = getString(R.string.failed)

                binding.actionBtn.isVisible = true
                binding.actionBtn.text = getString(R.string.retry)
                binding.actionBtn.setOnClickListener {
                    restartInstallProcess()
                }

                binding.layoutErrorDialog.apply {
                    errorText.text = state.serverErrorMessage.error.english
                    sourceTextview.text = state.serverErrorMessage.source.english
                    codeTextview.text = state.serverErrorMessage.code.toString()
                    solutionTextview.text = state.serverErrorMessage.solution.english
                    errorSourceCard.setCardBackgroundColor(Color.parseColor(state.serverErrorMessage.source.color))
                }
                showPaymentPluginInfo(false)

            }
        }
    }

    private fun isPackageExist(): Boolean = pluginPackage() != null

    private fun pluginPackage(): PackageInfo? {
        val packageName = InstallActivity.environment?.pluginPackageName ?: return null

        return try {
            requireContext().packageManager.getPackageInfo(packageName, 0)
        } catch (e: PackageManager.NameNotFoundException) {
            null
        }
    }

    private fun installedPluginVersion(): Long {
        return pluginPackage()?.let {
            @Suppress("DEPRECATION")
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P)
                it.longVersionCode
            else
                it.versionCode.toLong()
        } ?: -1
    }

    private fun getSharedPreferences(): SharedPreferences {
        return requireContext().getSharedPreferences(
            "nearpay-sdk-shared-preferences",
            Context.MODE_PRIVATE
        )
    }

    override fun onAttach(context: Context) {
        super.onAttach(context)
        if (context is NearpayInstallListener)
            activityListener = context
        else
            throw Throwable("$context must implement ${NearpayInstallListener::class.java.name}")
    }

    override fun onDetach() {
        super.onDetach()
        activityListener = null
    }
}