// Copyright 2015-2024 Nstream, inc.
//
// Licensed under the Redis Source Available License 2.0 (RSALv2) Agreement;
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://redis.com/legal/rsalv2-agreement/
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package nstream.adapter.geo;

import nstream.adapter.common.patches.SummaryPatch;
import swim.api.SwimLane;
import swim.api.lane.CommandLane;
import swim.api.lane.ValueLane;
import swim.structure.Record;
import swim.structure.Selector;
import swim.structure.Value;
import swim.uri.Uri;
import swim.uri.UriPattern;

/**
 * Patch to manage agents with longitude and latitude.
 * Automatically joins agents to appropriate {@link MapTilePatch} agents.
 */
public class GeoPointPatch extends SummaryPatch {

  protected static final String GEO_LATITUDE_KEY = "latitude";
  protected static final String GEO_LONGITUDE_KEY = "longitude";
  protected static final String GEO_VALUE_KEY = "value";

  protected static final Uri UPDATE_AGENT_LANE_URI = Uri.parse("updateAgent");

  protected Uri currentMapTile;

  public GeoPointPatch() {
  }

  /**
   * Extract the value to be stored in the {@link GeoPointPatch#geo} lane from an {@link GeoPointPatch#addEvent} event.
   * This can be configured in the agent properties by passing a {@link Selector} to 'extractValueFromEvent'.
   *
   * @param event raw event
   * @return value to be stored in lane
   */
  protected Value extractValueFromEvent(final Value event) {
    final Value extractDef = getProp("extractValueFromEvent");
    if (extractDef.isDistinct() && extractDef instanceof Selector) {
      return extractDef.evaluate(event).toValue();
    } else {
      return defaultExtractValueFromEvent(event);
    }
  }

  /**
   * Extract the value to be stored in the {@link GeoPointPatch#geo} lane from an {@link GeoPointPatch#addEvent} event.
   * The default behaviour is to return the raw event if no 'extractValueFromEvent' agent property is set.
   *
   * @param event raw event
   * @return default value to be stored in lane
   */
  protected Value defaultExtractValueFromEvent(final Value event) {
    return Value.absent();
  }

  /**
   * Extract the longitude from an {@link GeoPointPatch#addEvent} event.
   * This can be configured in the agent properties by passing a {@link Selector} to 'extractLongitude'.
   *
   * @param event raw event
   * @return longitude
   */
  protected double extractLongitudeFromEvent(final Value event) {
    final Value extractDef = getProp("extractLongitude");
    if (extractDef.isDistinct() && extractDef instanceof Selector) {
      return extractDef.evaluate(event).doubleValue(Double.MAX_VALUE);
    } else {
      return defaultExtractLongitudeFromEvent(event);
    }
  }

  /**
   * Extract the longitude from an {@link GeoPointPatch#addEvent} event.
   * The default behaviour is to extract value for key 'longitude' if no 'extractLongitude' agent property is set.
   *
   * @param event raw event
   * @return longitude
   */
  protected double defaultExtractLongitudeFromEvent(final Value event) {
    return event.get("longitude").doubleValue(Double.MAX_VALUE);
  }

  /**
   * Extract the latitude from an {@link GeoPointPatch#addEvent} event.
   * This can be configured in the agent properties by passing a {@link Selector} to 'extractLatitude'.
   *
   * @param event raw event
   * @return latitude
   */
  protected double extractLatitudeFromEvent(final Value event) {
    final Value extractDef = getProp("extractLatitude");
    if (extractDef.isDistinct() && extractDef instanceof Selector) {
      return extractDef.evaluate(event).doubleValue(Double.MAX_VALUE);
    } else {
      return defaultExtractLatitudeFromEvent(event);
    }
  }

  /**
   * Extract the latitude from an {@link GeoPointPatch#addEvent} event.
   * The default behaviour is to extract value for key 'latitude' if no 'extractLatitude' agent property is set.
   *
   * @param event raw event
   * @return latitude
   */
  protected double defaultExtractLatitudeFromEvent(final Value event) {
    return event.get("latitude").doubleValue(Double.MAX_VALUE);
  }

  /**
   * Get the value for 'longitude' set in the agent properties.
   *
   * @return 'longitude' from agent properties
   */
  protected double longitude() {
    return getProp("longitude").doubleValue(Double.MAX_VALUE);
  }

  /**
   * Get the value for 'latitude' set in the agent properties.
   *
   * @return 'latitude' from agent properties
   */
  protected double latitude() {
    return getProp("latitude").doubleValue(Double.MAX_VALUE);
  }

  /**
   * Get the value for 'minAgentZoom' set in the agent properties.
   * This is the minimum {@link MapTilePatch#tileZ} for which this agent will appear.
   * Default to 0
   *
   * @return 'minAgentZoom' from agent properties
   */
  protected int minAgentZoom() {
    return getProp("minAgentZoom").intValue(0);
  }

  /**
   * Get the value for 'maxAgentZoom' set in the agent properties.
   * This is the maximum {@link MapTilePatch#tileZ} for which this agent will appear.
   * Default to 20
   *
   * @return 'minAgentZoom' from agent properties
   */
  protected int maxAgentZoom() {
    return getProp("maxAgentZoom").intValue(20);
  }

  /**
   * Get the value for 'mapTileUriPattern' set in the agent properties.
   * This is the node uri pattern for the {@link MapTilePatch} agents.
   * Default to '/map/:tile'
   *
   * @return 'mapTileUriPattern' from agent properties
   */
  protected UriPattern mapTileUriPattern() {
    return UriPattern.parse(getProp("mapTileUriPattern").stringValue("/map/:tile"));
  }

  @SwimLane("addEvent")
  protected CommandLane<Value> addEvent = this.<Value>commandLane()
          .onCommand(v -> {
            trace("(GeoPointPatch) " + nodeUri()
                    + ".addEvent#onCommand(): value=" + v);
            this.geo.set(extractGeoFromEvent(v));
          });

  protected Value extractGeoFromEvent(final Value event) {
    final Record geoValue = Record.create(3);

    final Value value = extractValueFromEvent(event);
    if (value.isDefined()) {
      geoValue.slot(GEO_VALUE_KEY, value);
    }

    final double longitude = extractLongitudeFromEvent(event);
    if (longitude != Double.MAX_VALUE) {
      geoValue.slot(GEO_LONGITUDE_KEY, longitude);
    }
    final double latitude = extractLatitudeFromEvent(event);
    if (latitude != Double.MAX_VALUE) {
      geoValue.slot(GEO_LATITUDE_KEY, latitude);
    }

    return geoValue;
  }

  @SwimLane("geo")
  protected ValueLane<Value> geo = this.<Value>valueLane()
          .didSet((n, o) -> {
            trace("(GeoPointPatch) " + nodeUri()
                    + ".geo#didSet(): newValue=" + n + ", oldValue=" + o);
            updateMapTile(n, o);
          });

  /**
   * Update the map tile if required when {@link GeoPointPatch#geo} is updated.
   * If longitude or latitude changes then leave the current {@link MapTilePatch} and join the new one.
   *
   * @param newValue new value of lane
   * @param oldValue old value of lane
   */
  protected void updateMapTile(final Value newValue, final Value oldValue) {
    final double newLongitude = newValue.get(GEO_LONGITUDE_KEY).doubleValue(Double.MAX_VALUE);
    final double newLatitude = newValue.get(GEO_LATITUDE_KEY).doubleValue(Double.MAX_VALUE);

    final Uri newMapTileUri = calculateMapTileUri(newLongitude, newLatitude);

    if (newMapTileUri == null || !newMapTileUri.equals(this.currentMapTile)) {
      final Value payload = createMapTilePayload(newLongitude, newLatitude);
      relayToMapTile(this.currentMapTile, payload);
      relayToMapTile(newMapTileUri, payload);
      this.currentMapTile = newMapTileUri;
    }
  }

  /**
   * Create the value to send to the {@link MapTilePatch#updateAgent} lane.
   *
   * @return value to send
   */
  protected Value createMapTilePayload(final double longitude, final double latitude) {
    return Record.create(4)
            .slot("uri", Uri.form().mold(nodeUri()).toValue())
            .slot("minAgentZoom", minAgentZoom())
            .slot("longitude", longitude)
            .slot("latitude", latitude);
  }

  protected void relayToMapTile(final Uri mapTileUri, final Value value) {
    if (mapTileUri != null) {
      command(mapTileUri, UPDATE_AGENT_LANE_URI, value);
    }
  }

  /**
   * Given a longitude and latitude, construct the {@link MapTilePatch} node uri.
   *
   * @param longitude longitude
   * @param latitude latitude
   * @return map tile node uri
   */
  protected Uri calculateMapTileUri(final double longitude, final double latitude) {
    if (longitude == Double.MAX_VALUE || latitude == Double.MAX_VALUE) {
      return null;
    }
    final double x = SphericalMercator.projectLng(longitude);
    final double y = SphericalMercator.projectLat(latitude);
    final int tileX = (int) (x * (double) (1 << maxAgentZoom()));
    final int tileY = (int) (y * (double) (1 << maxAgentZoom()));
    final int tileZ = maxAgentZoom();
    return mapTileUriPattern().apply(tileX + "," + tileY + "," + tileZ);
  }

  @Override
  public void didStart() {
    info("(GeoPointPatch) " + nodeUri() + ": didStart");
    seedGeoFromProps();
  }

  protected void seedGeoFromProps() {
    final double longitude = longitude();
    if (longitude != Double.MAX_VALUE) {
      this.geo.set(
              this.geo.get().updated(GEO_LONGITUDE_KEY, longitude)
      );
    }
    final double latitude = latitude();
    if (latitude != Double.MAX_VALUE) {
      this.geo.set(
              this.geo.get().updated(GEO_LATITUDE_KEY, latitude)
      );
    }
  }

}
