// Copyright 2015-2024 Nstream, inc.
//
// Licensed under the Redis Source Available License 2.0 (RSALv2) Agreement;
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://redis.com/legal/rsalv2-agreement/
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package nstream.adapter.geo;

import nstream.adapter.common.patches.SummaryPatch;
import swim.api.SwimLane;
import swim.api.lane.CommandLane;
import swim.api.lane.JoinValueLane;
import swim.structure.Value;
import swim.uri.Uri;

public class MapTilePatch extends SummaryPatch {

  private static final Uri UPDATE_AGENT_LANE_URI = Uri.parse("updateAgent");
  private static final Uri GEO_LANE_URI = Uri.parse("geo");

  protected int tileX;
  protected int tileY;
  protected int tileZ;

  protected Uri parentTileUri;

  public MapTilePatch() {
  }

  protected Value extractGeoPointKeyFromValue(final Value value) {
    return value.get("uri");
  }

  protected int extractMinAgentZoomFromValue(final Value value) {
    return value.get("minAgentZoom").intValue(0);
  }

  protected double extractLongitudeFromValue(final Value value) {
    return value.get("longitude").doubleValue(Double.MAX_VALUE);
  }

  protected double extractLatitudeFromValue(final Value value) {
    return value.get("latitude").doubleValue(Double.MAX_VALUE);
  }

  @SwimLane("agents")
  protected JoinValueLane<Value, Value> agents = this.<Value, Value>joinValueLane()
          .didUpdate((k, n, o) -> {
            trace("(MapTilePatch) " + nodeUri() + ".agents#didUpdate(): "
                    + "key=" + k + ", newValue=" + n + ", oldValue=" + o);
          });

  @SwimLane("updateAgent")
  protected CommandLane<Value> updateAgent = this.<Value>commandLane()
          .onCommand(v -> {
            if (this.tileZ >= extractMinAgentZoomFromValue(v)) {
              trace("(MapTilePatch) " + nodeUri()
                      + ".updateAgent#onCommand(): value=" + v);
              if (processGeoPointUpdate(v)) {
                relayToParent(v);
              }
            }
          });

  protected boolean processGeoPointUpdate(final Value value) {
    final Value key = extractGeoPointKeyFromValue(value);
    final double longitude = extractLongitudeFromValue(value);
    final double latitude = extractLatitudeFromValue(value);

    if (longitude == Double.MAX_VALUE || latitude == Double.MAX_VALUE) {
      // No longitude or latitude so this point can not belong to this tile
      return removeGeoPoint(key);
    }

    final double x = SphericalMercator.projectLng(longitude);
    final double y = SphericalMercator.projectLat(latitude);
    final int tileX = (int) (x * (double) (1 << this.tileZ));
    final int tileY = (int) (y * (double) (1 << this.tileZ));

    if (tileX == this.tileX && tileY == this.tileY) {
      return downlinkGeoPoint(key);
    }

    return removeGeoPoint(key);
  }

  protected boolean downlinkGeoPoint(final Value key) {
    if (!this.agents.containsKey(key)) {
      this.agents.downlink(key)
              .nodeUri(Uri.form().cast(key))
              .laneUri(GEO_LANE_URI)
              .open();
      return true;
    }
    return false;
  }

  protected boolean removeGeoPoint(final Value key) {
    if (this.agents.containsKey(key)) {
      this.agents.remove(key);
      return true;
    }
    return false;
  }

  protected void relayToParent(final Value value) {
    command(this.parentTileUri, UPDATE_AGENT_LANE_URI, value);
  }

  @Override
  public void didStart() {
    info("(MapTilePatch) " + nodeUri() + ": didStart");
    setXYZ();
    setParentTile();
  }

  protected void setXYZ() {
    final String[] coordinates = nodeUri().path().foot().toString().split(",");
    this.tileX = Integer.parseInt(coordinates[0]);
    this.tileY = Integer.parseInt(coordinates[1]);
    this.tileZ = Integer.parseInt(coordinates[2]);
  }

  protected void setParentTile() {
    final int parentTileX = this.tileX / 2;
    final int parentTileY = this.tileY / 2;
    final int parentTileZ = this.tileZ - 1;
    parentTileUri = Uri.parse(nodeUri().base().body() + "/" + parentTileX + "," + parentTileY + "," + parentTileZ);
  }

}
