/*
 * Copyright 2019 California Institute of Technology (\"Caltech\").
 * U.S. Government sponsorship acknowledged.
 *
 * Licensed under the Apache License, Version 2.0 (the \"License\");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an \"AS IS\" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.opencaesar.oml.dsl.serializer;

import com.google.inject.Inject;
import io.opencaesar.oml.Annotation;
import io.opencaesar.oml.AnnotationProperty;
import io.opencaesar.oml.AnnotationPropertyReference;
import io.opencaesar.oml.Aspect;
import io.opencaesar.oml.AspectReference;
import io.opencaesar.oml.BooleanLiteral;
import io.opencaesar.oml.Concept;
import io.opencaesar.oml.ConceptInstance;
import io.opencaesar.oml.ConceptInstanceReference;
import io.opencaesar.oml.ConceptReference;
import io.opencaesar.oml.ConceptTypeAssertion;
import io.opencaesar.oml.DecimalLiteral;
import io.opencaesar.oml.Description;
import io.opencaesar.oml.DescriptionBundle;
import io.opencaesar.oml.DescriptionBundleExtension;
import io.opencaesar.oml.DescriptionBundleInclusion;
import io.opencaesar.oml.DescriptionBundleUsage;
import io.opencaesar.oml.DescriptionExtension;
import io.opencaesar.oml.DescriptionUsage;
import io.opencaesar.oml.DifferentFromPredicate;
import io.opencaesar.oml.DoubleLiteral;
import io.opencaesar.oml.EnumeratedScalar;
import io.opencaesar.oml.EnumeratedScalarReference;
import io.opencaesar.oml.FacetedScalar;
import io.opencaesar.oml.FacetedScalarReference;
import io.opencaesar.oml.FeaturePredicate;
import io.opencaesar.oml.ForwardRelation;
import io.opencaesar.oml.IntegerLiteral;
import io.opencaesar.oml.KeyAxiom;
import io.opencaesar.oml.LinkAssertion;
import io.opencaesar.oml.OmlPackage;
import io.opencaesar.oml.QuotedLiteral;
import io.opencaesar.oml.RelationCardinalityRestrictionAxiom;
import io.opencaesar.oml.RelationEntity;
import io.opencaesar.oml.RelationEntityPredicate;
import io.opencaesar.oml.RelationEntityReference;
import io.opencaesar.oml.RelationInstance;
import io.opencaesar.oml.RelationInstanceReference;
import io.opencaesar.oml.RelationRangeRestrictionAxiom;
import io.opencaesar.oml.RelationReference;
import io.opencaesar.oml.RelationTargetRestrictionAxiom;
import io.opencaesar.oml.RelationTypeAssertion;
import io.opencaesar.oml.ReverseRelation;
import io.opencaesar.oml.Rule;
import io.opencaesar.oml.RuleReference;
import io.opencaesar.oml.SameAsPredicate;
import io.opencaesar.oml.ScalarProperty;
import io.opencaesar.oml.ScalarPropertyCardinalityRestrictionAxiom;
import io.opencaesar.oml.ScalarPropertyRangeRestrictionAxiom;
import io.opencaesar.oml.ScalarPropertyReference;
import io.opencaesar.oml.ScalarPropertyValueAssertion;
import io.opencaesar.oml.ScalarPropertyValueRestrictionAxiom;
import io.opencaesar.oml.SpecializationAxiom;
import io.opencaesar.oml.Structure;
import io.opencaesar.oml.StructureInstance;
import io.opencaesar.oml.StructureReference;
import io.opencaesar.oml.StructuredProperty;
import io.opencaesar.oml.StructuredPropertyCardinalityRestrictionAxiom;
import io.opencaesar.oml.StructuredPropertyRangeRestrictionAxiom;
import io.opencaesar.oml.StructuredPropertyReference;
import io.opencaesar.oml.StructuredPropertyValueAssertion;
import io.opencaesar.oml.StructuredPropertyValueRestrictionAxiom;
import io.opencaesar.oml.TypePredicate;
import io.opencaesar.oml.Vocabulary;
import io.opencaesar.oml.VocabularyBundle;
import io.opencaesar.oml.VocabularyBundleExtension;
import io.opencaesar.oml.VocabularyBundleInclusion;
import io.opencaesar.oml.VocabularyExtension;
import io.opencaesar.oml.VocabularyUsage;
import io.opencaesar.oml.dsl.services.OmlGrammarAccess;
import java.util.Set;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.xtext.Action;
import org.eclipse.xtext.Parameter;
import org.eclipse.xtext.ParserRule;
import org.eclipse.xtext.serializer.ISerializationContext;
import org.eclipse.xtext.serializer.acceptor.SequenceFeeder;
import org.eclipse.xtext.serializer.sequencer.AbstractDelegatingSemanticSequencer;
import org.eclipse.xtext.serializer.sequencer.ITransientValueService.ValueTransient;

@SuppressWarnings("all")
public class OmlSemanticSequencer extends AbstractDelegatingSemanticSequencer {

	@Inject
	private OmlGrammarAccess grammarAccess;
	
	@Override
	public void sequence(ISerializationContext context, EObject semanticObject) {
		EPackage epackage = semanticObject.eClass().getEPackage();
		ParserRule rule = context.getParserRule();
		Action action = context.getAssignedAction();
		Set<Parameter> parameters = context.getEnabledBooleanParameters();
		if (epackage == OmlPackage.eINSTANCE)
			switch (semanticObject.eClass().getClassifierID()) {
			case OmlPackage.ANNOTATION:
				sequence_Annotation(context, (Annotation) semanticObject); 
				return; 
			case OmlPackage.ANNOTATION_PROPERTY:
				sequence_AnnotationProperty(context, (AnnotationProperty) semanticObject); 
				return; 
			case OmlPackage.ANNOTATION_PROPERTY_REFERENCE:
				sequence_AnnotationPropertyReference(context, (AnnotationPropertyReference) semanticObject); 
				return; 
			case OmlPackage.ASPECT:
				sequence_Aspect(context, (Aspect) semanticObject); 
				return; 
			case OmlPackage.ASPECT_REFERENCE:
				sequence_AspectReference(context, (AspectReference) semanticObject); 
				return; 
			case OmlPackage.BOOLEAN_LITERAL:
				sequence_BooleanLiteral(context, (BooleanLiteral) semanticObject); 
				return; 
			case OmlPackage.CONCEPT:
				sequence_Concept(context, (Concept) semanticObject); 
				return; 
			case OmlPackage.CONCEPT_INSTANCE:
				sequence_ConceptInstance(context, (ConceptInstance) semanticObject); 
				return; 
			case OmlPackage.CONCEPT_INSTANCE_REFERENCE:
				sequence_ConceptInstanceReference(context, (ConceptInstanceReference) semanticObject); 
				return; 
			case OmlPackage.CONCEPT_REFERENCE:
				sequence_ConceptReference(context, (ConceptReference) semanticObject); 
				return; 
			case OmlPackage.CONCEPT_TYPE_ASSERTION:
				sequence_ConceptTypeAssertion(context, (ConceptTypeAssertion) semanticObject); 
				return; 
			case OmlPackage.DECIMAL_LITERAL:
				sequence_DecimalLiteral(context, (DecimalLiteral) semanticObject); 
				return; 
			case OmlPackage.DESCRIPTION:
				sequence_Description(context, (Description) semanticObject); 
				return; 
			case OmlPackage.DESCRIPTION_BUNDLE:
				sequence_DescriptionBundle(context, (DescriptionBundle) semanticObject); 
				return; 
			case OmlPackage.DESCRIPTION_BUNDLE_EXTENSION:
				sequence_DescriptionBundleExtension(context, (DescriptionBundleExtension) semanticObject); 
				return; 
			case OmlPackage.DESCRIPTION_BUNDLE_INCLUSION:
				sequence_DescriptionBundleInclusion(context, (DescriptionBundleInclusion) semanticObject); 
				return; 
			case OmlPackage.DESCRIPTION_BUNDLE_USAGE:
				sequence_DescriptionBundleUsage(context, (DescriptionBundleUsage) semanticObject); 
				return; 
			case OmlPackage.DESCRIPTION_EXTENSION:
				sequence_DescriptionExtension(context, (DescriptionExtension) semanticObject); 
				return; 
			case OmlPackage.DESCRIPTION_USAGE:
				sequence_DescriptionUsage(context, (DescriptionUsage) semanticObject); 
				return; 
			case OmlPackage.DIFFERENT_FROM_PREDICATE:
				sequence_DifferentFromPredicate(context, (DifferentFromPredicate) semanticObject); 
				return; 
			case OmlPackage.DOUBLE_LITERAL:
				sequence_DoubleLiteral(context, (DoubleLiteral) semanticObject); 
				return; 
			case OmlPackage.ENUMERATED_SCALAR:
				sequence_EnumeratedScalar(context, (EnumeratedScalar) semanticObject); 
				return; 
			case OmlPackage.ENUMERATED_SCALAR_REFERENCE:
				sequence_EnumeratedScalarReference(context, (EnumeratedScalarReference) semanticObject); 
				return; 
			case OmlPackage.FACETED_SCALAR:
				sequence_FacetedScalar(context, (FacetedScalar) semanticObject); 
				return; 
			case OmlPackage.FACETED_SCALAR_REFERENCE:
				sequence_FacetedScalarReference(context, (FacetedScalarReference) semanticObject); 
				return; 
			case OmlPackage.FEATURE_PREDICATE:
				sequence_FeaturePredicate(context, (FeaturePredicate) semanticObject); 
				return; 
			case OmlPackage.FORWARD_RELATION:
				sequence_ForwardRelation(context, (ForwardRelation) semanticObject); 
				return; 
			case OmlPackage.INTEGER_LITERAL:
				sequence_IntegerLiteral(context, (IntegerLiteral) semanticObject); 
				return; 
			case OmlPackage.KEY_AXIOM:
				sequence_KeyAxiom(context, (KeyAxiom) semanticObject); 
				return; 
			case OmlPackage.LINK_ASSERTION:
				sequence_LinkAssertion(context, (LinkAssertion) semanticObject); 
				return; 
			case OmlPackage.QUOTED_LITERAL:
				sequence_QuotedLiteral(context, (QuotedLiteral) semanticObject); 
				return; 
			case OmlPackage.RELATION_CARDINALITY_RESTRICTION_AXIOM:
				sequence_RelationCardinalityRestrictionAxiom(context, (RelationCardinalityRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.RELATION_ENTITY:
				sequence_RelationEntity(context, (RelationEntity) semanticObject); 
				return; 
			case OmlPackage.RELATION_ENTITY_PREDICATE:
				sequence_RelationEntityPredicate(context, (RelationEntityPredicate) semanticObject); 
				return; 
			case OmlPackage.RELATION_ENTITY_REFERENCE:
				sequence_RelationEntityReference(context, (RelationEntityReference) semanticObject); 
				return; 
			case OmlPackage.RELATION_INSTANCE:
				sequence_RelationInstance(context, (RelationInstance) semanticObject); 
				return; 
			case OmlPackage.RELATION_INSTANCE_REFERENCE:
				sequence_RelationInstanceReference(context, (RelationInstanceReference) semanticObject); 
				return; 
			case OmlPackage.RELATION_RANGE_RESTRICTION_AXIOM:
				sequence_RelationRangeRestrictionAxiom(context, (RelationRangeRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.RELATION_REFERENCE:
				sequence_RelationReference(context, (RelationReference) semanticObject); 
				return; 
			case OmlPackage.RELATION_TARGET_RESTRICTION_AXIOM:
				sequence_RelationTargetRestrictionAxiom(context, (RelationTargetRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.RELATION_TYPE_ASSERTION:
				sequence_RelationTypeAssertion(context, (RelationTypeAssertion) semanticObject); 
				return; 
			case OmlPackage.REVERSE_RELATION:
				sequence_ReverseRelation(context, (ReverseRelation) semanticObject); 
				return; 
			case OmlPackage.RULE:
				sequence_Rule(context, (Rule) semanticObject); 
				return; 
			case OmlPackage.RULE_REFERENCE:
				sequence_RuleReference(context, (RuleReference) semanticObject); 
				return; 
			case OmlPackage.SAME_AS_PREDICATE:
				sequence_SameAsPredicate(context, (SameAsPredicate) semanticObject); 
				return; 
			case OmlPackage.SCALAR_PROPERTY:
				sequence_ScalarProperty(context, (ScalarProperty) semanticObject); 
				return; 
			case OmlPackage.SCALAR_PROPERTY_CARDINALITY_RESTRICTION_AXIOM:
				sequence_ScalarPropertyCardinalityRestrictionAxiom(context, (ScalarPropertyCardinalityRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.SCALAR_PROPERTY_RANGE_RESTRICTION_AXIOM:
				sequence_ScalarPropertyRangeRestrictionAxiom(context, (ScalarPropertyRangeRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.SCALAR_PROPERTY_REFERENCE:
				sequence_ScalarPropertyReference(context, (ScalarPropertyReference) semanticObject); 
				return; 
			case OmlPackage.SCALAR_PROPERTY_VALUE_ASSERTION:
				sequence_ScalarPropertyValueAssertion(context, (ScalarPropertyValueAssertion) semanticObject); 
				return; 
			case OmlPackage.SCALAR_PROPERTY_VALUE_RESTRICTION_AXIOM:
				sequence_ScalarPropertyValueRestrictionAxiom(context, (ScalarPropertyValueRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.SPECIALIZATION_AXIOM:
				sequence_SpecializationAxiom(context, (SpecializationAxiom) semanticObject); 
				return; 
			case OmlPackage.STRUCTURE:
				sequence_Structure(context, (Structure) semanticObject); 
				return; 
			case OmlPackage.STRUCTURE_INSTANCE:
				sequence_StructureInstance(context, (StructureInstance) semanticObject); 
				return; 
			case OmlPackage.STRUCTURE_REFERENCE:
				sequence_StructureReference(context, (StructureReference) semanticObject); 
				return; 
			case OmlPackage.STRUCTURED_PROPERTY:
				sequence_StructuredProperty(context, (StructuredProperty) semanticObject); 
				return; 
			case OmlPackage.STRUCTURED_PROPERTY_CARDINALITY_RESTRICTION_AXIOM:
				sequence_StructuredPropertyCardinalityRestrictionAxiom(context, (StructuredPropertyCardinalityRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.STRUCTURED_PROPERTY_RANGE_RESTRICTION_AXIOM:
				sequence_StructuredPropertyRangeRestrictionAxiom(context, (StructuredPropertyRangeRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.STRUCTURED_PROPERTY_REFERENCE:
				sequence_StructuredPropertyReference(context, (StructuredPropertyReference) semanticObject); 
				return; 
			case OmlPackage.STRUCTURED_PROPERTY_VALUE_ASSERTION:
				sequence_StructuredPropertyValueAssertion(context, (StructuredPropertyValueAssertion) semanticObject); 
				return; 
			case OmlPackage.STRUCTURED_PROPERTY_VALUE_RESTRICTION_AXIOM:
				sequence_StructuredPropertyValueRestrictionAxiom(context, (StructuredPropertyValueRestrictionAxiom) semanticObject); 
				return; 
			case OmlPackage.TYPE_PREDICATE:
				sequence_TypePredicate(context, (TypePredicate) semanticObject); 
				return; 
			case OmlPackage.VOCABULARY:
				sequence_Vocabulary(context, (Vocabulary) semanticObject); 
				return; 
			case OmlPackage.VOCABULARY_BUNDLE:
				sequence_VocabularyBundle(context, (VocabularyBundle) semanticObject); 
				return; 
			case OmlPackage.VOCABULARY_BUNDLE_EXTENSION:
				sequence_VocabularyBundleExtension(context, (VocabularyBundleExtension) semanticObject); 
				return; 
			case OmlPackage.VOCABULARY_BUNDLE_INCLUSION:
				sequence_VocabularyBundleInclusion(context, (VocabularyBundleInclusion) semanticObject); 
				return; 
			case OmlPackage.VOCABULARY_EXTENSION:
				sequence_VocabularyExtension(context, (VocabularyExtension) semanticObject); 
				return; 
			case OmlPackage.VOCABULARY_USAGE:
				sequence_VocabularyUsage(context, (VocabularyUsage) semanticObject); 
				return; 
			}
		if (errorAcceptor != null)
			errorAcceptor.accept(diagnosticProvider.createInvalidContextOrTypeDiagnostic(semanticObject, context));
	}
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns AnnotationPropertyReference
	 *     AnnotationPropertyReference returns AnnotationPropertyReference
	 *     VocabularyStatement returns AnnotationPropertyReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         property=[AnnotationProperty|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)?
	 *     )
	 * </pre>
	 */
	protected void sequence_AnnotationPropertyReference(ISerializationContext context, AnnotationPropertyReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns AnnotationProperty
	 *     Feature returns AnnotationProperty
	 *     Property returns AnnotationProperty
	 *     AnnotationProperty returns AnnotationProperty
	 *     VocabularyStatement returns AnnotationProperty
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* name=ID (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)?)
	 * </pre>
	 */
	protected void sequence_AnnotationProperty(ISerializationContext context, AnnotationProperty semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Annotation returns Annotation
	 *
	 * Constraint:
	 *     (property=[AnnotationProperty|Ref] (value=Literal | referenceValue=[Member|Ref])?)
	 * </pre>
	 */
	protected void sequence_Annotation(ISerializationContext context, Annotation semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns AspectReference
	 *     ClassifierReference returns AspectReference
	 *     EntityReference returns AspectReference
	 *     AspectReference returns AspectReference
	 *     VocabularyStatement returns AspectReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         aspect=[Aspect|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (ownedKeys+=KeyAxiom | ownedPropertyRestrictions+=PropertyRestrictionAxiom | ownedRelationRestrictions+=RelationRestrictionAxiom)*
	 *     )
	 * </pre>
	 */
	protected void sequence_AspectReference(ISerializationContext context, AspectReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns Aspect
	 *     Type returns Aspect
	 *     Classifier returns Aspect
	 *     Entity returns Aspect
	 *     Aspect returns Aspect
	 *     VocabularyStatement returns Aspect
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (ownedKeys+=KeyAxiom | ownedPropertyRestrictions+=PropertyRestrictionAxiom | ownedRelationRestrictions+=RelationRestrictionAxiom)*
	 *     )
	 * </pre>
	 */
	protected void sequence_Aspect(ISerializationContext context, Aspect semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Literal returns BooleanLiteral
	 *     BooleanLiteral returns BooleanLiteral
	 *
	 * Constraint:
	 *     value=Boolean
	 * </pre>
	 */
	protected void sequence_BooleanLiteral(ISerializationContext context, BooleanLiteral semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.BOOLEAN_LITERAL__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.BOOLEAN_LITERAL__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getBooleanLiteralAccess().getValueBooleanParserRuleCall_0(), semanticObject.isValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     NamedInstanceReference returns ConceptInstanceReference
	 *     ConceptInstanceReference returns ConceptInstanceReference
	 *     DescriptionStatement returns ConceptInstanceReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         instance=[ConceptInstance|Ref] 
	 *         (ownedTypes+=ConceptTypeAssertion ownedTypes+=ConceptTypeAssertion*)? 
	 *         ownedPropertyValues+=PropertyValueAssertion* 
	 *         ownedLinks+=LinkAssertion*
	 *     )
	 * </pre>
	 */
	protected void sequence_ConceptInstanceReference(ISerializationContext context, ConceptInstanceReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     NamedInstance returns ConceptInstance
	 *     ConceptInstance returns ConceptInstance
	 *     DescriptionStatement returns ConceptInstance
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedTypes+=ConceptTypeAssertion ownedTypes+=ConceptTypeAssertion*)? 
	 *         (ownedPropertyValues+=PropertyValueAssertion | ownedLinks+=LinkAssertion)*
	 *     )
	 * </pre>
	 */
	protected void sequence_ConceptInstance(ISerializationContext context, ConceptInstance semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns ConceptReference
	 *     ClassifierReference returns ConceptReference
	 *     EntityReference returns ConceptReference
	 *     ConceptReference returns ConceptReference
	 *     VocabularyStatement returns ConceptReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         concept=[Concept|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (ownedKeys+=KeyAxiom | ownedPropertyRestrictions+=PropertyRestrictionAxiom | ownedRelationRestrictions+=RelationRestrictionAxiom)*
	 *     )
	 * </pre>
	 */
	protected void sequence_ConceptReference(ISerializationContext context, ConceptReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     ConceptTypeAssertion returns ConceptTypeAssertion
	 *
	 * Constraint:
	 *     type=[Concept|Ref]
	 * </pre>
	 */
	protected void sequence_ConceptTypeAssertion(ISerializationContext context, ConceptTypeAssertion semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.CONCEPT_TYPE_ASSERTION__TYPE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.CONCEPT_TYPE_ASSERTION__TYPE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getConceptTypeAssertionAccess().getTypeConceptRefParserRuleCall_0_1(), semanticObject.eGet(OmlPackage.Literals.CONCEPT_TYPE_ASSERTION__TYPE, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns Concept
	 *     Type returns Concept
	 *     Classifier returns Concept
	 *     Entity returns Concept
	 *     Concept returns Concept
	 *     VocabularyStatement returns Concept
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (ownedKeys+=KeyAxiom | ownedPropertyRestrictions+=PropertyRestrictionAxiom | ownedRelationRestrictions+=RelationRestrictionAxiom)* 
	 *         (enumeratedInstances+=[ConceptInstance|Ref] enumeratedInstances+=[ConceptInstance|Ref]*)?
	 *     )
	 * </pre>
	 */
	protected void sequence_Concept(ISerializationContext context, Concept semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Literal returns DecimalLiteral
	 *     DecimalLiteral returns DecimalLiteral
	 *
	 * Constraint:
	 *     value=Decimal
	 * </pre>
	 */
	protected void sequence_DecimalLiteral(ISerializationContext context, DecimalLiteral semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.DECIMAL_LITERAL__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.DECIMAL_LITERAL__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getDecimalLiteralAccess().getValueDecimalParserRuleCall_0(), semanticObject.getValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     DescriptionBundleImport returns DescriptionBundleExtension
	 *     DescriptionBundleExtension returns DescriptionBundleExtension
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_DescriptionBundleExtension(ISerializationContext context, DescriptionBundleExtension semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     DescriptionBundleImport returns DescriptionBundleInclusion
	 *     DescriptionBundleInclusion returns DescriptionBundleInclusion
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_DescriptionBundleInclusion(ISerializationContext context, DescriptionBundleInclusion semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     DescriptionBundleImport returns DescriptionBundleUsage
	 *     DescriptionBundleUsage returns DescriptionBundleUsage
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_DescriptionBundleUsage(ISerializationContext context, DescriptionBundleUsage semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Ontology returns DescriptionBundle
	 *     DescriptionBox returns DescriptionBundle
	 *     DescriptionBundle returns DescriptionBundle
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID ownedImports+=DescriptionBundleImport*)
	 * </pre>
	 */
	protected void sequence_DescriptionBundle(ISerializationContext context, DescriptionBundle semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     DescriptionImport returns DescriptionExtension
	 *     DescriptionExtension returns DescriptionExtension
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_DescriptionExtension(ISerializationContext context, DescriptionExtension semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     DescriptionImport returns DescriptionUsage
	 *     DescriptionUsage returns DescriptionUsage
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_DescriptionUsage(ISerializationContext context, DescriptionUsage semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Ontology returns Description
	 *     DescriptionBox returns Description
	 *     Description returns Description
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID ownedImports+=DescriptionImport* ownedStatements+=DescriptionStatement*)
	 * </pre>
	 */
	protected void sequence_Description(ISerializationContext context, Description semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Predicate returns DifferentFromPredicate
	 *     BinaryPredicate returns DifferentFromPredicate
	 *     DifferentFromPredicate returns DifferentFromPredicate
	 *
	 * Constraint:
	 *     (variable1=ID (variable2=ID | instance2=[NamedInstance|CrossRef]))
	 * </pre>
	 */
	protected void sequence_DifferentFromPredicate(ISerializationContext context, DifferentFromPredicate semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Literal returns DoubleLiteral
	 *     DoubleLiteral returns DoubleLiteral
	 *
	 * Constraint:
	 *     value=Double
	 * </pre>
	 */
	protected void sequence_DoubleLiteral(ISerializationContext context, DoubleLiteral semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.DOUBLE_LITERAL__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.DOUBLE_LITERAL__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getDoubleLiteralAccess().getValueDoubleParserRuleCall_0(), semanticObject.getValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns EnumeratedScalarReference
	 *     EnumeratedScalarReference returns EnumeratedScalarReference
	 *     VocabularyStatement returns EnumeratedScalarReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         scalar=[EnumeratedScalar|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)?
	 *     )
	 * </pre>
	 */
	protected void sequence_EnumeratedScalarReference(ISerializationContext context, EnumeratedScalarReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns EnumeratedScalar
	 *     Type returns EnumeratedScalar
	 *     Scalar returns EnumeratedScalar
	 *     EnumeratedScalar returns EnumeratedScalar
	 *     VocabularyStatement returns EnumeratedScalar
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (literals+=Literal literals+=Literal*)?
	 *     )
	 * </pre>
	 */
	protected void sequence_EnumeratedScalar(ISerializationContext context, EnumeratedScalar semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns FacetedScalarReference
	 *     FacetedScalarReference returns FacetedScalarReference
	 *     VocabularyStatement returns FacetedScalarReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         scalar=[FacetedScalar|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)?
	 *     )
	 * </pre>
	 */
	protected void sequence_FacetedScalarReference(ISerializationContext context, FacetedScalarReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns FacetedScalar
	 *     Type returns FacetedScalar
	 *     Scalar returns FacetedScalar
	 *     FacetedScalar returns FacetedScalar
	 *     VocabularyStatement returns FacetedScalar
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (
	 *             length=UnsignedInteger | 
	 *             minLength=UnsignedInteger | 
	 *             maxLength=UnsignedInteger | 
	 *             pattern=STRING | 
	 *             language=ID | 
	 *             minInclusive=Literal | 
	 *             minExclusive=Literal | 
	 *             maxInclusive=Literal | 
	 *             maxExclusive=Literal
	 *         )*
	 *     )
	 * </pre>
	 */
	protected void sequence_FacetedScalar(ISerializationContext context, FacetedScalar semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Predicate returns FeaturePredicate
	 *     BinaryPredicate returns FeaturePredicate
	 *     FeaturePredicate returns FeaturePredicate
	 *
	 * Constraint:
	 *     (feature=[Feature|Ref] variable1=ID (variable2=ID | literal2=Literal | instance2=[NamedInstance|CrossRef]))
	 * </pre>
	 */
	protected void sequence_FeaturePredicate(ISerializationContext context, FeaturePredicate semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Relation returns ForwardRelation
	 *     ForwardRelation returns ForwardRelation
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* name=ID)
	 * </pre>
	 */
	protected void sequence_ForwardRelation(ISerializationContext context, ForwardRelation semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Literal returns IntegerLiteral
	 *     IntegerLiteral returns IntegerLiteral
	 *
	 * Constraint:
	 *     value=Integer
	 * </pre>
	 */
	protected void sequence_IntegerLiteral(ISerializationContext context, IntegerLiteral semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.INTEGER_LITERAL__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.INTEGER_LITERAL__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getIntegerLiteralAccess().getValueIntegerParserRuleCall_0(), semanticObject.getValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     KeyAxiom returns KeyAxiom
	 *
	 * Constraint:
	 *     (properties+=[Feature|Ref] properties+=[Feature|Ref]*)
	 * </pre>
	 */
	protected void sequence_KeyAxiom(ISerializationContext context, KeyAxiom semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     LinkAssertion returns LinkAssertion
	 *
	 * Constraint:
	 *     (relation=[Relation|Ref] target=[NamedInstance|Ref])
	 * </pre>
	 */
	protected void sequence_LinkAssertion(ISerializationContext context, LinkAssertion semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.LINK_ASSERTION__RELATION) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.LINK_ASSERTION__RELATION));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.LINK_ASSERTION__TARGET) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.LINK_ASSERTION__TARGET));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getLinkAssertionAccess().getRelationRelationRefParserRuleCall_0_0_1(), semanticObject.eGet(OmlPackage.Literals.LINK_ASSERTION__RELATION, false));
		feeder.accept(grammarAccess.getLinkAssertionAccess().getTargetNamedInstanceRefParserRuleCall_1_0_1(), semanticObject.eGet(OmlPackage.Literals.LINK_ASSERTION__TARGET, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Literal returns QuotedLiteral
	 *     QuotedLiteral returns QuotedLiteral
	 *
	 * Constraint:
	 *     (value=STRING (type=[Scalar|Ref] | langTag=ID)?)
	 * </pre>
	 */
	protected void sequence_QuotedLiteral(ISerializationContext context, QuotedLiteral semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     RelationRestrictionAxiom returns RelationCardinalityRestrictionAxiom
	 *     RelationCardinalityRestrictionAxiom returns RelationCardinalityRestrictionAxiom
	 *
	 * Constraint:
	 *     (relation=[Relation|Ref] kind=CardinalityRestrictionKind cardinality=UnsignedInteger range=[Entity|Ref]?)
	 * </pre>
	 */
	protected void sequence_RelationCardinalityRestrictionAxiom(ISerializationContext context, RelationCardinalityRestrictionAxiom semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Predicate returns RelationEntityPredicate
	 *     BinaryPredicate returns RelationEntityPredicate
	 *     RelationEntityPredicate returns RelationEntityPredicate
	 *
	 * Constraint:
	 *     (entity=[RelationEntity|Ref] variable1=ID entityVariable=ID (variable2=ID | instance2=[NamedInstance|CrossRef]))
	 * </pre>
	 */
	protected void sequence_RelationEntityPredicate(ISerializationContext context, RelationEntityPredicate semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns RelationEntityReference
	 *     ClassifierReference returns RelationEntityReference
	 *     EntityReference returns RelationEntityReference
	 *     RelationEntityReference returns RelationEntityReference
	 *     VocabularyStatement returns RelationEntityReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         entity=[RelationEntity|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (ownedKeys+=KeyAxiom | ownedPropertyRestrictions+=PropertyRestrictionAxiom | ownedRelationRestrictions+=RelationRestrictionAxiom)*
	 *     )
	 * </pre>
	 */
	protected void sequence_RelationEntityReference(ISerializationContext context, RelationEntityReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns RelationEntity
	 *     Type returns RelationEntity
	 *     Classifier returns RelationEntity
	 *     Entity returns RelationEntity
	 *     RelationEntity returns RelationEntity
	 *     VocabularyStatement returns RelationEntity
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         source=[Entity|Ref] 
	 *         target=[Entity|Ref] 
	 *         (forwardRelation=ForwardRelation | reverseRelation=ReverseRelation)* 
	 *         functional?='functional'? 
	 *         (
	 *             (
	 *                 inverseFunctional?='inverse' | 
	 *                 symmetric?='symmetric' | 
	 *                 asymmetric?='asymmetric' | 
	 *                 reflexive?='reflexive' | 
	 *                 irreflexive?='irreflexive' | 
	 *                 transitive?='transitive'
	 *             )? 
	 *             functional?='functional'?
	 *         )* 
	 *         ownedKeys+=KeyAxiom? 
	 *         ((ownedPropertyRestrictions+=PropertyRestrictionAxiom | ownedRelationRestrictions+=RelationRestrictionAxiom)? ownedKeys+=KeyAxiom?)*
	 *     )
	 * </pre>
	 */
	protected void sequence_RelationEntity(ISerializationContext context, RelationEntity semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     NamedInstanceReference returns RelationInstanceReference
	 *     RelationInstanceReference returns RelationInstanceReference
	 *     DescriptionStatement returns RelationInstanceReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         instance=[RelationInstance|Ref] 
	 *         (ownedTypes+=RelationTypeAssertion ownedTypes+=RelationTypeAssertion*)? 
	 *         ownedPropertyValues+=PropertyValueAssertion* 
	 *         ownedLinks+=LinkAssertion*
	 *     )
	 * </pre>
	 */
	protected void sequence_RelationInstanceReference(ISerializationContext context, RelationInstanceReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     NamedInstance returns RelationInstance
	 *     RelationInstance returns RelationInstance
	 *     DescriptionStatement returns RelationInstance
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedTypes+=RelationTypeAssertion ownedTypes+=RelationTypeAssertion*)? 
	 *         sources+=[NamedInstance|Ref] 
	 *         sources+=[NamedInstance|Ref]* 
	 *         targets+=[NamedInstance|Ref] 
	 *         targets+=[NamedInstance|Ref]* 
	 *         (ownedPropertyValues+=PropertyValueAssertion | ownedLinks+=LinkAssertion)*
	 *     )
	 * </pre>
	 */
	protected void sequence_RelationInstance(ISerializationContext context, RelationInstance semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     RelationRestrictionAxiom returns RelationRangeRestrictionAxiom
	 *     RelationRangeRestrictionAxiom returns RelationRangeRestrictionAxiom
	 *
	 * Constraint:
	 *     (kind=RangeRestrictionKind relation=[Relation|Ref] range=[Entity|Ref])
	 * </pre>
	 */
	protected void sequence_RelationRangeRestrictionAxiom(ISerializationContext context, RelationRangeRestrictionAxiom semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.RELATION_RANGE_RESTRICTION_AXIOM__KIND) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.RELATION_RANGE_RESTRICTION_AXIOM__KIND));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.RELATION_RESTRICTION_AXIOM__RELATION) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.RELATION_RESTRICTION_AXIOM__RELATION));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.RELATION_RANGE_RESTRICTION_AXIOM__RANGE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.RELATION_RANGE_RESTRICTION_AXIOM__RANGE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getRelationRangeRestrictionAxiomAccess().getKindRangeRestrictionKindEnumRuleCall_1_0(), semanticObject.getKind());
		feeder.accept(grammarAccess.getRelationRangeRestrictionAxiomAccess().getRelationRelationRefParserRuleCall_3_0_1(), semanticObject.eGet(OmlPackage.Literals.RELATION_RESTRICTION_AXIOM__RELATION, false));
		feeder.accept(grammarAccess.getRelationRangeRestrictionAxiomAccess().getRangeEntityRefParserRuleCall_5_0_1(), semanticObject.eGet(OmlPackage.Literals.RELATION_RANGE_RESTRICTION_AXIOM__RANGE, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     RelationReference returns RelationReference
	 *     VocabularyStatement returns RelationReference
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* relation=[Relation|Ref])
	 * </pre>
	 */
	protected void sequence_RelationReference(ISerializationContext context, RelationReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     RelationRestrictionAxiom returns RelationTargetRestrictionAxiom
	 *     RelationTargetRestrictionAxiom returns RelationTargetRestrictionAxiom
	 *
	 * Constraint:
	 *     (relation=[Relation|Ref] target=[NamedInstance|Ref])
	 * </pre>
	 */
	protected void sequence_RelationTargetRestrictionAxiom(ISerializationContext context, RelationTargetRestrictionAxiom semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.RELATION_RESTRICTION_AXIOM__RELATION) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.RELATION_RESTRICTION_AXIOM__RELATION));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.RELATION_TARGET_RESTRICTION_AXIOM__TARGET) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.RELATION_TARGET_RESTRICTION_AXIOM__TARGET));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getRelationTargetRestrictionAxiomAccess().getRelationRelationRefParserRuleCall_2_0_1(), semanticObject.eGet(OmlPackage.Literals.RELATION_RESTRICTION_AXIOM__RELATION, false));
		feeder.accept(grammarAccess.getRelationTargetRestrictionAxiomAccess().getTargetNamedInstanceRefParserRuleCall_4_0_1(), semanticObject.eGet(OmlPackage.Literals.RELATION_TARGET_RESTRICTION_AXIOM__TARGET, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     RelationTypeAssertion returns RelationTypeAssertion
	 *
	 * Constraint:
	 *     type=[RelationEntity|Ref]
	 * </pre>
	 */
	protected void sequence_RelationTypeAssertion(ISerializationContext context, RelationTypeAssertion semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.RELATION_TYPE_ASSERTION__TYPE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.RELATION_TYPE_ASSERTION__TYPE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getRelationTypeAssertionAccess().getTypeRelationEntityRefParserRuleCall_0_1(), semanticObject.eGet(OmlPackage.Literals.RELATION_TYPE_ASSERTION__TYPE, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Relation returns ReverseRelation
	 *     ReverseRelation returns ReverseRelation
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* name=ID)
	 * </pre>
	 */
	protected void sequence_ReverseRelation(ISerializationContext context, ReverseRelation semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     RuleReference returns RuleReference
	 *     VocabularyStatement returns RuleReference
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* rule=[Rule|Ref])
	 * </pre>
	 */
	protected void sequence_RuleReference(ISerializationContext context, RuleReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Rule returns Rule
	 *     VocabularyStatement returns Rule
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         antecedent+=Predicate 
	 *         antecedent+=Predicate* 
	 *         consequent+=Predicate 
	 *         consequent+=Predicate*
	 *     )
	 * </pre>
	 */
	protected void sequence_Rule(ISerializationContext context, Rule semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Predicate returns SameAsPredicate
	 *     BinaryPredicate returns SameAsPredicate
	 *     SameAsPredicate returns SameAsPredicate
	 *
	 * Constraint:
	 *     (variable1=ID (variable2=ID | instance2=[NamedInstance|CrossRef]))
	 * </pre>
	 */
	protected void sequence_SameAsPredicate(ISerializationContext context, SameAsPredicate semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyRestrictionAxiom returns ScalarPropertyCardinalityRestrictionAxiom
	 *     ScalarPropertyRestrictionAxiom returns ScalarPropertyCardinalityRestrictionAxiom
	 *     ScalarPropertyCardinalityRestrictionAxiom returns ScalarPropertyCardinalityRestrictionAxiom
	 *
	 * Constraint:
	 *     (property=[ScalarProperty|Ref] kind=CardinalityRestrictionKind cardinality=UnsignedInteger range=[Scalar|Ref]?)
	 * </pre>
	 */
	protected void sequence_ScalarPropertyCardinalityRestrictionAxiom(ISerializationContext context, ScalarPropertyCardinalityRestrictionAxiom semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyRestrictionAxiom returns ScalarPropertyRangeRestrictionAxiom
	 *     ScalarPropertyRestrictionAxiom returns ScalarPropertyRangeRestrictionAxiom
	 *     ScalarPropertyRangeRestrictionAxiom returns ScalarPropertyRangeRestrictionAxiom
	 *
	 * Constraint:
	 *     (kind=RangeRestrictionKind property=[ScalarProperty|Ref] range=[Scalar|Ref])
	 * </pre>
	 */
	protected void sequence_ScalarPropertyRangeRestrictionAxiom(ISerializationContext context, ScalarPropertyRangeRestrictionAxiom semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RANGE_RESTRICTION_AXIOM__KIND) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RANGE_RESTRICTION_AXIOM__KIND));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RESTRICTION_AXIOM__PROPERTY) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RESTRICTION_AXIOM__PROPERTY));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RANGE_RESTRICTION_AXIOM__RANGE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RANGE_RESTRICTION_AXIOM__RANGE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getScalarPropertyRangeRestrictionAxiomAccess().getKindRangeRestrictionKindEnumRuleCall_1_0(), semanticObject.getKind());
		feeder.accept(grammarAccess.getScalarPropertyRangeRestrictionAxiomAccess().getPropertyScalarPropertyRefParserRuleCall_4_0_1(), semanticObject.eGet(OmlPackage.Literals.SCALAR_PROPERTY_RESTRICTION_AXIOM__PROPERTY, false));
		feeder.accept(grammarAccess.getScalarPropertyRangeRestrictionAxiomAccess().getRangeScalarRefParserRuleCall_6_0_1(), semanticObject.eGet(OmlPackage.Literals.SCALAR_PROPERTY_RANGE_RESTRICTION_AXIOM__RANGE, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns ScalarPropertyReference
	 *     ScalarPropertyReference returns ScalarPropertyReference
	 *     VocabularyStatement returns ScalarPropertyReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         property=[ScalarProperty|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)?
	 *     )
	 * </pre>
	 */
	protected void sequence_ScalarPropertyReference(ISerializationContext context, ScalarPropertyReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyValueAssertion returns ScalarPropertyValueAssertion
	 *     ScalarPropertyValueAssertion returns ScalarPropertyValueAssertion
	 *
	 * Constraint:
	 *     (property=[ScalarProperty|Ref] value=Literal)
	 * </pre>
	 */
	protected void sequence_ScalarPropertyValueAssertion(ISerializationContext context, ScalarPropertyValueAssertion semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_VALUE_ASSERTION__PROPERTY) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_VALUE_ASSERTION__PROPERTY));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_VALUE_ASSERTION__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_VALUE_ASSERTION__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getScalarPropertyValueAssertionAccess().getPropertyScalarPropertyRefParserRuleCall_0_0_1(), semanticObject.eGet(OmlPackage.Literals.SCALAR_PROPERTY_VALUE_ASSERTION__PROPERTY, false));
		feeder.accept(grammarAccess.getScalarPropertyValueAssertionAccess().getValueLiteralParserRuleCall_1_0(), semanticObject.getValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyRestrictionAxiom returns ScalarPropertyValueRestrictionAxiom
	 *     ScalarPropertyRestrictionAxiom returns ScalarPropertyValueRestrictionAxiom
	 *     ScalarPropertyValueRestrictionAxiom returns ScalarPropertyValueRestrictionAxiom
	 *
	 * Constraint:
	 *     (property=[ScalarProperty|Ref] value=Literal)
	 * </pre>
	 */
	protected void sequence_ScalarPropertyValueRestrictionAxiom(ISerializationContext context, ScalarPropertyValueRestrictionAxiom semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RESTRICTION_AXIOM__PROPERTY) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_RESTRICTION_AXIOM__PROPERTY));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_VALUE_RESTRICTION_AXIOM__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SCALAR_PROPERTY_VALUE_RESTRICTION_AXIOM__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getScalarPropertyValueRestrictionAxiomAccess().getPropertyScalarPropertyRefParserRuleCall_3_0_1(), semanticObject.eGet(OmlPackage.Literals.SCALAR_PROPERTY_RESTRICTION_AXIOM__PROPERTY, false));
		feeder.accept(grammarAccess.getScalarPropertyValueRestrictionAxiomAccess().getValueLiteralParserRuleCall_5_0(), semanticObject.getValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns ScalarProperty
	 *     Feature returns ScalarProperty
	 *     Property returns ScalarProperty
	 *     SemanticProperty returns ScalarProperty
	 *     ScalarProperty returns ScalarProperty
	 *     VocabularyStatement returns ScalarProperty
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (domain=[Classifier|Ref] range=[Scalar|Ref] functional?='functional'?)?
	 *     )
	 * </pre>
	 */
	protected void sequence_ScalarProperty(ISerializationContext context, ScalarProperty semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializationAxiom returns SpecializationAxiom
	 *
	 * Constraint:
	 *     specializedTerm=[SpecializableTerm|Ref]
	 * </pre>
	 */
	protected void sequence_SpecializationAxiom(ISerializationContext context, SpecializationAxiom semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.SPECIALIZATION_AXIOM__SPECIALIZED_TERM) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.SPECIALIZATION_AXIOM__SPECIALIZED_TERM));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getSpecializationAxiomAccess().getSpecializedTermSpecializableTermRefParserRuleCall_0_1(), semanticObject.eGet(OmlPackage.Literals.SPECIALIZATION_AXIOM__SPECIALIZED_TERM, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     StructureInstance returns StructureInstance
	 *
	 * Constraint:
	 *     (type=[Structure|Ref] ownedPropertyValues+=PropertyValueAssertion*)
	 * </pre>
	 */
	protected void sequence_StructureInstance(ISerializationContext context, StructureInstance semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns StructureReference
	 *     ClassifierReference returns StructureReference
	 *     StructureReference returns StructureReference
	 *     VocabularyStatement returns StructureReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         structure=[Structure|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         ownedPropertyRestrictions+=PropertyRestrictionAxiom*
	 *     )
	 * </pre>
	 */
	protected void sequence_StructureReference(ISerializationContext context, StructureReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns Structure
	 *     Type returns Structure
	 *     Classifier returns Structure
	 *     Structure returns Structure
	 *     VocabularyStatement returns Structure
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         ownedPropertyRestrictions+=PropertyRestrictionAxiom*
	 *     )
	 * </pre>
	 */
	protected void sequence_Structure(ISerializationContext context, Structure semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyRestrictionAxiom returns StructuredPropertyCardinalityRestrictionAxiom
	 *     StructuredPropertyRestrictionAxiom returns StructuredPropertyCardinalityRestrictionAxiom
	 *     StructuredPropertyCardinalityRestrictionAxiom returns StructuredPropertyCardinalityRestrictionAxiom
	 *
	 * Constraint:
	 *     (property=[StructuredProperty|Ref] kind=CardinalityRestrictionKind cardinality=UnsignedInteger range=[Structure|Ref]?)
	 * </pre>
	 */
	protected void sequence_StructuredPropertyCardinalityRestrictionAxiom(ISerializationContext context, StructuredPropertyCardinalityRestrictionAxiom semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyRestrictionAxiom returns StructuredPropertyRangeRestrictionAxiom
	 *     StructuredPropertyRestrictionAxiom returns StructuredPropertyRangeRestrictionAxiom
	 *     StructuredPropertyRangeRestrictionAxiom returns StructuredPropertyRangeRestrictionAxiom
	 *
	 * Constraint:
	 *     (kind=RangeRestrictionKind property=[StructuredProperty|Ref] range=[Structure|Ref])
	 * </pre>
	 */
	protected void sequence_StructuredPropertyRangeRestrictionAxiom(ISerializationContext context, StructuredPropertyRangeRestrictionAxiom semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RANGE_RESTRICTION_AXIOM__KIND) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RANGE_RESTRICTION_AXIOM__KIND));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RESTRICTION_AXIOM__PROPERTY) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RESTRICTION_AXIOM__PROPERTY));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RANGE_RESTRICTION_AXIOM__RANGE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RANGE_RESTRICTION_AXIOM__RANGE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getStructuredPropertyRangeRestrictionAxiomAccess().getKindRangeRestrictionKindEnumRuleCall_1_0(), semanticObject.getKind());
		feeder.accept(grammarAccess.getStructuredPropertyRangeRestrictionAxiomAccess().getPropertyStructuredPropertyRefParserRuleCall_4_0_1(), semanticObject.eGet(OmlPackage.Literals.STRUCTURED_PROPERTY_RESTRICTION_AXIOM__PROPERTY, false));
		feeder.accept(grammarAccess.getStructuredPropertyRangeRestrictionAxiomAccess().getRangeStructureRefParserRuleCall_6_0_1(), semanticObject.eGet(OmlPackage.Literals.STRUCTURED_PROPERTY_RANGE_RESTRICTION_AXIOM__RANGE, false));
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTermReference returns StructuredPropertyReference
	 *     StructuredPropertyReference returns StructuredPropertyReference
	 *     VocabularyStatement returns StructuredPropertyReference
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         property=[StructuredProperty|Ref] 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)?
	 *     )
	 * </pre>
	 */
	protected void sequence_StructuredPropertyReference(ISerializationContext context, StructuredPropertyReference semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyValueAssertion returns StructuredPropertyValueAssertion
	 *     StructuredPropertyValueAssertion returns StructuredPropertyValueAssertion
	 *
	 * Constraint:
	 *     (property=[StructuredProperty|Ref] value=StructureInstance)
	 * </pre>
	 */
	protected void sequence_StructuredPropertyValueAssertion(ISerializationContext context, StructuredPropertyValueAssertion semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_VALUE_ASSERTION__PROPERTY) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_VALUE_ASSERTION__PROPERTY));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_VALUE_ASSERTION__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_VALUE_ASSERTION__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getStructuredPropertyValueAssertionAccess().getPropertyStructuredPropertyRefParserRuleCall_0_0_1(), semanticObject.eGet(OmlPackage.Literals.STRUCTURED_PROPERTY_VALUE_ASSERTION__PROPERTY, false));
		feeder.accept(grammarAccess.getStructuredPropertyValueAssertionAccess().getValueStructureInstanceParserRuleCall_1_0(), semanticObject.getValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     PropertyRestrictionAxiom returns StructuredPropertyValueRestrictionAxiom
	 *     StructuredPropertyRestrictionAxiom returns StructuredPropertyValueRestrictionAxiom
	 *     StructuredPropertyValueRestrictionAxiom returns StructuredPropertyValueRestrictionAxiom
	 *
	 * Constraint:
	 *     (property=[StructuredProperty|Ref] value=StructureInstance)
	 * </pre>
	 */
	protected void sequence_StructuredPropertyValueRestrictionAxiom(ISerializationContext context, StructuredPropertyValueRestrictionAxiom semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RESTRICTION_AXIOM__PROPERTY) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_RESTRICTION_AXIOM__PROPERTY));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_VALUE_RESTRICTION_AXIOM__VALUE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.STRUCTURED_PROPERTY_VALUE_RESTRICTION_AXIOM__VALUE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getStructuredPropertyValueRestrictionAxiomAccess().getPropertyStructuredPropertyRefParserRuleCall_3_0_1(), semanticObject.eGet(OmlPackage.Literals.STRUCTURED_PROPERTY_RESTRICTION_AXIOM__PROPERTY, false));
		feeder.accept(grammarAccess.getStructuredPropertyValueRestrictionAxiomAccess().getValueStructureInstanceParserRuleCall_5_0(), semanticObject.getValue());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     SpecializableTerm returns StructuredProperty
	 *     Feature returns StructuredProperty
	 *     Property returns StructuredProperty
	 *     SemanticProperty returns StructuredProperty
	 *     StructuredProperty returns StructuredProperty
	 *     VocabularyStatement returns StructuredProperty
	 *
	 * Constraint:
	 *     (
	 *         ownedAnnotations+=Annotation* 
	 *         name=ID 
	 *         (ownedSpecializations+=SpecializationAxiom ownedSpecializations+=SpecializationAxiom*)? 
	 *         (domain=[Classifier|Ref] range=[Structure|Ref] functional?='functional'?)?
	 *     )
	 * </pre>
	 */
	protected void sequence_StructuredProperty(ISerializationContext context, StructuredProperty semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Predicate returns TypePredicate
	 *     UnaryPredicate returns TypePredicate
	 *     TypePredicate returns TypePredicate
	 *
	 * Constraint:
	 *     (type=[Type|Ref] variable=ID)
	 * </pre>
	 */
	protected void sequence_TypePredicate(ISerializationContext context, TypePredicate semanticObject) {
		if (errorAcceptor != null) {
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.TYPE_PREDICATE__TYPE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.TYPE_PREDICATE__TYPE));
			if (transientValues.isValueTransient(semanticObject, OmlPackage.Literals.UNARY_PREDICATE__VARIABLE) == ValueTransient.YES)
				errorAcceptor.accept(diagnosticProvider.createFeatureValueMissing(semanticObject, OmlPackage.Literals.UNARY_PREDICATE__VARIABLE));
		}
		SequenceFeeder feeder = createSequencerFeeder(context, semanticObject);
		feeder.accept(grammarAccess.getTypePredicateAccess().getTypeTypeRefParserRuleCall_0_0_1(), semanticObject.eGet(OmlPackage.Literals.TYPE_PREDICATE__TYPE, false));
		feeder.accept(grammarAccess.getTypePredicateAccess().getVariableIDTerminalRuleCall_2_0(), semanticObject.getVariable());
		feeder.finish();
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     VocabularyBundleImport returns VocabularyBundleExtension
	 *     VocabularyBundleExtension returns VocabularyBundleExtension
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_VocabularyBundleExtension(ISerializationContext context, VocabularyBundleExtension semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     VocabularyBundleImport returns VocabularyBundleInclusion
	 *     VocabularyBundleInclusion returns VocabularyBundleInclusion
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_VocabularyBundleInclusion(ISerializationContext context, VocabularyBundleInclusion semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Ontology returns VocabularyBundle
	 *     VocabularyBox returns VocabularyBundle
	 *     VocabularyBundle returns VocabularyBundle
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID ownedImports+=VocabularyBundleImport*)
	 * </pre>
	 */
	protected void sequence_VocabularyBundle(ISerializationContext context, VocabularyBundle semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     VocabularyImport returns VocabularyExtension
	 *     VocabularyExtension returns VocabularyExtension
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_VocabularyExtension(ISerializationContext context, VocabularyExtension semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     VocabularyImport returns VocabularyUsage
	 *     VocabularyUsage returns VocabularyUsage
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID?)
	 * </pre>
	 */
	protected void sequence_VocabularyUsage(ISerializationContext context, VocabularyUsage semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
	/**
	 * <pre>
	 * Contexts:
	 *     Ontology returns Vocabulary
	 *     VocabularyBox returns Vocabulary
	 *     Vocabulary returns Vocabulary
	 *
	 * Constraint:
	 *     (ownedAnnotations+=Annotation* namespace=NAMESPACE prefix=ID ownedImports+=VocabularyImport* ownedStatements+=VocabularyStatement*)
	 * </pre>
	 */
	protected void sequence_Vocabulary(ISerializationContext context, Vocabulary semanticObject) {
		genericSequencer.createSequence(context, semanticObject);
	}
	
	
}
