/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.exporter.zipkin;

import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.AttributeType;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.trace.StatusCode;
import io.opentelemetry.exporter.zipkin.ZipkinSpanExporterBuilder;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.common.InstrumentationLibraryInfo;
import io.opentelemetry.sdk.internal.ThrottlingLogger;
import io.opentelemetry.sdk.resources.Resource;
import io.opentelemetry.sdk.trace.data.EventData;
import io.opentelemetry.sdk.trace.data.SpanData;
import io.opentelemetry.sdk.trace.data.StatusData;
import io.opentelemetry.sdk.trace.export.SpanExporter;
import io.opentelemetry.semconv.resource.attributes.ResourceAttributes;
import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import zipkin2.Callback;
import zipkin2.Endpoint;
import zipkin2.Span;
import zipkin2.codec.BytesEncoder;
import zipkin2.reporter.Sender;

public final class ZipkinSpanExporter
implements SpanExporter {
    public static final String DEFAULT_ENDPOINT = "http://localhost:9411/api/v2/spans";
    public static final Logger baseLogger = Logger.getLogger(ZipkinSpanExporter.class.getName());
    private final ThrottlingLogger logger = new ThrottlingLogger(baseLogger);
    static final String OTEL_DROPPED_ATTRIBUTES_COUNT = "otel.dropped_attributes_count";
    static final String OTEL_DROPPED_EVENTS_COUNT = "otel.dropped_events_count";
    static final String OTEL_STATUS_CODE = "otel.status_code";
    static final AttributeKey<String> STATUS_ERROR = AttributeKey.stringKey((String)"error");
    static final String KEY_INSTRUMENTATION_LIBRARY_NAME = "otel.library.name";
    static final String KEY_INSTRUMENTATION_LIBRARY_VERSION = "otel.library.version";
    private final BytesEncoder<Span> encoder;
    private final Sender sender;
    @Nullable
    private final InetAddress localAddress;

    ZipkinSpanExporter(BytesEncoder<Span> encoder, Sender sender) {
        this.encoder = encoder;
        this.sender = sender;
        this.localAddress = ZipkinSpanExporter.produceLocalIp();
    }

    @Nullable
    static InetAddress produceLocalIp() {
        try {
            Enumeration<NetworkInterface> nics = NetworkInterface.getNetworkInterfaces();
            while (nics.hasMoreElements()) {
                NetworkInterface nic = nics.nextElement();
                Enumeration<InetAddress> addresses = nic.getInetAddresses();
                while (addresses.hasMoreElements()) {
                    InetAddress address = addresses.nextElement();
                    if (!address.isSiteLocalAddress()) continue;
                    return address;
                }
            }
        }
        catch (Exception e) {
            baseLogger.log(Level.FINE, "error reading nics", e);
        }
        return null;
    }

    Span generateSpan(SpanData spanData) {
        InstrumentationLibraryInfo instrumentationLibraryInfo;
        StatusData status;
        Endpoint endpoint = this.getEndpoint(spanData);
        long startTimestamp = ZipkinSpanExporter.toEpochMicros(spanData.getStartEpochNanos());
        long endTimestamp = ZipkinSpanExporter.toEpochMicros(spanData.getEndEpochNanos());
        Span.Builder spanBuilder = Span.newBuilder().traceId(spanData.getTraceId()).id(spanData.getSpanId()).kind(ZipkinSpanExporter.toSpanKind(spanData)).name(spanData.getName()).timestamp(ZipkinSpanExporter.toEpochMicros(spanData.getStartEpochNanos())).duration(Math.max(1L, endTimestamp - startTimestamp)).localEndpoint(endpoint);
        if (spanData.getParentSpanContext().isValid()) {
            spanBuilder.parentId(spanData.getParentSpanId());
        }
        Attributes spanAttributes = spanData.getAttributes();
        spanAttributes.forEach((key, value) -> spanBuilder.putTag(key.getKey(), ZipkinSpanExporter.valueToString(key, value)));
        int droppedAttributes = spanData.getTotalAttributeCount() - spanAttributes.size();
        if (droppedAttributes > 0) {
            spanBuilder.putTag(OTEL_DROPPED_ATTRIBUTES_COUNT, String.valueOf(droppedAttributes));
        }
        if ((status = spanData.getStatus()).getStatusCode() != StatusCode.UNSET) {
            spanBuilder.putTag(OTEL_STATUS_CODE, status.getStatusCode().toString());
            if (status.getStatusCode() == StatusCode.ERROR && spanAttributes.get(STATUS_ERROR) == null) {
                spanBuilder.putTag(STATUS_ERROR.getKey(), ZipkinSpanExporter.nullToEmpty(status.getDescription()));
            }
        }
        if (!(instrumentationLibraryInfo = spanData.getInstrumentationLibraryInfo()).getName().isEmpty()) {
            spanBuilder.putTag(KEY_INSTRUMENTATION_LIBRARY_NAME, instrumentationLibraryInfo.getName());
        }
        if (instrumentationLibraryInfo.getVersion() != null) {
            spanBuilder.putTag(KEY_INSTRUMENTATION_LIBRARY_VERSION, instrumentationLibraryInfo.getVersion());
        }
        for (EventData annotation : spanData.getEvents()) {
            spanBuilder.addAnnotation(ZipkinSpanExporter.toEpochMicros(annotation.getEpochNanos()), annotation.getName());
        }
        int droppedEvents = spanData.getTotalRecordedEvents() - spanData.getEvents().size();
        if (droppedEvents > 0) {
            spanBuilder.putTag(OTEL_DROPPED_EVENTS_COUNT, String.valueOf(droppedEvents));
        }
        return spanBuilder.build();
    }

    private static String nullToEmpty(String value) {
        return value != null ? value : "";
    }

    private Endpoint getEndpoint(SpanData spanData) {
        Attributes resourceAttributes = spanData.getResource().getAttributes();
        Endpoint.Builder endpoint = Endpoint.newBuilder().ip(this.localAddress);
        String serviceNameValue = (String)resourceAttributes.get(ResourceAttributes.SERVICE_NAME);
        if (serviceNameValue == null) {
            serviceNameValue = (String)Resource.getDefault().getAttribute(ResourceAttributes.SERVICE_NAME);
        }
        if (serviceNameValue != null) {
            endpoint.serviceName(serviceNameValue);
        }
        return endpoint.build();
    }

    @Nullable
    private static Span.Kind toSpanKind(SpanData spanData) {
        switch (spanData.getKind()) {
            case SERVER: {
                return Span.Kind.SERVER;
            }
            case CLIENT: {
                return Span.Kind.CLIENT;
            }
            case PRODUCER: {
                return Span.Kind.PRODUCER;
            }
            case CONSUMER: {
                return Span.Kind.CONSUMER;
            }
            case INTERNAL: {
                return null;
            }
        }
        return null;
    }

    private static long toEpochMicros(long epochNanos) {
        return TimeUnit.NANOSECONDS.toMicros(epochNanos);
    }

    private static String valueToString(AttributeKey<?> key, Object attributeValue) {
        AttributeType type = key.getType();
        switch (type) {
            case STRING: 
            case BOOLEAN: 
            case LONG: 
            case DOUBLE: {
                return String.valueOf(attributeValue);
            }
            case STRING_ARRAY: 
            case BOOLEAN_ARRAY: 
            case LONG_ARRAY: 
            case DOUBLE_ARRAY: {
                return ZipkinSpanExporter.commaSeparated((List)attributeValue);
            }
        }
        throw new IllegalStateException("Unknown attribute type: " + type);
    }

    private static String commaSeparated(List<?> values) {
        StringBuilder builder = new StringBuilder();
        for (Object value : values) {
            if (builder.length() != 0) {
                builder.append(',');
            }
            builder.append(value);
        }
        return builder.toString();
    }

    public CompletableResultCode export(Collection<SpanData> spanDataList) {
        ArrayList<byte[]> encodedSpans = new ArrayList<byte[]>(spanDataList.size());
        for (SpanData spanData : spanDataList) {
            encodedSpans.add(this.encoder.encode((Object)this.generateSpan(spanData)));
        }
        final CompletableResultCode result = new CompletableResultCode();
        this.sender.sendSpans(encodedSpans).enqueue((Callback)new Callback<Void>(){

            public void onSuccess(Void value) {
                result.succeed();
            }

            public void onError(Throwable t) {
                ZipkinSpanExporter.this.logger.log(Level.WARNING, "Failed to export spans", t);
                result.fail();
            }
        });
        return result;
    }

    public CompletableResultCode flush() {
        return CompletableResultCode.ofSuccess();
    }

    public CompletableResultCode shutdown() {
        try {
            this.sender.close();
        }
        catch (IOException e) {
            this.logger.log(Level.WARNING, "Exception while closing the Zipkin Sender instance", (Throwable)e);
        }
        return CompletableResultCode.ofSuccess();
    }

    public static ZipkinSpanExporterBuilder builder() {
        return new ZipkinSpanExporterBuilder();
    }

    @Nullable
    InetAddress getLocalAddressForTest() {
        return this.localAddress;
    }
}

