package io.overcoded.vaadin.tinymce;

import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.customfield.CustomField;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.component.dependency.NpmPackage;
import io.overcoded.vaadin.tinymce.config.TinyMceConfig;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.util.UUID;

/**
 * CustomField which should contain the TinyMCE rich-text editor
 */
@Slf4j
@Tag("tinymce-field")
@JsModule("./@overcoded/tinymce/tinymce-field.ts")
@JsModule("./@overcoded/tinymce/tinymce-handlers.js")
@NpmPackage(value = "@tinymce/tinymce-webcomponent", version = "2.0.1")
@JsModule("@tinymce/tinymce-webcomponent/dist/tinymce-webcomponent.js")
public class TinyMceField extends CustomField<String> {
    /**
     * Configuration of TinyMCE
     */
    @Getter
    private TinyMceConfig config;
    /**
     * Label of the input field
     */
    @Getter
    private String label;
    /**
     * Identifier of the field, label and generated editor
     */
    @Getter
    private String editorId;
    /**
     * Content of the rich-text editor
     */
    @Getter
    private String content;
    /**
     * Backend state of editor read-only option
     */
    @Getter
    private boolean editorReadOnly;

    public TinyMceField() {
        this(TinyMceConfig.builder().build());
    }

    public TinyMceField(TinyMceConfig config) {
        addListener(TinyMceUpdateEvent.class, this::updateContent);
        String generatedId = UUID.randomUUID().toString();
        setConfig(config);
        setId(generatedId);
        setEditorReadOnly(false);
        setEditorId(generatedId);
    }

    @Override
    protected String generateModelValue() {
        return getContent();
    }

    @Override
    protected void setPresentationValue(String s) {
        setContent(s);
    }

    public void setEditorId(String editorId) {
        this.editorId = editorId;
        getElement().setProperty("editorId", editorId);
    }

    public void setConfig(TinyMceConfig config) {
        this.config = config;
        getElement().setPropertyBean("config", config);
    }

    public void setContent(String content) {
        this.content = content;
        super.setValue(content);
        getElement().setProperty("content", content);
    }

    @Override
    public void setLabel(String label) {
        this.label = label;
        super.setLabel(label);
        getElement().setProperty("label", label);
    }

    @Override
    public void setValue(String content) {
        setContent(content);
    }

    public void setEditorReadOnly(boolean editorReadOnly) {
        this.editorReadOnly = editorReadOnly;
        super.setReadOnly(editorReadOnly);
        getElement().setProperty("editorReadOnly", editorReadOnly);
    }

    @Override
    public void setReadOnly(boolean readOnly) {
        setEditorReadOnly(readOnly);
    }


    private void updateContent(TinyMceUpdateEvent event) {
        setContent(event.getContent());
    }
}
