/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.common.util.btree.sets;

import com.google.common.annotations.Beta;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.TimeoutTimer;
import io.pravega.common.concurrent.Futures;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.AsyncIterator;
import io.pravega.common.util.ByteArrayComparator;
import io.pravega.common.util.btree.sets.BTreeSetPage;
import io.pravega.common.util.btree.sets.ItemIterator;
import io.pravega.common.util.btree.sets.PageCollection;
import io.pravega.common.util.btree.sets.PagePointer;
import io.pravega.common.util.btree.sets.TreeModificationContext;
import io.pravega.common.util.btree.sets.UpdateItem;
import java.time.Duration;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import lombok.Generated;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
@Beta
public class BTreeSet {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(BTreeSet.class);
    public static final Comparator<ArrayView> COMPARATOR = new ByteArrayComparator()::compare;
    private static final Comparator<PagePointer> POINTER_COMPARATOR = PagePointer.getComparator(COMPARATOR);
    private final int maxPageSize;
    private final int maxItemSize;
    @NonNull
    private final ReadPage read;
    @NonNull
    private final PersistPages update;
    @NonNull
    private final Executor executor;
    @NonNull
    private final String traceLogId;

    public BTreeSet(int maxPageSize, int maxItemSize, @NonNull ReadPage read, @NonNull PersistPages update, @NonNull Executor executor, String traceLogId) {
        if (read == null) {
            throw new NullPointerException("read is marked non-null but is null");
        }
        if (update == null) {
            throw new NullPointerException("update is marked non-null but is null");
        }
        if (executor == null) {
            throw new NullPointerException("executor is marked non-null but is null");
        }
        Preconditions.checkArgument(maxItemSize < maxPageSize / 2, "maxItemSize must be at most half of maxPageSize.");
        this.maxItemSize = maxItemSize;
        this.maxPageSize = maxPageSize;
        this.read = read;
        this.update = update;
        this.executor = executor;
        this.traceLogId = traceLogId == null ? "" : traceLogId;
    }

    public CompletableFuture<Void> update(@Nullable Collection<? extends ArrayView> toInsert, @Nullable Collection<? extends ArrayView> toRemove, @NonNull Supplier<Long> getNextPageId, @NonNull Duration timeout) {
        if (getNextPageId == null) {
            throw new NullPointerException("getNextPageId is marked non-null but is null");
        }
        if (timeout == null) {
            throw new NullPointerException("timeout is marked non-null but is null");
        }
        TimeoutTimer timer = new TimeoutTimer(timeout);
        ArrayList<UpdateItem> updates = new ArrayList<UpdateItem>();
        int insertCount = this.collectUpdates(toInsert, false, updates);
        int removeCount = this.collectUpdates(toRemove, true, updates);
        updates.sort(UpdateItem::compareTo);
        log.debug("{}: Update (Insert={}, Remove={}).", this.traceLogId, insertCount, removeCount);
        if (updates.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        Preconditions.checkArgument(updates.get(0).getItem().getLength() > 0, "No empty items allowed.");
        return ((CompletableFuture)this.applyUpdates(updates.iterator(), timer).thenApply(pageCollection -> this.processModifiedPages((PageCollection)pageCollection, getNextPageId))).thenComposeAsync(pageCollection -> this.writePages((PageCollection)pageCollection, timer), this.executor);
    }

    private int collectUpdates(Collection<? extends ArrayView> items, boolean isRemoval, List<UpdateItem> updates) {
        if (items == null) {
            return 0;
        }
        for (ArrayView arrayView : items) {
            Preconditions.checkArgument(arrayView.getLength() <= this.maxItemSize, "Item exceeds maximum allowed length (%s).", this.maxItemSize);
            updates.add(new UpdateItem(arrayView, isRemoval));
        }
        return items.size();
    }

    private CompletableFuture<PageCollection> applyUpdates(Iterator<UpdateItem> items, TimeoutTimer timer) {
        PageCollection pageCollection = new PageCollection();
        AtomicReference<Object> lastPage = new AtomicReference<Object>(null);
        ArrayList lastPageUpdates = new ArrayList();
        return Futures.loop(items::hasNext, () -> {
            UpdateItem next = (UpdateItem)items.next();
            return this.locatePage(next.getItem(), pageCollection, timer).thenAccept(page -> {
                BTreeSetPage.LeafPage last = (BTreeSetPage.LeafPage)lastPage.get();
                if (page != last) {
                    if (last != null) {
                        last.update(lastPageUpdates);
                    }
                    lastPage.set(page);
                    lastPageUpdates.clear();
                }
                lastPageUpdates.add(next);
            });
        }, this.executor).thenApplyAsync(v -> {
            if (lastPage.get() != null) {
                ((BTreeSetPage.LeafPage)lastPage.get()).update(lastPageUpdates);
            }
            return pageCollection;
        }, this.executor);
    }

    private PageCollection processModifiedPages(PageCollection pageCollection, Supplier<Long> getNewPageId) {
        Collection<BTreeSetPage> candidates = pageCollection.getLeafPages();
        while (!candidates.isEmpty()) {
            TreeModificationContext tmc = new TreeModificationContext(pageCollection);
            for (BTreeSetPage p2 : candidates) {
                if (p2.getItemCount() == 0) {
                    this.deletePage(p2, tmc);
                    continue;
                }
                this.splitPageIfNecessary(p2, getNewPageId, tmc);
            }
            tmc.accept(BTreeSetPage.IndexPage::addChildren, BTreeSetPage.IndexPage::removeChildren, POINTER_COMPARATOR);
            candidates = tmc.getModifiedParents();
        }
        pageCollection.getIndexPages().forEach(p -> {
            if (p.isModified()) {
                p.seal();
            }
        });
        return pageCollection;
    }

    private void deletePage(BTreeSetPage p, TreeModificationContext context) {
        if (p.getPagePointer().hasParent()) {
            context.getPageCollection().pageDeleted(p);
            context.deleted(p.getPagePointer());
            log.debug("{}: Deleted empty page {}.", (Object)this.traceLogId, (Object)p.getPagePointer());
        } else if (p.isIndexPage()) {
            p = BTreeSetPage.emptyLeafRoot();
            p.markModified();
            context.getPageCollection().pageUpdated(p);
            log.debug("{}: Replaced empty Index Root with empty Leaf Root.", (Object)this.traceLogId);
        }
    }

    private void splitPageIfNecessary(BTreeSetPage p, Supplier<Long> getNewPageId, TreeModificationContext context) {
        List<BTreeSetPage> splits = p.split(this.maxPageSize, getNewPageId);
        if (splits == null) {
            return;
        }
        if (p.getPagePointer().hasParent()) {
            Preconditions.checkArgument(splits.get(0).getPagePointer().getPageId() == p.getPagePointer().getPageId(), "First split result (%s) not current page (%s).", (Object)splits.get(0).getPagePointer(), (Object)p.getPagePointer());
        } else {
            context.getPageCollection().pageUpdated(BTreeSetPage.emptyIndexRoot());
        }
        splits.forEach(splitPage -> {
            context.getPageCollection().pageUpdated((BTreeSetPage)splitPage);
            context.created(splitPage.getPagePointer());
        });
        log.debug("{}: Page '{}' split into {}: {}.", this.traceLogId, p, splits.size(), splits);
    }

    private CompletableFuture<Void> writePages(@NonNull PageCollection pageCollection, TimeoutTimer timer) {
        if (pageCollection == null) {
            throw new NullPointerException("pageCollection is marked non-null but is null");
        }
        HashSet<Long> processedPageIds = new HashSet<Long>();
        ArrayList<Map.Entry<Long, ArrayView>> toWrite = new ArrayList<Map.Entry<Long, ArrayView>>();
        this.collectWriteCandidates(pageCollection.getLeafPages(), toWrite, processedPageIds, pageCollection);
        this.collectWriteCandidates(pageCollection.getIndexPages(), toWrite, processedPageIds, pageCollection);
        this.collectWriteCandidates(pageCollection.getDeletedPagesParents(), toWrite, processedPageIds, pageCollection);
        log.debug("{}: Persist (Updates={}, Deletions={}).", this.traceLogId, toWrite.size(), pageCollection.getDeletedPageIds().size());
        return this.update.apply(toWrite, pageCollection.getDeletedPageIds(), timer.getRemaining());
    }

    private void collectWriteCandidates(Collection<BTreeSetPage> candidates, List<Map.Entry<Long, ArrayView>> toWrite, Set<Long> processedIds, PageCollection pageCollection) {
        while (!candidates.isEmpty()) {
            ArrayList<BTreeSetPage> next = new ArrayList<BTreeSetPage>();
            candidates.stream().filter(p -> p.isModified() && !processedIds.contains(p.getPagePointer().getPageId())).forEach(p -> {
                toWrite.add(new AbstractMap.SimpleImmutableEntry<Long, ArrayView>(p.getPagePointer().getPageId(), p.getData()));
                BTreeSetPage parent = pageCollection.get(p.getPagePointer().getParentPageId());
                assert (p.getPagePointer().hasParent() == (parent != null));
                processedIds.add(p.getPagePointer().getPageId());
                if (parent != null) {
                    next.add(parent);
                }
            });
            candidates = next;
        }
    }

    public AsyncIterator<List<ArrayView>> iterator(@Nullable ArrayView firstItem, boolean firstItemInclusive, @Nullable ArrayView lastItem, boolean lastItemInclusive, @NonNull Duration fetchTimeout) {
        if (fetchTimeout == null) {
            throw new NullPointerException("fetchTimeout is marked non-null but is null");
        }
        return new ItemIterator(firstItem, firstItemInclusive, lastItem, lastItemInclusive, this::locatePage, fetchTimeout);
    }

    private CompletableFuture<BTreeSetPage.LeafPage> locatePage(ArrayView item, PageCollection pageCollection, TimeoutTimer timer) {
        AtomicReference<PagePointer> pagePointer = new AtomicReference<PagePointer>(PagePointer.root());
        CompletableFuture<BTreeSetPage.LeafPage> result = new CompletableFuture<BTreeSetPage.LeafPage>();
        CompletableFuture<Void> loop = Futures.loop(() -> !result.isDone(), () -> this.fetchPage((PagePointer)pagePointer.get(), pageCollection, timer.getRemaining()).thenAccept(page -> {
            if (page.isIndexPage()) {
                pagePointer.set(((BTreeSetPage.IndexPage)page).getChildPage(item, 0));
            } else {
                result.complete((BTreeSetPage.LeafPage)page);
            }
        }), this.executor);
        Futures.exceptionListener(loop, result::completeExceptionally);
        return result;
    }

    private CompletableFuture<BTreeSetPage> fetchPage(PagePointer pagePointer, PageCollection pageCollection, Duration timeout) {
        BTreeSetPage fromCache = pageCollection.get(pagePointer.getPageId());
        if (fromCache != null) {
            return CompletableFuture.completedFuture(fromCache);
        }
        return this.read.apply(pagePointer.getPageId(), timeout).thenApply(data -> {
            BTreeSetPage page;
            if (data == null) {
                Preconditions.checkArgument(!pagePointer.hasParent(), "Missing page contents for %s.", (Object)pagePointer);
                page = BTreeSetPage.emptyLeafRoot();
                log.debug("{}: Initialized empty root.", (Object)this.traceLogId);
            } else {
                page = BTreeSetPage.parse(pagePointer, data);
                log.debug("{}: Loaded page {}.", (Object)this.traceLogId, (Object)page);
            }
            pageCollection.pageLoaded(page);
            return page;
        });
    }

    @FunctionalInterface
    public static interface PersistPages {
        public CompletableFuture<Void> apply(List<Map.Entry<Long, ArrayView>> var1, Collection<Long> var2, Duration var3);
    }

    @FunctionalInterface
    public static interface ReadPage {
        public CompletableFuture<ArrayView> apply(long var1, Duration var3);
    }
}

