/*
 * Decompiled with CFR 0.152.
 */
package io.pravega.common.util.btree.sets;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.pravega.common.util.ArrayView;
import io.pravega.common.util.BitConverter;
import io.pravega.common.util.ByteArrayComparator;
import io.pravega.common.util.ByteArraySegment;
import io.pravega.common.util.IllegalDataFormatException;
import io.pravega.common.util.btree.SearchResult;
import io.pravega.common.util.btree.sets.BTreeSet;
import io.pravega.common.util.btree.sets.PagePointer;
import io.pravega.common.util.btree.sets.UpdateItem;
import java.beans.ConstructorProperties;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.function.Supplier;
import javax.annotation.concurrent.NotThreadSafe;
import lombok.Generated;
import lombok.NonNull;

@NotThreadSafe
abstract class BTreeSetPage {
    private static final byte CURRENT_VERSION = 0;
    private static final int VERSION_OFFSET = 0;
    private static final int VERSION_LENGTH = 1;
    private static final int FLAGS_OFFSET = 1;
    private static final int FLAGS_LENGTH = 1;
    private static final byte FLAG_NONE = 0;
    private static final byte FLAG_INDEX_PAGE = 1;
    private static final int TOTAL_SIZE_OFFSET = 2;
    private static final int TOTAL_SIZE_LENGTH = 4;
    private static final int COUNT_OFFSET = 6;
    private static final int COUNT_LENGTH = 4;
    private static final int ITEM_OFFSETS_OFFSET = 10;
    private static final int ITEM_OFFSET_LENGTH = 4;
    private static final int PAGE_ID_LENGTH = 8;
    @VisibleForTesting
    static final int HEADER_FOOTER_LENGTH = 18;
    private static final Comparator<ArrayView> COMPARATOR = BTreeSet.COMPARATOR;
    private final PagePointer pagePointer;
    private ArrayView data;
    private int itemCount;
    private boolean modified;

    private BTreeSetPage(@NonNull PagePointer pagePointer) {
        if (pagePointer == null) {
            throw new NullPointerException("pagePointer is marked non-null but is null");
        }
        this.pagePointer = pagePointer;
        this.data = this.newContents(0, 0, this.pagePointer.getPageId());
        this.itemCount = 0;
        this.modified = false;
    }

    private BTreeSetPage(@NonNull PagePointer pagePointer, @NonNull ArrayView data) {
        if (pagePointer == null) {
            throw new NullPointerException("pagePointer is marked non-null but is null");
        }
        if (data == null) {
            throw new NullPointerException("data is marked non-null but is null");
        }
        this.pagePointer = pagePointer;
        this.loadContents(data);
        this.modified = false;
    }

    static BTreeSetPage parse(PagePointer pagePointer, @NonNull ArrayView contents) {
        if (contents == null) {
            throw new NullPointerException("contents is marked non-null but is null");
        }
        byte version = contents.get(0);
        if (version != 0) {
            throw new IllegalDataFormatException("Unsupported version. PageId=%s, Expected=%s, Actual=%s.", pagePointer.getPageId(), (byte)0, version);
        }
        int size = BitConverter.readInt(contents, 2);
        if (size < 18 || size > contents.getLength()) {
            throw new IllegalDataFormatException("Invalid size. PageId=%s, Expected in range [%s, %s], actual=%s.", pagePointer.getPageId(), 2, contents.getLength(), size);
        }
        if (size < contents.getLength()) {
            contents = contents.slice(0, size);
        }
        long serializedPageId = BitConverter.readLong(contents, contents.getLength() - 8);
        if (pagePointer.getPageId() != serializedPageId) {
            throw new IllegalDataFormatException("Invalid serialized Page Id. Expected=%s, Actual=%s.", pagePointer.getPageId(), serializedPageId);
        }
        byte flags = contents.get(1);
        boolean isIndex = (flags & 1) == 1;
        return isIndex ? new IndexPage(pagePointer, contents) : new LeafPage(pagePointer, contents);
    }

    static LeafPage emptyLeafRoot() {
        return new LeafPage(PagePointer.root());
    }

    static IndexPage emptyIndexRoot() {
        return new IndexPage(PagePointer.root());
    }

    private ArrayView newContents(int itemCount, int contentsSize, long pageId) {
        ByteArraySegment contents = new ByteArraySegment(new byte[18 + itemCount * 4 + contentsSize]);
        contents.set(0, (byte)0);
        contents.set(1, this.getFlags());
        this.setSize(contents, contents.getLength());
        BitConverter.writeInt(contents, 6, itemCount);
        BitConverter.writeLong(contents, contents.getLength() - 8, pageId);
        return contents;
    }

    private void loadContents(ArrayView contents) {
        int itemCount = BitConverter.readInt(contents, 6);
        if (itemCount < 0) {
            throw new IllegalDataFormatException("Invalid ItemCount. PageId=%s, Actual=%s.", this.pagePointer.getPageId(), itemCount);
        }
        this.itemCount = itemCount;
        this.data = contents;
    }

    private byte getFlags() {
        byte result = 0;
        if (this.isIndexPage()) {
            result = (byte)(result | 1);
        }
        return result;
    }

    abstract int getValueLength();

    abstract boolean isIndexPage();

    abstract void seal();

    int size() {
        return this.data.getLength();
    }

    int getContentSize() {
        return this.size() - 18 - this.itemCount * 4;
    }

    void markModified() {
        this.modified = true;
    }

    public String toString() {
        return String.format("%s: %s, Size=%s, ContentSize=%s, Count=%s.", this.isIndexPage() ? "I" : "L", this.pagePointer, this.size(), this.getContentSize(), this.getItemCount());
    }

    ArrayView getItemAt(int position) {
        Preconditions.checkArgument(position >= 0 && position < this.itemCount, "position must be non-negative and smaller than the item count (%s). Given %s.", this.itemCount, position);
        int offset = this.getOffset(position);
        int length = this.getOffset(position + 1) - offset - this.getValueLength();
        return this.data.slice(offset, length);
    }

    private void setItemAt(int position, ArrayView newItem) {
        Preconditions.checkArgument(position >= 0 && position < this.itemCount, "position must be non-negative and smaller than the item count (%s). Given %s.", this.itemCount, position);
        int offset = this.getOffset(position);
        int length = this.getOffset(position + 1) - offset - this.getValueLength();
        int delta = newItem.getLength() - length;
        Preconditions.checkArgument(delta <= 0, "Cannot replace an item (%s) with a bigger one (%s).", length, newItem.getLength());
        this.copyData(newItem, this.data.slice(offset, newItem.getLength()));
        if (delta != 0) {
            for (int pos = position + 1; pos < this.itemCount; ++pos) {
                this.setOffset(this.data, pos, this.getOffset(pos) + delta);
            }
            byte[] array = this.data.array();
            int arrayOffset = this.data.arrayOffset();
            for (int index = offset + length; index < this.data.getLength(); ++index) {
                array[arrayOffset + index + delta] = array[arrayOffset + index];
            }
            this.data = this.data.slice(0, this.data.getLength() + delta);
            this.setSize(this.data, this.data.getLength());
        }
    }

    List<ArrayView> getItems(int firstPos, int lastPos) {
        Preconditions.checkArgument(firstPos <= lastPos, "firstPos must be smaller than or equal to lastPos.");
        Preconditions.checkArgument(firstPos >= 0, "firstPos must be a non-negative integer.");
        Preconditions.checkArgument(lastPos < this.itemCount, "lastPos must be less than %s.", this.itemCount);
        ArrayList<ArrayView> result = new ArrayList<ArrayView>();
        int offset = this.getOffset(firstPos);
        for (int pos = firstPos; pos <= lastPos; ++pos) {
            int nextOffset = this.getOffset(pos + 1);
            result.add(this.data.slice(offset, nextOffset - offset - this.getValueLength()));
            offset = nextOffset;
        }
        return result;
    }

    SearchResult search(@NonNull ArrayView item, int startPos) {
        if (item == null) {
            throw new NullPointerException("item is marked non-null but is null");
        }
        int endPos = this.getItemCount();
        Preconditions.checkArgument(startPos >= 0 && startPos <= endPos, "startPos must be non-negative and smaller than the number of items.");
        while (startPos < endPos) {
            int midPos = startPos + (endPos - startPos) / 2;
            int c = COMPARATOR.compare(item, this.getItemAt(midPos));
            if (c == 0) {
                return new SearchResult(midPos, true);
            }
            if (c < 0) {
                endPos = midPos;
                continue;
            }
            startPos = midPos + 1;
        }
        return new SearchResult(startPos, false);
    }

    private <T> void update(List<UpdateItem> updates, List<T> values, SerializeValue<T> serializeValue) {
        assert (updates.size() > 0);
        UpdateInfo<T> updateInfo = this.preProcessUpdate(updates, values);
        assert (this.getValueLength() == 0 || updateInfo.inserts.isEmpty() || values.size() == updates.size() && serializeValue != null);
        this.data = this.applyUpdates(updateInfo, serializeValue);
        this.itemCount = updateInfo.newCount;
        assert (this.getContentSize() == updateInfo.newContentSize);
        this.markModified();
    }

    private <T> UpdateInfo<T> preProcessUpdate(List<UpdateItem> updates, List<T> values) {
        Preconditions.checkArgument(updates.get(0).getItem().getLength() > 0, "No empty items allowed.");
        HashSet<Integer> removedPositions = new HashSet<Integer>();
        ArrayList inserts = new ArrayList();
        int sizeDelta = 0;
        ArrayView lastItem = null;
        int lastPos = 0;
        for (int i = 0; i < updates.size(); ++i) {
            UpdateItem u = updates.get(i);
            if (lastItem != null) {
                Preconditions.checkArgument(COMPARATOR.compare(lastItem, u.getItem()) < 0, "Items must be sorted and no duplicates are allowed.");
            }
            SearchResult searchResult = this.search(u.getItem(), lastPos);
            int itemLength = u.getItem().getLength() + this.getValueLength();
            if (searchResult.isExactMatch()) {
                if (u.isRemoval()) {
                    removedPositions.add(searchResult.getPosition());
                    sizeDelta -= itemLength;
                }
            } else if (!u.isRemoval()) {
                int newPos = searchResult.getPosition() - removedPositions.size() + inserts.size();
                inserts.add(new InsertInfo<Object>(updates.get(i).getItem(), (values == null ? null : (Object)values.get(i)), newPos));
                sizeDelta += itemLength;
            }
            lastPos = searchResult.getPosition();
            lastItem = u.getItem();
        }
        int newCount = this.getItemCount() + inserts.size() - removedPositions.size();
        int newContentSize = this.getContentSize() + sizeDelta;
        assert (newContentSize >= 0);
        return new UpdateInfo(removedPositions, inserts, newCount, newContentSize);
    }

    private <T> ArrayView applyUpdates(UpdateInfo<T> updates, SerializeValue<T> serializeValue) {
        ArrayView newData = this.newContents(updates.newCount, updates.newContentSize, this.pagePointer.getPageId());
        if (updates.newCount == 0) {
            return newData;
        }
        int targetPos = 0;
        int targetOffset = 10 + updates.newCount * 4;
        int insertIndex = 0;
        for (int sourcePos = 0; sourcePos < this.itemCount; ++sourcePos) {
            ArrayView sourceItem = this.getItemAt(sourcePos);
            while (insertIndex < updates.inserts.size() && COMPARATOR.compare(sourceItem, updates.inserts.get((int)insertIndex).item) > 0) {
                InsertInfo insert = updates.inserts.get(insertIndex);
                this.insert(insert, newData, targetPos, targetOffset, serializeValue);
                targetOffset += insert.item.getLength() + this.getValueLength();
                ++insertIndex;
                ++targetPos;
            }
            if (updates.removedPositions.contains(sourcePos)) continue;
            this.setOffset(newData, targetPos, targetOffset);
            ArrayView sourceSlice = this.data.slice(this.getOffset(sourcePos), sourceItem.getLength() + this.getValueLength());
            ArrayView targetSlice = newData.slice(targetOffset, sourceSlice.getLength());
            this.copyData(sourceSlice, targetSlice);
            targetOffset += targetSlice.getLength();
            ++targetPos;
        }
        while (insertIndex < updates.inserts.size()) {
            InsertInfo insert = updates.inserts.get(insertIndex);
            this.insert(insert, newData, targetPos, targetOffset, serializeValue);
            targetOffset += insert.item.getLength() + this.getValueLength();
            ++insertIndex;
            ++targetPos;
        }
        return newData;
    }

    private <T> void insert(InsertInfo<T> insert, ArrayView dataBuffer, int targetPos, int targetOffset, SerializeValue<T> serializeValue) {
        this.setOffset(dataBuffer, targetPos, targetOffset);
        this.copyData(insert.item, dataBuffer.slice(targetOffset, insert.item.getLength()));
        if (insert.value != null) {
            serializeValue.accept(dataBuffer.slice(targetOffset + insert.item.getLength(), this.getValueLength()), 0, insert.value);
        }
    }

    List<BTreeSetPage> split(int maxPageSize, @NonNull Supplier<Long> getNewPageId) {
        if (getNewPageId == null) {
            throw new NullPointerException("getNewPageId is marked non-null but is null");
        }
        if (this.size() <= maxPageSize) {
            return null;
        }
        int itemCount = this.getItemCount();
        int adjustedContentLength = this.size() - 18;
        int maxContentLength = maxPageSize - 18;
        int splitCount = adjustedContentLength / maxContentLength;
        if (adjustedContentLength % maxContentLength != 0) {
            ++splitCount;
        }
        int splitThreshold = Math.min(maxContentLength, adjustedContentLength / splitCount);
        ArrayList<BTreeSetPage> result = new ArrayList<BTreeSetPage>();
        int sourcePos = 0;
        int currentPageSouceIndex = this.getOffset(sourcePos);
        while (sourcePos < itemCount) {
            int newContentSize;
            int itemLength;
            ArrayList<Integer> newOffsets = new ArrayList<Integer>();
            ArrayView firstKey = null;
            int itemSourceIndex = currentPageSouceIndex;
            for (newContentSize = 0; sourcePos < itemCount && newContentSize < splitThreshold; newContentSize += itemLength, ++sourcePos) {
                int nextIndex = this.getOffset(sourcePos + 1);
                itemLength = nextIndex - itemSourceIndex + 4;
                if (firstKey == null) {
                    firstKey = this.getItemAt(sourcePos);
                } else if (newContentSize + itemLength > maxContentLength) break;
                newOffsets.add(itemSourceIndex - currentPageSouceIndex);
                itemSourceIndex = nextIndex;
            }
            assert (firstKey != null);
            assert (newOffsets.size() > 0);
            assert (newContentSize <= maxContentLength);
            PagePointer pagePointer = this.getSplitPagePointer(firstKey, getNewPageId, result.isEmpty());
            result.add(this.createSplitPage(pagePointer, currentPageSouceIndex, newContentSize, newOffsets));
            currentPageSouceIndex = itemSourceIndex;
        }
        return result;
    }

    private PagePointer getSplitPagePointer(ArrayView firstKey, Supplier<Long> getNewPageId, boolean isFirstItem) {
        long pageId;
        long parentId;
        if (isFirstItem) {
            ArrayView arrayView = firstKey = this.pagePointer.getKey() == null ? firstKey : this.pagePointer.getKey();
        }
        if (this.pagePointer.hasParent()) {
            parentId = this.pagePointer.getParentPageId();
            pageId = isFirstItem ? this.pagePointer.getPageId() : getNewPageId.get().longValue();
        } else {
            parentId = -1L;
            pageId = getNewPageId.get();
        }
        return new PagePointer(firstKey, pageId, parentId);
    }

    private BTreeSetPage createSplitPage(PagePointer pointer, int fromIndex, int length, List<Integer> relativeOffsets) {
        int offsetsLength = 4 * relativeOffsets.size();
        int offsetAdjustment = 10 + offsetsLength;
        for (int i = 0; i < relativeOffsets.size(); ++i) {
            relativeOffsets.set(i, relativeOffsets.get(i) + offsetAdjustment);
        }
        int actualContentLength = length - offsetsLength;
        ArrayView newPageContents = this.newContents(relativeOffsets.size(), actualContentLength, pointer.getPageId());
        for (int i = 0; i < relativeOffsets.size(); ++i) {
            this.setOffset(newPageContents, i, relativeOffsets.get(i));
        }
        ArrayView sourceSlice = this.data.slice(fromIndex, actualContentLength);
        ArrayView targetSlice = newPageContents.slice(relativeOffsets.get(0), actualContentLength);
        this.copyData(sourceSlice, targetSlice);
        BTreeSetPage result = BTreeSetPage.parse(pointer, newPageContents);
        Preconditions.checkState(result.getItemCount() > 0, "Page split resulted in empty page.");
        result.markModified();
        result.seal();
        return result;
    }

    private ArrayView getValueAt(int position) {
        Preconditions.checkArgument(position >= 0 && position < this.itemCount, "position must be non-negative and smaller than the item count (%s). Given %s.", this.itemCount, position);
        int offset = this.getOffset(position + 1) - this.getValueLength();
        return this.data.slice(offset, this.getValueLength());
    }

    private int getOffset(int position) {
        assert (position >= 0 && position <= this.itemCount);
        if (position == this.itemCount) {
            return this.data.getLength() - 8;
        }
        return BitConverter.readInt(this.data, 10 + position * 4);
    }

    private void setOffset(ArrayView contents, int position, int offset) {
        BitConverter.writeInt(contents, 10 + position * 4, offset);
    }

    private void setSize(ArrayView contents, int size) {
        BitConverter.writeInt(contents, 2, size);
    }

    private void copyData(ArrayView from, ArrayView to) {
        from.copyTo(to.array(), to.arrayOffset(), from.getLength());
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public PagePointer getPagePointer() {
        return this.pagePointer;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public ArrayView getData() {
        return this.data;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public int getItemCount() {
        return this.itemCount;
    }

    @SuppressFBWarnings(justification="generated code")
    @Generated
    public boolean isModified() {
        return this.modified;
    }

    @FunctionalInterface
    private static interface SerializeValue<T> {
        public void accept(ArrayView var1, int var2, T var3);
    }

    private static class InsertInfo<T> {
        final ArrayView item;
        final T value;
        final int targetPos;

        @ConstructorProperties(value={"item", "value", "targetPos"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public InsertInfo(ArrayView item, T value, int targetPos) {
            this.item = item;
            this.value = value;
            this.targetPos = targetPos;
        }
    }

    private static class UpdateInfo<T> {
        final Collection<Integer> removedPositions;
        final List<InsertInfo<T>> inserts;
        final int newCount;
        final int newContentSize;

        @ConstructorProperties(value={"removedPositions", "inserts", "newCount", "newContentSize"})
        @SuppressFBWarnings(justification="generated code")
        @Generated
        public UpdateInfo(Collection<Integer> removedPositions, List<InsertInfo<T>> inserts, int newCount, int newContentSize) {
            this.removedPositions = removedPositions;
            this.inserts = inserts;
            this.newCount = newCount;
            this.newContentSize = newContentSize;
        }
    }

    static class LeafPage
    extends BTreeSetPage {
        LeafPage(PagePointer pagePointer) {
            super(pagePointer);
        }

        LeafPage(PagePointer pagePointer, ArrayView contents) {
            super(pagePointer, contents);
        }

        @Override
        int getValueLength() {
            return 0;
        }

        @Override
        boolean isIndexPage() {
            return false;
        }

        @Override
        void seal() {
            Preconditions.checkState(this.getItemCount() > 0, "Leaf Page split resulted in empty page.");
        }

        void update(List<UpdateItem> updates) {
            ((BTreeSetPage)this).update(updates, null, null);
        }
    }

    static class IndexPage
    extends BTreeSetPage {
        IndexPage(PagePointer pagePointer) {
            super(pagePointer);
        }

        IndexPage(PagePointer pagePointer, ArrayView contents) {
            super(pagePointer, contents);
        }

        @Override
        int getValueLength() {
            return 8;
        }

        @Override
        boolean isIndexPage() {
            return true;
        }

        @Override
        void seal() {
            if (this.getItemCount() > 0) {
                ((BTreeSetPage)this).setItemAt(0, new ByteArraySegment(ByteArrayComparator.getMinValue()));
            }
        }

        void addChildren(@NonNull List<PagePointer> pages) {
            if (pages == null) {
                throw new NullPointerException("pages is marked non-null but is null");
            }
            ArrayList updates = new ArrayList();
            ArrayList values = new ArrayList();
            pages.forEach(p -> {
                updates.add(new UpdateItem(p.getKey(), false));
                values.add(p.getPageId());
            });
            ((BTreeSetPage)this).update(updates, values, BitConverter::writeLong);
        }

        void removeChildren(@NonNull List<PagePointer> pages) {
            if (pages == null) {
                throw new NullPointerException("pages is marked non-null but is null");
            }
            ArrayList updates = new ArrayList();
            pages.forEach(p -> updates.add(new UpdateItem(p.getKey(), true)));
            ((BTreeSetPage)this).update(updates, null, null);
        }

        PagePointer getChildPage(@NonNull ArrayView forItem, int startPos) {
            if (forItem == null) {
                throw new NullPointerException("forItem is marked non-null but is null");
            }
            SearchResult searchResult = this.search(forItem, startPos);
            int position = searchResult.getPosition();
            if (!searchResult.isExactMatch()) {
                --position;
            }
            if (position >= 0) {
                ArrayView serializedValue = ((BTreeSetPage)this).getValueAt(position);
                long pageId = BitConverter.readLong(serializedValue, 0);
                return new PagePointer(this.getItemAt(position), pageId, this.getPagePointer().getPageId());
            }
            return null;
        }
    }
}

