/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.accumulo;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import io.airlift.log.Logger;
import io.prestosql.plugin.accumulo.AccumuloErrorCode;
import io.prestosql.plugin.accumulo.AccumuloTableManager;
import io.prestosql.plugin.accumulo.Types;
import io.prestosql.plugin.accumulo.conf.AccumuloConfig;
import io.prestosql.plugin.accumulo.conf.AccumuloSessionProperties;
import io.prestosql.plugin.accumulo.conf.AccumuloTableProperties;
import io.prestosql.plugin.accumulo.index.IndexLookup;
import io.prestosql.plugin.accumulo.index.Indexer;
import io.prestosql.plugin.accumulo.io.AccumuloPageSink;
import io.prestosql.plugin.accumulo.metadata.AccumuloTable;
import io.prestosql.plugin.accumulo.metadata.AccumuloView;
import io.prestosql.plugin.accumulo.metadata.ZooKeeperMetadataManager;
import io.prestosql.plugin.accumulo.model.AccumuloColumnConstraint;
import io.prestosql.plugin.accumulo.model.AccumuloColumnHandle;
import io.prestosql.plugin.accumulo.model.TabletSplitMetadata;
import io.prestosql.plugin.accumulo.serializers.AccumuloRowSerializer;
import io.prestosql.spi.ErrorCodeSupplier;
import io.prestosql.spi.PrestoException;
import io.prestosql.spi.StandardErrorCode;
import io.prestosql.spi.connector.ColumnMetadata;
import io.prestosql.spi.connector.ConnectorSession;
import io.prestosql.spi.connector.ConnectorTableMetadata;
import io.prestosql.spi.connector.SchemaTableName;
import io.prestosql.spi.predicate.Domain;
import io.prestosql.spi.predicate.Marker;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.client.AccumuloSecurityException;
import org.apache.accumulo.core.client.Connector;
import org.apache.accumulo.core.client.IteratorSetting;
import org.apache.accumulo.core.client.Scanner;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.data.Key;
import org.apache.accumulo.core.data.PartialKey;
import org.apache.accumulo.core.data.Range;
import org.apache.accumulo.core.data.Value;
import org.apache.accumulo.core.security.Authorizations;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.io.BinaryComparable;
import org.apache.hadoop.io.Text;

public class AccumuloClient {
    private static final Logger LOG = Logger.get(AccumuloClient.class);
    private static final Splitter COMMA_SPLITTER = Splitter.on((char)',').omitEmptyStrings().trimResults();
    private final ZooKeeperMetadataManager metaManager;
    private final Authorizations auths;
    private final AccumuloTableManager tableManager;
    private final Connector connector;
    private final IndexLookup indexLookup;
    private final String username;

    @Inject
    public AccumuloClient(Connector connector, AccumuloConfig config, ZooKeeperMetadataManager metaManager, AccumuloTableManager tableManager, IndexLookup indexLookup) throws AccumuloException, AccumuloSecurityException {
        this.connector = Objects.requireNonNull(connector, "connector is null");
        this.username = Objects.requireNonNull(config, "config is null").getUsername();
        this.metaManager = Objects.requireNonNull(metaManager, "metaManager is null");
        this.tableManager = Objects.requireNonNull(tableManager, "tableManager is null");
        this.indexLookup = Objects.requireNonNull(indexLookup, "indexLookup is null");
        this.auths = connector.securityOperations().getUserAuthorizations(this.username);
    }

    public AccumuloTable createTable(ConnectorTableMetadata meta) {
        this.validateCreateTable(meta);
        Map tableProperties = meta.getProperties();
        String rowIdColumn = AccumuloClient.getRowIdColumn(meta);
        List<AccumuloColumnHandle> columns = AccumuloClient.getColumnHandles(meta, rowIdColumn);
        AccumuloTable table = new AccumuloTable(meta.getTable().getSchemaName(), meta.getTable().getTableName(), columns, rowIdColumn, AccumuloTableProperties.isExternal(tableProperties), AccumuloTableProperties.getSerializerClass(tableProperties), AccumuloTableProperties.getScanAuthorizations(tableProperties));
        this.metaManager.createTableMetadata(table);
        this.tableManager.ensureNamespace(table.getSchema());
        if (!this.tableManager.exists(table.getFullTableName())) {
            this.tableManager.createAccumuloTable(table.getFullTableName());
        }
        this.setLocalityGroups(tableProperties, table);
        this.createIndexTables(table);
        return table;
    }

    private void validateCreateTable(ConnectorTableMetadata meta) {
        AccumuloClient.validateColumns(meta);
        AccumuloClient.validateLocalityGroups(meta);
        if (!AccumuloTableProperties.isExternal(meta.getProperties())) {
            this.validateInternalTable(meta);
        }
    }

    private static void validateColumns(ConnectorTableMetadata meta) {
        ImmutableSet.Builder columnNameBuilder = ImmutableSet.builder();
        for (ColumnMetadata column : meta.getColumns()) {
            if (Types.isMapType(column.getType()) && (Types.isMapType(Types.getKeyType(column.getType())) || Types.isMapType(Types.getValueType(column.getType())) || Types.isArrayType(Types.getKeyType(column.getType())) || Types.isArrayType(Types.getValueType(column.getType())))) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_TABLE_PROPERTY, "Key/value types of a MAP column must be plain types");
            }
            columnNameBuilder.add((Object)column.getName().toLowerCase(Locale.ENGLISH));
        }
        if (columnNameBuilder.build().size() != meta.getColumns().size()) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_TABLE_PROPERTY, "Duplicate column names are not supported");
        }
        Optional<Map<String, Pair<String, String>>> columnMapping = AccumuloTableProperties.getColumnMapping(meta.getProperties());
        if (columnMapping.isPresent()) {
            long distinctMappings = columnMapping.get().values().stream().distinct().count();
            if (distinctMappings != (long)columnMapping.get().size()) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_TABLE_PROPERTY, "Duplicate column family/qualifier pair detected in column mapping, check the value of column_mapping");
            }
            String reservedRowIdColumn = AccumuloPageSink.ROW_ID_COLUMN.toString();
            if (columnMapping.get().values().stream().filter(pair -> ((String)pair.getKey()).equals(reservedRowIdColumn) && ((String)pair.getValue()).equals(reservedRowIdColumn)).count() > 0L) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_TABLE_PROPERTY, String.format("Column familiy/qualifier mapping of %s:%s is reserved", reservedRowIdColumn, reservedRowIdColumn));
            }
        } else if (AccumuloTableProperties.isExternal(meta.getProperties())) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_TABLE_PROPERTY, "Column generation for external tables is not supported, must specify column_mapping");
        }
    }

    private static void validateLocalityGroups(ConnectorTableMetadata meta) {
        Optional<Map<String, Set<String>>> groups = AccumuloTableProperties.getLocalityGroups(meta.getProperties());
        if (!groups.isPresent()) {
            return;
        }
        String rowIdColumn = AccumuloClient.getRowIdColumn(meta);
        for (Map.Entry<String, Set<String>> g : groups.get().entrySet()) {
            if (g.getValue().contains(rowIdColumn)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_TABLE_PROPERTY, "Row ID column cannot be in a locality group");
            }
            int matchingColumns = 0;
            for (ColumnMetadata column : meta.getColumns()) {
                if (!g.getValue().contains(column.getName().toLowerCase(Locale.ENGLISH)) || ++matchingColumns != g.getValue().size()) continue;
                break;
            }
            if (matchingColumns == g.getValue().size()) continue;
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_TABLE_PROPERTY, "Unknown Presto column defined for locality group " + g.getKey());
        }
    }

    private void validateInternalTable(ConnectorTableMetadata meta) {
        String table = AccumuloTable.getFullTableName(meta.getTable());
        String indexTable = Indexer.getIndexTableName(meta.getTable());
        String metricsTable = Indexer.getMetricsTableName(meta.getTable());
        if (this.tableManager.exists(table)) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_EXISTS, "Cannot create internal table when an Accumulo table already exists");
        }
        if (AccumuloTableProperties.getIndexColumns(meta.getProperties()).isPresent() && (this.tableManager.exists(indexTable) || this.tableManager.exists(metricsTable))) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_EXISTS, "Internal table is indexed, but the index table and/or index metrics table(s) already exist");
        }
    }

    private static String getRowIdColumn(ConnectorTableMetadata meta) {
        Optional<String> rowIdColumn = AccumuloTableProperties.getRowId(meta.getProperties());
        return rowIdColumn.orElse(((ColumnMetadata)meta.getColumns().get(0)).getName()).toLowerCase(Locale.ENGLISH);
    }

    private static List<AccumuloColumnHandle> getColumnHandles(ConnectorTableMetadata meta, String rowIdColumn) {
        Map<String, Pair<String, String>> mapping = AccumuloTableProperties.getColumnMapping(meta.getProperties()).orElse(AccumuloClient.autoGenerateMapping(meta.getColumns(), AccumuloTableProperties.getLocalityGroups(meta.getProperties())));
        Optional<List<String>> indexedColumns = AccumuloTableProperties.getIndexColumns(meta.getProperties());
        ImmutableList.Builder cBuilder = ImmutableList.builder();
        for (int ordinal = 0; ordinal < meta.getColumns().size(); ++ordinal) {
            ColumnMetadata cm = (ColumnMetadata)meta.getColumns().get(ordinal);
            if (cm.getName().equalsIgnoreCase(rowIdColumn)) {
                cBuilder.add((Object)new AccumuloColumnHandle(rowIdColumn, Optional.empty(), Optional.empty(), cm.getType(), ordinal, "Accumulo row ID", false));
                continue;
            }
            if (!mapping.containsKey(cm.getName())) {
                throw new InvalidParameterException(String.format("Misconfigured mapping for presto column %s", cm.getName()));
            }
            Pair<String, String> famqual = mapping.get(cm.getName());
            boolean indexed = indexedColumns.isPresent() && indexedColumns.get().contains(cm.getName().toLowerCase(Locale.ENGLISH));
            String comment = String.format("Accumulo column %s:%s. Indexed: %b", famqual.getLeft(), famqual.getRight(), indexed);
            cBuilder.add((Object)new AccumuloColumnHandle(cm.getName(), Optional.of(famqual.getLeft()), Optional.of(famqual.getRight()), cm.getType(), ordinal, comment, indexed));
        }
        return cBuilder.build();
    }

    private void setLocalityGroups(Map<String, Object> tableProperties, AccumuloTable table) {
        Optional<Map<String, Set<String>>> groups = AccumuloTableProperties.getLocalityGroups(tableProperties);
        if (!groups.isPresent()) {
            LOG.debug("No locality groups to set");
            return;
        }
        ImmutableMap.Builder localityGroupsBuilder = ImmutableMap.builder();
        for (Map.Entry<String, Set<String>> g : groups.get().entrySet()) {
            ImmutableSet.Builder familyBuilder = ImmutableSet.builder();
            for (String col : g.getValue()) {
                AccumuloColumnHandle handle = (AccumuloColumnHandle)table.getColumns().stream().filter(x -> x.getName().equals(col)).collect(Collectors.toList()).get(0);
                familyBuilder.add((Object)new Text(handle.getFamily().get()));
            }
            localityGroupsBuilder.put((Object)g.getKey(), (Object)familyBuilder.build());
        }
        ImmutableMap localityGroups = localityGroupsBuilder.build();
        LOG.debug("Setting locality groups: {}", new Object[]{localityGroups});
        this.tableManager.setLocalityGroups(table.getFullTableName(), (Map<String, Set<Text>>)localityGroups);
    }

    private void createIndexTables(AccumuloTable table) {
        if (!table.isIndexed()) {
            return;
        }
        if (!this.tableManager.exists(table.getIndexTableName())) {
            this.tableManager.createAccumuloTable(table.getIndexTableName());
        }
        if (!this.tableManager.exists(table.getMetricsTableName())) {
            this.tableManager.createAccumuloTable(table.getMetricsTableName());
        }
        Map<String, Set<Text>> indexGroups = Indexer.getLocalityGroups(table);
        this.tableManager.setLocalityGroups(table.getIndexTableName(), indexGroups);
        this.tableManager.setLocalityGroups(table.getMetricsTableName(), indexGroups);
        for (IteratorSetting setting : Indexer.getMetricIterators(table)) {
            this.tableManager.setIterator(table.getMetricsTableName(), setting);
        }
    }

    private static Map<String, Pair<String, String>> autoGenerateMapping(List<ColumnMetadata> columns, Optional<Map<String, Set<String>>> groups) {
        HashMap<String, Pair<String, String>> mapping = new HashMap<String, Pair<String, String>>();
        for (ColumnMetadata column : columns) {
            Optional<String> family = AccumuloClient.getColumnLocalityGroup(column.getName(), groups);
            mapping.put(column.getName(), (Pair<String, String>)Pair.of((Object)family.orElse(column.getName()), (Object)column.getName()));
        }
        return mapping;
    }

    private static Optional<String> getColumnLocalityGroup(String columnName, Optional<Map<String, Set<String>>> groups) {
        if (groups.isPresent()) {
            for (Map.Entry<String, Set<String>> group : groups.get().entrySet()) {
                if (!group.getValue().contains(columnName.toLowerCase(Locale.ENGLISH))) continue;
                return Optional.of(group.getKey());
            }
        }
        return Optional.empty();
    }

    public void dropTable(AccumuloTable table) {
        SchemaTableName tableName = new SchemaTableName(table.getSchema(), table.getTable());
        if (this.metaManager.getTable(tableName) != null) {
            this.metaManager.deleteTableMetadata(tableName);
        }
        if (!table.isExternal()) {
            String fullTableName = table.getFullTableName();
            if (this.tableManager.exists(fullTableName)) {
                this.tableManager.deleteAccumuloTable(fullTableName);
            }
            if (table.isIndexed()) {
                String metricsTableName;
                String indexTableName = Indexer.getIndexTableName(tableName);
                if (this.tableManager.exists(indexTableName)) {
                    this.tableManager.deleteAccumuloTable(indexTableName);
                }
                if (this.tableManager.exists(metricsTableName = Indexer.getMetricsTableName(tableName))) {
                    this.tableManager.deleteAccumuloTable(metricsTableName);
                }
            }
        }
    }

    public void renameTable(SchemaTableName oldName, SchemaTableName newName) {
        if (!oldName.getSchemaName().equals(newName.getSchemaName())) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Accumulo does not support renaming tables to different namespaces (schemas)");
        }
        AccumuloTable oldTable = this.getTable(oldName);
        if (oldTable == null) {
            throw new io.prestosql.spi.connector.TableNotFoundException(oldName);
        }
        AccumuloTable newTable = new AccumuloTable(oldTable.getSchema(), newName.getTableName(), oldTable.getColumns(), oldTable.getRowId(), oldTable.isExternal(), oldTable.getSerializerClassName(), oldTable.getScanAuthorizations());
        if (!this.tableManager.exists(oldTable.getFullTableName())) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_DNE, String.format("Table %s does not exist", oldTable.getFullTableName()));
        }
        if (this.tableManager.exists(newTable.getFullTableName())) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_EXISTS, String.format("Table %s already exists", newTable.getFullTableName()));
        }
        this.renameIndexTables(oldTable, newTable);
        this.tableManager.renameAccumuloTable(oldTable.getFullTableName(), newTable.getFullTableName());
        this.metaManager.deleteTableMetadata(oldTable.getSchemaTableName());
        this.metaManager.createTableMetadata(newTable);
    }

    private void renameIndexTables(AccumuloTable oldTable, AccumuloTable newTable) {
        if (!oldTable.isIndexed()) {
            return;
        }
        if (!this.tableManager.exists(oldTable.getIndexTableName())) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_DNE, String.format("Table %s does not exist", oldTable.getIndexTableName()));
        }
        if (this.tableManager.exists(newTable.getIndexTableName())) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_EXISTS, String.format("Table %s already exists", newTable.getIndexTableName()));
        }
        if (!this.tableManager.exists(oldTable.getMetricsTableName())) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_DNE, String.format("Table %s does not exist", oldTable.getMetricsTableName()));
        }
        if (this.tableManager.exists(newTable.getMetricsTableName())) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.ACCUMULO_TABLE_EXISTS, String.format("Table %s already exists", newTable.getMetricsTableName()));
        }
        this.tableManager.renameAccumuloTable(oldTable.getIndexTableName(), newTable.getIndexTableName());
        this.tableManager.renameAccumuloTable(oldTable.getMetricsTableName(), newTable.getMetricsTableName());
    }

    public void createView(SchemaTableName viewName, String viewData) {
        if (this.getSchemaNames().contains(viewName.getSchemaName())) {
            if (this.getViewNames(viewName.getSchemaName()).contains(viewName.getTableName())) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, "View already exists");
            }
            if (this.getTableNames(viewName.getSchemaName()).contains(viewName.getTableName())) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_VIEW, "View already exists as data table");
            }
        }
        this.metaManager.createViewMetadata(new AccumuloView(viewName.getSchemaName(), viewName.getTableName(), viewData));
    }

    public void createOrReplaceView(SchemaTableName viewName, String viewData) {
        if (this.getView(viewName) != null) {
            this.metaManager.deleteViewMetadata(viewName);
        }
        this.metaManager.createViewMetadata(new AccumuloView(viewName.getSchemaName(), viewName.getTableName(), viewData));
    }

    public void dropView(SchemaTableName viewName) {
        this.metaManager.deleteViewMetadata(viewName);
    }

    public void renameColumn(AccumuloTable table, String source, String target) {
        if (!table.getColumns().stream().anyMatch(columnHandle -> columnHandle.getName().equalsIgnoreCase(source))) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NOT_FOUND, String.format("Failed to find source column %s to rename to %s", source, target));
        }
        ImmutableList.Builder newColumnList = ImmutableList.builder();
        for (AccumuloColumnHandle columnHandle2 : table.getColumns()) {
            if (columnHandle2.getName().equalsIgnoreCase(source)) {
                newColumnList.add((Object)new AccumuloColumnHandle(target, columnHandle2.getFamily(), columnHandle2.getQualifier(), columnHandle2.getType(), columnHandle2.getOrdinal(), columnHandle2.getComment(), columnHandle2.isIndexed()));
                continue;
            }
            newColumnList.add((Object)columnHandle2);
        }
        AccumuloTable newTable = new AccumuloTable(table.getSchema(), table.getTable(), (List<AccumuloColumnHandle>)newColumnList.build(), table.getRowId().equalsIgnoreCase(source) ? target : table.getRowId(), table.isExternal(), table.getSerializerClassName(), table.getScanAuthorizations());
        this.metaManager.deleteTableMetadata(new SchemaTableName(table.getSchema(), table.getTable()));
        this.metaManager.createTableMetadata(newTable);
    }

    public Set<String> getSchemaNames() {
        return this.metaManager.getSchemaNames();
    }

    public Set<String> getTableNames(String schema) {
        Objects.requireNonNull(schema, "schema is null");
        return this.metaManager.getTableNames(schema);
    }

    public AccumuloTable getTable(SchemaTableName table) {
        Objects.requireNonNull(table, "schema table name is null");
        return this.metaManager.getTable(table);
    }

    public Set<String> getViewNames(String schema) {
        Objects.requireNonNull(schema, "schema is null");
        return this.metaManager.getViewNames(schema);
    }

    public AccumuloView getView(SchemaTableName viewName) {
        Objects.requireNonNull(viewName, "schema table name is null");
        return this.metaManager.getView(viewName);
    }

    public List<TabletSplitMetadata> getTabletSplits(ConnectorSession session, String schema, String table, Optional<Domain> rowIdDomain, List<AccumuloColumnConstraint> constraints, AccumuloRowSerializer serializer) {
        try {
            Authorizations auths;
            String tableName = AccumuloTable.getFullTableName(schema, table);
            LOG.debug("Getting tablet splits for table %s", new Object[]{tableName});
            Collection<Range> rowIdRanges = AccumuloClient.getRangesFromDomain(rowIdDomain, serializer);
            ArrayList<TabletSplitMetadata> tabletSplits = new ArrayList<TabletSplitMetadata>();
            if (AccumuloSessionProperties.isOptimizeIndexEnabled(session) && this.indexLookup.applyIndex(schema, table, session, constraints, rowIdRanges, tabletSplits, serializer, auths = this.getScanAuthorizations(session, schema, table))) {
                return tabletSplits;
            }
            Collection<Range> splitRanges = AccumuloSessionProperties.isOptimizeSplitRangesEnabled(session) ? this.splitByTabletBoundaries(tableName, rowIdRanges) : rowIdRanges;
            boolean fetchTabletLocations = AccumuloSessionProperties.isOptimizeLocalityEnabled(session);
            LOG.debug("Fetching tablet locations: %s", new Object[]{fetchTabletLocations});
            for (Range range : splitRanges) {
                if (fetchTabletLocations) {
                    tabletSplits.add(new TabletSplitMetadata(this.getTabletLocation(tableName, range.getStartKey()), (List<Range>)ImmutableList.of((Object)range)));
                    continue;
                }
                tabletSplits.add(new TabletSplitMetadata(Optional.empty(), (List<Range>)ImmutableList.of((Object)range)));
            }
            LOG.debug("Number of splits for table %s is %d with %d ranges", new Object[]{tableName, tabletSplits.size(), splitRanges.size()});
            return tabletSplits;
        }
        catch (Exception e) {
            throw new PrestoException((ErrorCodeSupplier)AccumuloErrorCode.UNEXPECTED_ACCUMULO_ERROR, "Failed to get splits from Accumulo", (Throwable)e);
        }
    }

    private Authorizations getScanAuthorizations(ConnectorSession session, String schema, String table) throws AccumuloException, AccumuloSecurityException {
        String sessionScanUser = AccumuloSessionProperties.getScanUsername(session);
        if (sessionScanUser != null) {
            Authorizations scanAuths = this.connector.securityOperations().getUserAuthorizations(sessionScanUser);
            LOG.debug("Using session scan auths for user %s: %s", new Object[]{sessionScanUser, scanAuths});
            return scanAuths;
        }
        AccumuloTable accumuloTable = this.getTable(new SchemaTableName(schema, table));
        if (accumuloTable == null) {
            throw new io.prestosql.spi.connector.TableNotFoundException(new SchemaTableName(schema, table));
        }
        Optional<String> strAuths = accumuloTable.getScanAuthorizations();
        if (strAuths.isPresent()) {
            Authorizations scanAuths = new Authorizations((String[])Iterables.toArray((Iterable)COMMA_SPLITTER.split((CharSequence)strAuths.get()), String.class));
            LOG.debug("scan_auths table property set, using: %s", new Object[]{scanAuths});
            return scanAuths;
        }
        LOG.debug("scan_auths table property not set, using connector auths: %s", new Object[]{this.auths});
        return this.auths;
    }

    private Collection<Range> splitByTabletBoundaries(String tableName, Collection<Range> ranges) throws TableNotFoundException, AccumuloException, AccumuloSecurityException {
        ImmutableSet.Builder rangeBuilder = ImmutableSet.builder();
        for (Range range : ranges) {
            if (range.getStartKey() != null && range.getEndKey() != null && range.getStartKey().equals((Object)range.getEndKey())) {
                rangeBuilder.add((Object)range);
                continue;
            }
            rangeBuilder.addAll((Iterable)this.connector.tableOperations().splitRangeByTablets(tableName, range, Integer.MAX_VALUE));
        }
        return rangeBuilder.build();
    }

    private Optional<String> getTabletLocation(String table, Key key) {
        try {
            String tableId = (String)this.connector.tableOperations().tableIdMap().get(table);
            Scanner scanner = this.connector.createScanner("accumulo.metadata", this.auths);
            scanner.fetchColumnFamily(new Text("loc"));
            Key defaultTabletRow = new Key((CharSequence)(tableId + '<'));
            Key start = new Key((CharSequence)tableId);
            Key end = defaultTabletRow.followingKey(PartialKey.ROW);
            scanner.setRange(new Range(start, end));
            Optional<String> location = Optional.empty();
            if (key == null) {
                Iterator iter = scanner.iterator();
                if (iter.hasNext()) {
                    location = Optional.of(((Value)((Map.Entry)iter.next()).getValue()).toString());
                }
            } else {
                Text splitCompareKey = new Text();
                key.getRow(splitCompareKey);
                Text scannedCompareKey = new Text();
                for (Map.Entry entry : scanner) {
                    byte[] keyBytes = ((Key)entry.getKey()).getRow().copyBytes();
                    if (keyBytes[keyBytes.length - 1] == 60) {
                        location = Optional.of(((Value)entry.getValue()).toString());
                        break;
                    }
                    scannedCompareKey.set(keyBytes, 3, keyBytes.length - 3);
                    if (scannedCompareKey.getLength() <= 0) continue;
                    int compareTo = splitCompareKey.compareTo((BinaryComparable)scannedCompareKey);
                    if (compareTo > 0) break;
                    location = Optional.of(((Value)entry.getValue()).toString());
                }
                scanner.close();
            }
            return location.isPresent() ? location : this.getDefaultTabletLocation(table);
        }
        catch (Exception e) {
            LOG.error("Failed to get tablet location, returning dummy location", new Object[]{e});
            return Optional.empty();
        }
    }

    private Optional<String> getDefaultTabletLocation(String fulltable) {
        try {
            String tableId = (String)this.connector.tableOperations().tableIdMap().get(fulltable);
            Scanner scan = this.connector.createScanner("accumulo.metadata", this.connector.securityOperations().getUserAuthorizations(this.username));
            scan.fetchColumnFamily(new Text("loc"));
            scan.setRange(new Range((CharSequence)(tableId + '<')));
            Optional<String> location = Optional.empty();
            for (Map.Entry entry : scan) {
                if (location.isPresent()) {
                    throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.FUNCTION_IMPLEMENTATION_ERROR, "Scan for default tablet returned more than one entry");
                }
                location = Optional.of(((Value)entry.getValue()).toString());
            }
            scan.close();
            return location;
        }
        catch (Exception e) {
            LOG.error("Failed to get tablet location, returning dummy location", new Object[]{e});
            return Optional.empty();
        }
    }

    public static Collection<Range> getRangesFromDomain(Optional<Domain> domain, AccumuloRowSerializer serializer) throws io.prestosql.spi.connector.TableNotFoundException {
        if (!domain.isPresent()) {
            return ImmutableSet.of((Object)new Range());
        }
        ImmutableSet.Builder rangeBuilder = ImmutableSet.builder();
        for (io.prestosql.spi.predicate.Range range : domain.get().getValues().getRanges().getOrderedRanges()) {
            rangeBuilder.add((Object)AccumuloClient.getRangeFromPrestoRange(range, serializer));
        }
        return rangeBuilder.build();
    }

    private static Range getRangeFromPrestoRange(io.prestosql.spi.predicate.Range prestoRange, AccumuloRowSerializer serializer) throws io.prestosql.spi.connector.TableNotFoundException {
        Range accumuloRange;
        if (prestoRange.isAll()) {
            accumuloRange = new Range();
        } else if (prestoRange.isSingleValue()) {
            Text split = new Text(serializer.encode(prestoRange.getType(), prestoRange.getSingleValue()));
            accumuloRange = new Range(split);
        } else if (prestoRange.getLow().isLowerUnbounded()) {
            boolean inclusive = prestoRange.getHigh().getBound() == Marker.Bound.EXACTLY;
            Text split = new Text(serializer.encode(prestoRange.getType(), prestoRange.getHigh().getValue()));
            accumuloRange = new Range(null, false, split, inclusive);
        } else if (prestoRange.getHigh().isUpperUnbounded()) {
            boolean inclusive = prestoRange.getLow().getBound() == Marker.Bound.EXACTLY;
            Text split = new Text(serializer.encode(prestoRange.getType(), prestoRange.getLow().getValue()));
            accumuloRange = new Range(split, inclusive, null, false);
        } else {
            boolean startKeyInclusive = prestoRange.getLow().getBound() == Marker.Bound.EXACTLY;
            Text startSplit = new Text(serializer.encode(prestoRange.getType(), prestoRange.getLow().getValue()));
            boolean endKeyInclusive = prestoRange.getHigh().getBound() == Marker.Bound.EXACTLY;
            Text endSplit = new Text(serializer.encode(prestoRange.getType(), prestoRange.getHigh().getValue()));
            accumuloRange = new Range(startSplit, startKeyInclusive, endSplit, endKeyInclusive);
        }
        return accumuloRange;
    }
}

