/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.plugin.postgresql;

import io.prestosql.plugin.postgresql.PostgreSqlQueryRunner;
import io.prestosql.plugin.postgresql.TestingPostgreSqlServer;
import io.prestosql.testing.AbstractTestIntegrationSmokeTest;
import io.prestosql.testing.QueryRunner;
import io.prestosql.tpch.TpchTable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.UUID;
import java.util.stream.Stream;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.annotations.AfterClass;
import org.testng.annotations.Test;

@Test
public class TestPostgreSqlIntegrationSmokeTest
extends AbstractTestIntegrationSmokeTest {
    protected TestingPostgreSqlServer postgreSqlServer;

    protected QueryRunner createQueryRunner() throws Exception {
        this.postgreSqlServer = new TestingPostgreSqlServer();
        this.execute("CREATE EXTENSION file_fdw");
        return PostgreSqlQueryRunner.createPostgreSqlQueryRunner(this.postgreSqlServer, TpchTable.CUSTOMER, TpchTable.NATION, TpchTable.ORDERS, TpchTable.REGION);
    }

    @AfterClass(alwaysRun=true)
    public final void destroy() {
        this.postgreSqlServer.close();
    }

    @Test
    public void testDropTable() {
        this.assertUpdate("CREATE TABLE test_drop AS SELECT 123 x", 1L);
        Assert.assertTrue((boolean)this.getQueryRunner().tableExists(this.getSession(), "test_drop"));
        this.assertUpdate("DROP TABLE test_drop");
        Assert.assertFalse((boolean)this.getQueryRunner().tableExists(this.getSession(), "test_drop"));
    }

    @Test
    public void testInsert() throws Exception {
        this.execute("CREATE TABLE tpch.test_insert (x bigint, y varchar(100))");
        this.assertUpdate("INSERT INTO test_insert VALUES (123, 'test')", 1L);
        this.assertQuery("SELECT * FROM test_insert", "SELECT 123 x, 'test' y");
        this.assertUpdate("DROP TABLE test_insert");
    }

    @Test
    public void testInsertInPresenceOfNotSupportedColumn() throws Exception {
        this.execute("CREATE TABLE tpch.test_insert_not_supported_column_present(x bigint, y decimal(50,0), z varchar(10))");
        this.assertQuery("SELECT column_name FROM information_schema.columns WHERE table_name = 'test_insert_not_supported_column_present'", "VALUES 'x', 'z'");
        this.assertUpdate("INSERT INTO test_insert_not_supported_column_present (x, z) VALUES (123, 'test')", 1L);
        this.assertQuery("SELECT x, z FROM test_insert_not_supported_column_present", "SELECT 123, 'test'");
        this.assertUpdate("DROP TABLE test_insert_not_supported_column_present");
    }

    @Test
    public void testViews() throws Exception {
        this.execute("CREATE OR REPLACE VIEW tpch.test_view AS SELECT * FROM tpch.orders");
        Assert.assertTrue((boolean)this.getQueryRunner().tableExists(this.getSession(), "test_view"));
        this.assertQuery("SELECT orderkey FROM test_view", "SELECT orderkey FROM orders");
        this.execute("DROP VIEW IF EXISTS tpch.test_view");
    }

    @Test
    public void testMaterializedView() throws Exception {
        this.execute("CREATE MATERIALIZED VIEW tpch.test_mv as SELECT * FROM tpch.orders");
        Assert.assertTrue((boolean)this.getQueryRunner().tableExists(this.getSession(), "test_mv"));
        this.assertQuery("SELECT orderkey FROM test_mv", "SELECT orderkey FROM orders");
        this.execute("DROP MATERIALIZED VIEW tpch.test_mv");
    }

    @Test
    public void testForeignTable() throws Exception {
        this.execute("CREATE SERVER devnull FOREIGN DATA WRAPPER file_fdw");
        this.execute("CREATE FOREIGN TABLE tpch.test_ft (x bigint) SERVER devnull OPTIONS (filename '/dev/null')");
        Assert.assertTrue((boolean)this.getQueryRunner().tableExists(this.getSession(), "test_ft"));
        this.computeActual("SELECT * FROM test_ft");
        this.execute("DROP FOREIGN TABLE tpch.test_ft");
        this.execute("DROP SERVER devnull");
    }

    @Test
    public void testSystemTable() {
        Assertions.assertThat((Iterable)this.computeActual("SHOW TABLES FROM pg_catalog").getOnlyColumnAsSet()).contains(new Object[]{"pg_tables", "pg_views", "pg_type", "pg_index"});
        Assertions.assertThat((Iterable)this.computeActual("SELECT typname FROM pg_catalog.pg_type").getOnlyColumnAsSet()).contains(new Object[]{"char", "text"});
        Assertions.assertThat((Stream)this.computeActual("SELECT tablename FROM pg_catalog.pg_tables WHERE schemaname = 'tpch'").getOnlyColumn()).contains(new Object[]{"orders"});
    }

    @Test
    public void testTableWithNoSupportedColumns() throws Exception {
        String unsupportedDataType = "interval";
        String supportedDataType = "varchar(5)";
        try (AutoCloseable ignore1 = this.withTable("tpch.no_supported_columns", String.format("(c %s)", unsupportedDataType));
             AutoCloseable ignore2 = this.withTable("tpch.supported_columns", String.format("(good %s)", supportedDataType));
             AutoCloseable ignore3 = this.withTable("tpch.no_columns", "()");){
            Assertions.assertThat((Iterable)this.computeActual("SHOW TABLES").getOnlyColumnAsSet()).contains(new Object[]{"orders", "no_supported_columns", "supported_columns", "no_columns"});
            this.assertQueryFails("SELECT c FROM no_supported_columns", "Table 'tpch.no_supported_columns' not found");
            this.assertQueryFails("SELECT * FROM no_supported_columns", "Table 'tpch.no_supported_columns' not found");
            this.assertQueryFails("SELECT 'a' FROM no_supported_columns", "Table 'tpch.no_supported_columns' not found");
            this.assertQueryFails("SELECT c FROM no_columns", "Table 'tpch.no_columns' not found");
            this.assertQueryFails("SELECT * FROM no_columns", "Table 'tpch.no_columns' not found");
            this.assertQueryFails("SELECT 'a' FROM no_columns", "Table 'tpch.no_columns' not found");
            this.assertQueryFails("SELECT c FROM non_existent", ".* Table .*tpch.non_existent.* does not exist");
            this.assertQueryFails("SELECT * FROM non_existent", ".* Table .*tpch.non_existent.* does not exist");
            this.assertQueryFails("SELECT 'a' FROM non_existent", ".* Table .*tpch.non_existent.* does not exist");
            this.assertQuery("SHOW COLUMNS FROM no_supported_columns", "SELECT 'nothing' WHERE false");
            this.assertQuery("SHOW COLUMNS FROM no_columns", "SELECT 'nothing' WHERE false");
            Assertions.assertThat((Stream)this.computeActual("SHOW TABLES").getOnlyColumn()).contains(new Object[]{"orders", "no_supported_columns", "supported_columns", "no_columns"});
            Assertions.assertThat((Stream)this.computeActual("SELECT table_name FROM information_schema.tables WHERE table_schema = 'tpch'").getOnlyColumn()).contains(new Object[]{"orders", "no_supported_columns", "supported_columns", "no_columns"});
            this.assertQuery("SHOW COLUMNS FROM supported_columns", "VALUES ('good', 'varchar(5)', '', '')");
            this.computeActual("SELECT column_name FROM information_schema.columns WHERE table_schema = 'tpch'");
        }
    }

    @Test
    public void testInsertWithFailureDoesNotLeaveBehindOrphanedTable() throws Exception {
        String schemaName = String.format("tmp_schema_%s", UUID.randomUUID().toString().replaceAll("-", ""));
        try (AutoCloseable schema = this.withSchema(schemaName);
             AutoCloseable table = this.withTable(String.format("%s.test_cleanup", schemaName), "(x INTEGER)");){
            this.assertQuery(String.format("SELECT table_name FROM information_schema.tables WHERE table_schema = '%s'", schemaName), "VALUES 'test_cleanup'");
            this.execute(String.format("ALTER TABLE %s.test_cleanup ADD CHECK (x > 0)", schemaName));
            this.assertQueryFails(String.format("INSERT INTO %s.test_cleanup (x) VALUES (0)", schemaName), "ERROR: new row .* violates check constraint [\\s\\S]*");
            this.assertQuery(String.format("SELECT table_name FROM information_schema.tables WHERE table_schema = '%s'", schemaName), "VALUES 'test_cleanup'");
        }
    }

    @Test
    public void testDecimalPredicatePushdown() throws Exception {
        try (AutoCloseable ignoreTable = this.withTable("tpch.test_decimal_pushdown", "(short_decimal decimal(9, 3), long_decimal decimal(30, 10))");){
            this.execute("INSERT INTO tpch.test_decimal_pushdown VALUES (123.321, 123456789.987654321)");
            this.assertQuery("SELECT * FROM tpch.test_decimal_pushdown WHERE short_decimal <= 124", "VALUES (123.321, 123456789.987654321)");
            this.assertQuery("SELECT * FROM tpch.test_decimal_pushdown WHERE long_decimal <= 123456790", "VALUES (123.321, 123456789.987654321)");
            this.assertQuery("SELECT * FROM tpch.test_decimal_pushdown WHERE short_decimal <= 123.321", "VALUES (123.321, 123456789.987654321)");
            this.assertQuery("SELECT * FROM tpch.test_decimal_pushdown WHERE long_decimal <= 123456789.987654321", "VALUES (123.321, 123456789.987654321)");
            this.assertQuery("SELECT * FROM tpch.test_decimal_pushdown WHERE short_decimal = 123.321", "VALUES (123.321, 123456789.987654321)");
            this.assertQuery("SELECT * FROM tpch.test_decimal_pushdown WHERE long_decimal = 123456789.987654321", "VALUES (123.321, 123456789.987654321)");
        }
    }

    @Test
    public void testCharPredicatePushdown() throws Exception {
        try (AutoCloseable ignoreTable = this.withTable("tpch.test_char_pushdown", "(char_1 char(1), char_5 char(5), char_10 char(10))");){
            this.execute("INSERT INTO tpch.test_char_pushdown VALUES('0', '0'    , '0'         ),('1', '12345', '1234567890')");
            this.assertQuery("SELECT * FROM tpch.test_char_pushdown WHERE char_1 = '0' AND char_5 = '0'", "VALUES ('0', '0    ', '0         ')");
            this.assertQuery("SELECT * FROM tpch.test_char_pushdown WHERE char_5 = CHAR'12345' AND char_10 = '1234567890'", "VALUES ('1', '12345', '1234567890')");
            this.assertQuery("SELECT * FROM tpch.test_char_pushdown WHERE char_10 = CHAR'0'", "VALUES ('0', '0    ', '0         ')");
        }
    }

    @Test
    public void testCharTrailingSpace() throws Exception {
        this.execute("CREATE TABLE tpch.char_trailing_space (x char(10))");
        this.assertUpdate("INSERT INTO char_trailing_space VALUES ('test')", 1L);
        this.assertQuery("SELECT * FROM char_trailing_space WHERE x = char 'test'", "VALUES 'test'");
        this.assertQuery("SELECT * FROM char_trailing_space WHERE x = char 'test  '", "VALUES 'test'");
        this.assertQuery("SELECT * FROM char_trailing_space WHERE x = char 'test        '", "VALUES 'test'");
        Assert.assertEquals((int)this.getQueryRunner().execute("SELECT * FROM char_trailing_space WHERE x = char ' test'").getRowCount(), (int)0);
        this.assertUpdate("DROP TABLE char_trailing_space");
    }

    @Test
    public void testInsertIntoNotNullColumn() {
        String createTableSql = String.format("CREATE TABLE %s.tpch.test_insert_not_null (\n   column_a date,\n   column_b date NOT NULL\n)", this.getSession().getCatalog().get());
        this.assertUpdate(createTableSql);
        Assert.assertEquals((Object)this.computeScalar("SHOW CREATE TABLE test_insert_not_null"), (Object)createTableSql);
        this.assertQueryFails("INSERT INTO test_insert_not_null (column_a) VALUES (date '2012-12-31')", "(?s).*null value in column \"column_b\" violates not-null constraint.*");
        this.assertQueryFails("INSERT INTO test_insert_not_null (column_a, column_b) VALUES (date '2012-12-31', null)", "NULL value not allowed for NOT NULL column: column_b");
        this.assertUpdate("ALTER TABLE test_insert_not_null ADD COLUMN column_c BIGINT NOT NULL");
        createTableSql = String.format("CREATE TABLE %s.tpch.test_insert_not_null (\n   column_a date,\n   column_b date NOT NULL,\n   column_c bigint NOT NULL\n)", this.getSession().getCatalog().get());
        Assert.assertEquals((Object)this.computeScalar("SHOW CREATE TABLE test_insert_not_null"), (Object)createTableSql);
        this.assertQueryFails("INSERT INTO test_insert_not_null (column_b) VALUES (date '2012-12-31')", "(?s).*null value in column \"column_c\" violates not-null constraint.*");
        this.assertQueryFails("INSERT INTO test_insert_not_null (column_b, column_c) VALUES (date '2012-12-31', null)", "NULL value not allowed for NOT NULL column: column_c");
        this.assertUpdate("INSERT INTO test_insert_not_null (column_b, column_c) VALUES (date '2012-12-31', 1)", 1L);
        this.assertUpdate("INSERT INTO test_insert_not_null (column_a, column_b, column_c) VALUES (date '2013-01-01', date '2013-01-02', 2)", 1L);
        this.assertQuery("SELECT * FROM test_insert_not_null", "VALUES (NULL, CAST('2012-12-31' AS DATE), 1), (CAST('2013-01-01' AS DATE), CAST('2013-01-02' AS DATE), 2)");
        this.assertUpdate("DROP TABLE test_insert_not_null");
    }

    @Test
    public void testAggregationPushdown() throws Exception {
        this.assertAggregationPushedDown("SELECT count(*) FROM nation");
        this.assertAggregationPushedDown("SELECT count(nationkey) FROM nation");
        this.assertAggregationPushedDown("SELECT count(1) FROM nation");
        this.assertAggregationPushedDown("SELECT count() FROM nation");
        this.assertAggregationPushedDown("SELECT regionkey, min(nationkey) FROM nation GROUP BY regionkey");
        this.assertAggregationPushedDown("SELECT regionkey, max(nationkey) FROM nation GROUP BY regionkey");
        this.assertAggregationPushedDown("SELECT regionkey, sum(nationkey) FROM nation GROUP BY regionkey");
        this.assertAggregationPushedDown("SELECT regionkey, avg(nationkey) FROM nation GROUP BY regionkey");
        try (AutoCloseable ignoreTable = this.withTable("tpch.test_aggregation_pushdown", "(short_decimal decimal(9, 3), long_decimal decimal(30, 10))");){
            this.execute("INSERT INTO tpch.test_aggregation_pushdown VALUES (100.000, 100000000.000000000)");
            this.execute("INSERT INTO tpch.test_aggregation_pushdown VALUES (123.321, 123456789.987654321)");
            this.assertAggregationPushedDown("SELECT min(short_decimal), min(long_decimal) FROM test_aggregation_pushdown");
            this.assertAggregationPushedDown("SELECT max(short_decimal), max(long_decimal) FROM test_aggregation_pushdown");
            this.assertAggregationPushedDown("SELECT sum(short_decimal), sum(long_decimal) FROM test_aggregation_pushdown");
            this.assertAggregationPushedDown("SELECT avg(short_decimal), avg(long_decimal) FROM test_aggregation_pushdown");
        }
    }

    @Test
    public void testColumnComment() throws Exception {
        try (AutoCloseable ignore = this.withTable("tpch.test_column_comment", "(col1 bigint, col2 bigint, col3 bigint)");){
            this.execute("COMMENT ON COLUMN tpch.test_column_comment.col1 IS 'test comment'");
            this.execute("COMMENT ON COLUMN tpch.test_column_comment.col2 IS ''");
            this.assertQuery("SELECT column_name, comment FROM information_schema.columns WHERE table_schema = 'tpch' AND table_name = 'test_column_comment'", "VALUES ('col1', 'test comment'), ('col2', null), ('col3', null)");
        }
    }

    private AutoCloseable withSchema(String schema) throws Exception {
        this.execute(String.format("CREATE SCHEMA %s", schema));
        return () -> {
            try {
                this.execute(String.format("DROP SCHEMA %s", schema));
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
        };
    }

    private AutoCloseable withTable(String tableName, String tableDefinition) throws Exception {
        this.execute(String.format("CREATE TABLE %s%s", tableName, tableDefinition));
        return () -> {
            try {
                this.execute(String.format("DROP TABLE %s", tableName));
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
        };
    }

    private void execute(String sql) throws SQLException {
        try (Connection connection = DriverManager.getConnection(this.postgreSqlServer.getJdbcUrl());
             Statement statement = connection.createStatement();){
            statement.execute(sql);
        }
    }
}

