package io.quarkus.test.utils;

import static org.junit.jupiter.api.Assertions.fail;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;

public final class PropertiesUtils {

    private static final String AUTOGENERATED_COMMENT = "This properties file has been auto generated "
            + "by Quarkus QE Test Framework";
    private static final String PROPERTY_START_TAG = "${";
    private static final String PROPERTY_END_TAG = "}";
    private static final String PROPERTY_WITH_OPTIONAL = ":";

    private PropertiesUtils() {

    }

    /**
     * Try to resolve the value property from the value if the content is contained between ${xxx}.
     *
     * @param value
     * @return
     */
    public static String resolveProperty(String value) {
        if (StringUtils.startsWith(value, PROPERTY_START_TAG)) {
            String propertyKey = StringUtils.substringBetween(value, PROPERTY_START_TAG, PROPERTY_END_TAG);
            String defaultValue = StringUtils.EMPTY;
            if (StringUtils.contains(propertyKey, PROPERTY_WITH_OPTIONAL)) {
                String[] propertyKeySplit = propertyKey.split(PROPERTY_WITH_OPTIONAL);
                propertyKey = propertyKeySplit[0];
                defaultValue = propertyKeySplit[1];
            }

            return System.getProperty(propertyKey, defaultValue);
        }

        return value;
    }

    public static Map<String, String> toMap(String propertiesFile) {
        try (InputStream in = ClassLoader.getSystemResourceAsStream(propertiesFile)) {
            return toMap(in);
        } catch (IOException e) {
            fail("Could not load map from system resource. Caused by " + e);
        }

        return null;
    }

    public static Map<String, String> toMap(Path path) {
        try (InputStream in = Files.newInputStream(path)) {
            return toMap(in);
        } catch (IOException e) {
            fail("Could not load map from path. Caused by " + e);
        }

        return null;
    }

    public static Map<String, String> toMap(InputStream is) {
        Properties properties = new Properties();
        try {
            properties.load(is);
        } catch (IOException e) {
            fail("Could not load map. Caused by " + e);
        }

        return (Map) properties;
    }

    public static void fromMap(Map<String, String> map, Path target) {

        try {
            Properties properties = new Properties();
            properties.putAll(map);
            properties.store(new FileOutputStream(target.toAbsolutePath().toFile()), AUTOGENERATED_COMMENT);
        } catch (Exception e) {
            fail("Could not create file in target. Caused by " + e);
        }
    }

}
