/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.tls.cli.letsencrypt;

import io.quarkus.tls.cli.letsencrypt.AcmeClient;
import io.smallrye.certs.CertificateUtils;
import io.vertx.core.json.JsonObject;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;
import org.wildfly.security.x500.cert.X509CertificateChainAndSigningKey;
import org.wildfly.security.x500.cert.acme.AcmeAccount;
import org.wildfly.security.x500.cert.acme.AcmeException;

public class LetsEncryptHelpers {
    static System.Logger LOGGER = System.getLogger("lets-encrypt");

    public static void writePrivateKeyAndCertificateChainsAsPem(PrivateKey pk, X509Certificate[] chain, File privateKeyFile, File certificateChainFile) throws Exception {
        if (pk == null) {
            throw new IllegalArgumentException("The private key cannot be null");
        }
        if (chain == null || chain.length == 0) {
            throw new IllegalArgumentException("The certificate chain cannot be null or empty");
        }
        CertificateUtils.writePrivateKeyToPem(pk, privateKeyFile);
        if (chain.length == 1) {
            CertificateUtils.writeCertificateToPEM(chain[0], certificateChainFile, new X509Certificate[0]);
            return;
        }
        X509Certificate[] restOfTheChain = new X509Certificate[chain.length - 1];
        System.arraycopy(chain, 1, restOfTheChain, 0, chain.length - 1);
        CertificateUtils.writeCertificateToPEM(chain[0], certificateChainFile, restOfTheChain);
    }

    public static X509Certificate loadCertificateFromPEM(String pemFilePath) throws IOException, CertificateException {
        try (PemReader pemReader = new PemReader(new FileReader(pemFilePath));){
            PemObject pemObject = pemReader.readPemObject();
            if (pemObject == null) {
                throw new IOException("Invalid PEM file: No PEM content found.");
            }
            byte[] content = pemObject.getContent();
            CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
            X509Certificate x509Certificate = (X509Certificate)certificateFactory.generateCertificate(new ByteArrayInputStream(content));
            return x509Certificate;
        }
    }

    public static String createAccount(AcmeClient acmeClient, String letsEncryptPath, boolean staging, String contactEmail) {
        LOGGER.log(System.Logger.Level.INFO, "\ud83d\udd35 Creating {0} Let's Encrypt account", staging ? "staging" : "production");
        AcmeAccount acmeAccount = AcmeAccount.builder().setTermsOfServiceAgreed(true).setServerUrl("https://acme-v02.api.letsencrypt.org/directory").setStagingServerUrl("https://acme-staging-v02.api.letsencrypt.org/directory").setContactUrls(new String[]{"mailto:" + contactEmail}).build();
        try {
            if (!acmeClient.createAccount(acmeAccount, staging)) {
                LOGGER.log(System.Logger.Level.INFO, "\ud83d\udd35 {0} Let's Encrypt account {1} already exists", staging ? "Staging" : "Production", contactEmail);
            } else {
                LOGGER.log(System.Logger.Level.INFO, "\ud83d\udd35 {0} Let's Encrypt account {1} has been created", staging ? "Staging" : "Production", contactEmail);
            }
        }
        catch (AcmeException ex) {
            LOGGER.log(System.Logger.Level.ERROR, "\u26a0\ufe0f Failed to create Let's Encrypt account");
            throw new RuntimeException(ex);
        }
        JsonObject accountJson = LetsEncryptHelpers.convertAccountToJson(acmeAccount);
        LetsEncryptHelpers.saveAccount(letsEncryptPath, accountJson);
        return accountJson.encode();
    }

    private static JsonObject convertAccountToJson(AcmeAccount acmeAccount) {
        JsonObject json = new JsonObject();
        json.put("account-url", acmeAccount.getAccountUrl());
        json.put("contact-url", acmeAccount.getContactUrls()[0]);
        if (acmeAccount.getPrivateKey() != null) {
            json.put("private-key", new String(Base64.getEncoder().encode(acmeAccount.getPrivateKey().getEncoded()), StandardCharsets.US_ASCII));
        }
        if (acmeAccount.getCertificate() != null) {
            try {
                json.put("certificate", new String(Base64.getEncoder().encode(acmeAccount.getCertificate().getEncoded()), StandardCharsets.US_ASCII));
            }
            catch (CertificateEncodingException ex) {
                LOGGER.log(System.Logger.Level.INFO, "\u26a0\ufe0f Failed to get encoded certificate data");
                throw new RuntimeException(ex);
            }
        }
        if (acmeAccount.getKeyAlgorithmName() != null) {
            json.put("key-algorithm", acmeAccount.getKeyAlgorithmName());
        }
        json.put("key-size", acmeAccount.getKeySize());
        return json;
    }

    private static void saveAccount(String letsEncryptPath, JsonObject accountJson) {
        LOGGER.log(System.Logger.Level.DEBUG, "Saving account to {0}", letsEncryptPath);
        Path accountPath = Paths.get(letsEncryptPath + "/account.json", new String[0]);
        try {
            Files.copy(new ByteArrayInputStream(accountJson.encode().getBytes(StandardCharsets.US_ASCII)), accountPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException ex) {
            throw new RuntimeException("Failure to save the account", ex);
        }
    }

    public static void issueCertificate(AcmeClient acmeClient, File letsEncryptPath, boolean staging, String domain, File certChainPemLoc, File privateKeyPemLoc) {
        X509CertificateChainAndSigningKey certChainAndPrivateKey;
        AcmeAccount acmeAccount = LetsEncryptHelpers.getAccount(letsEncryptPath);
        try {
            certChainAndPrivateKey = acmeClient.obtainCertificateChain(acmeAccount, staging, domain);
        }
        catch (AcmeException t) {
            throw new RuntimeException(t.getMessage());
        }
        LOGGER.log(System.Logger.Level.INFO, "\ud83d\udd35 Certificate and private key issued, converting them to PEM files");
        try {
            LetsEncryptHelpers.writePrivateKeyAndCertificateChainsAsPem(certChainAndPrivateKey.getSigningKey(), certChainAndPrivateKey.getCertificateChain(), privateKeyPemLoc, certChainPemLoc);
        }
        catch (Exception ex) {
            throw new RuntimeException("Failure to copy certificate pem");
        }
    }

    private static AcmeAccount getAccount(File letsEncryptPath) {
        LOGGER.log(System.Logger.Level.DEBUG, "Getting account from {0}", letsEncryptPath);
        JsonObject json = LetsEncryptHelpers.readAccountJson(letsEncryptPath);
        AcmeAccount.Builder builder = AcmeAccount.builder().setTermsOfServiceAgreed(true).setServerUrl("https://acme-v02.api.letsencrypt.org/directory").setStagingServerUrl("https://acme-staging-v02.api.letsencrypt.org/directory");
        String keyAlgorithm = json.getString("key-algorithm");
        builder.setKeyAlgorithmName(keyAlgorithm);
        builder.setKeySize(json.getInteger("key-size"));
        if (json.containsKey("private-key") && json.containsKey("certificate")) {
            PrivateKey privateKey = LetsEncryptHelpers.getPrivateKey(json.getString("private-key"), keyAlgorithm);
            X509Certificate certificate = LetsEncryptHelpers.getCertificate(json.getString("certificate"));
            builder.setKey(certificate, privateKey);
        }
        AcmeAccount acmeAccount = builder.build();
        acmeAccount.setContactUrls(new String[]{json.getString("contact-url")});
        acmeAccount.setAccountUrl(json.getString("account-url"));
        return acmeAccount;
    }

    private static JsonObject readAccountJson(File letsEncryptPath) {
        JsonObject jsonObject;
        LOGGER.log(System.Logger.Level.DEBUG, "Reading account information from {0}", letsEncryptPath);
        Path accountPath = Paths.get(letsEncryptPath + "/account.json", new String[0]);
        FileInputStream fis = new FileInputStream(accountPath.toString());
        try {
            jsonObject = new JsonObject(new String(fis.readAllBytes(), StandardCharsets.US_ASCII));
        }
        catch (Throwable throwable) {
            try {
                try {
                    fis.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to read the account file, you must create account first");
            }
        }
        fis.close();
        return jsonObject;
    }

    private static X509Certificate getCertificate(String encodedCert) {
        try {
            byte[] encodedBytes = Base64.getDecoder().decode(encodedCert);
            return (X509Certificate)CertificateFactory.getInstance("X.509").generateCertificate(new ByteArrayInputStream(encodedBytes));
        }
        catch (Exception ex) {
            throw new RuntimeException("Failure to create a certificate", ex);
        }
    }

    private static PrivateKey getPrivateKey(String encodedKey, String keyAlgorithm) {
        try {
            KeyFactory f = KeyFactory.getInstance(keyAlgorithm == null || "RSA".equals(keyAlgorithm) ? "RSA" : "EC");
            byte[] encodedBytes = Base64.getDecoder().decode(encodedKey);
            PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(encodedBytes);
            return f.generatePrivate(spec);
        }
        catch (Exception ex) {
            throw new RuntimeException("Failure to create a private key", ex);
        }
    }

    public static void renewCertificate(AcmeClient acmeClient, File letsEncryptPath, boolean staging, String domain, File certChainPemLoc, File privateKeyPemLoc) {
        LOGGER.log(System.Logger.Level.INFO, "\ud83d\udd35 Renewing {0} Let's Encrypt certificate chain and private key", staging ? "staging" : "production");
        LetsEncryptHelpers.issueCertificate(acmeClient, letsEncryptPath, staging, domain, certChainPemLoc, privateKeyPemLoc);
    }

    public static void deactivateAccount(AcmeClient acmeClient, File letsEncryptPath, boolean staging) throws IOException {
        AcmeAccount acmeAccount = LetsEncryptHelpers.getAccount(letsEncryptPath);
        LOGGER.log(System.Logger.Level.INFO, "Deactivating {0} Let's Encrypt account", staging ? "staging" : "production");
        acmeClient.deactivateAccount(acmeAccount, staging);
        LOGGER.log(System.Logger.Level.INFO, "Removing account file from {0}", letsEncryptPath);
        Path accountPath = Paths.get(letsEncryptPath + "/account.json", new String[0]);
        Files.deleteIfExists(accountPath);
    }

    public static void adjustPermissions(File certFile, File keyFile) {
        if (!certFile.setReadable(true, false)) {
            LOGGER.log(System.Logger.Level.ERROR, "Failed to set certificate file readable");
        }
        if (!certFile.setWritable(true, true)) {
            LOGGER.log(System.Logger.Level.ERROR, "Failed to set certificate file as not writable");
        }
        if (!keyFile.setReadable(true, false)) {
            LOGGER.log(System.Logger.Level.ERROR, "Failed to set key file as readable");
        }
        if (!keyFile.setWritable(true, true)) {
            LOGGER.log(System.Logger.Level.ERROR, "Failed to set key file as not writable");
        }
    }
}

