/*
 * Decompiled with CFR 0.152.
 */
package io.rapidw.mqtt.codec;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.EncoderException;
import io.netty.handler.codec.MessageToMessageEncoder;
import io.rapidw.mqtt.codec.MqttConnAckPacket;
import io.rapidw.mqtt.codec.MqttConnectPacket;
import io.rapidw.mqtt.codec.MqttConnectReturnCode;
import io.rapidw.mqtt.codec.MqttDisconnectPacket;
import io.rapidw.mqtt.codec.MqttPacket;
import io.rapidw.mqtt.codec.MqttPingReqPacket;
import io.rapidw.mqtt.codec.MqttPingRespPacket;
import io.rapidw.mqtt.codec.MqttPublishPacket;
import io.rapidw.mqtt.codec.MqttQosLevel;
import io.rapidw.mqtt.codec.MqttSubAckPacket;
import io.rapidw.mqtt.codec.MqttSubscribePacket;
import io.rapidw.mqtt.codec.MqttTopicAndQosLevel;
import io.rapidw.mqtt.codec.MqttUnsubAckPacket;
import io.rapidw.mqtt.codec.MqttUnsubscribePacket;
import io.rapidw.mqtt.codec.MqttWill;
import io.rapidw.mqtt.codec.ValidationUtils;
import java.util.LinkedList;
import java.util.List;
import lombok.Generated;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ChannelHandler.Sharable
public class MqttEncoder
extends MessageToMessageEncoder<MqttPacket> {
    @SuppressFBWarnings(justification="generated code")
    @Generated
    private static final Logger log = LoggerFactory.getLogger(MqttEncoder.class);
    public static final MqttEncoder INSTANCE = new MqttEncoder();
    private static byte[] PROTOCOL_NAME_BYTES = ValidationUtils.validateAndEncodeString("MQTT", "protocol name");

    private MqttEncoder() {
    }

    protected void encode(ChannelHandlerContext ctx, MqttPacket packet, List<Object> out) {
        out.add(MqttEncoder.doEncode(ctx.alloc(), packet));
    }

    static ByteBuf doEncode(ByteBufAllocator byteBufAllocator, MqttPacket packet) {
        switch (packet.getType()) {
            case CONNECT: {
                return MqttEncoder.encodeConnect(byteBufAllocator, (MqttConnectPacket)packet);
            }
            case CONNACK: {
                return MqttEncoder.encodeConnAck(byteBufAllocator, (MqttConnAckPacket)packet);
            }
            case PUBLISH: {
                return MqttEncoder.encodePublish(byteBufAllocator, (MqttPublishPacket)packet);
            }
            case SUBSCRIBE: {
                return MqttEncoder.encodeSubscribe(byteBufAllocator, (MqttSubscribePacket)packet);
            }
            case SUBACK: {
                return MqttEncoder.encodeSubAck(byteBufAllocator, (MqttSubAckPacket)packet);
            }
            case UNSUBSCRIBE: {
                return MqttEncoder.encodeUnsubscribe(byteBufAllocator, (MqttUnsubscribePacket)packet);
            }
            case UNSUBACK: {
                return MqttEncoder.encodeUnsubAck(byteBufAllocator, (MqttUnsubAckPacket)packet);
            }
            case PINGREQ: {
                return MqttEncoder.encodePingReq(byteBufAllocator, (MqttPingReqPacket)packet);
            }
            case PINGRESP: {
                return MqttEncoder.encodePingResp(byteBufAllocator, (MqttPingRespPacket)packet);
            }
            case DISCONNECT: {
                return MqttEncoder.encodeDisconnect(byteBufAllocator, (MqttDisconnectPacket)packet);
            }
        }
        throw new EncoderException("Unknown message type");
    }

    private static ByteBuf encodeConnect(ByteBufAllocator byteBufAllocator, @NonNull MqttConnectPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        int variableHeaderSize = 2 + PROTOCOL_NAME_BYTES.length + 1 + 1 + 2;
        byte[] clientIdBytes = ValidationUtils.validateAndEncodeString(packet.getClientId(), "client id");
        int clientIdSize = 2 + clientIdBytes.length;
        int willSize = 0;
        byte[] willTopicBytes = null;
        byte[] willMessage = null;
        MqttWill will = packet.getWill();
        if (will != null) {
            willTopicBytes = ValidationUtils.validateAndEncodeString(will.getTopic(), "will topic");
            willMessage = ValidationUtils.requireNonNull(will.getMessage(), "will message");
            willSize += 2 + willTopicBytes.length;
            willSize += 2 + willMessage.length;
        }
        int usernamePasswordSize = 0;
        byte[] usernameBytes = null;
        byte[] passwordBytes = null;
        String username = packet.getUsername();
        byte[] password = packet.getPassword();
        if (username != null) {
            usernameBytes = ValidationUtils.validateAndEncodeString(username, "username");
            usernamePasswordSize += 2 + usernameBytes.length;
            if (password != null) {
                passwordBytes = ValidationUtils.validateByteArray(packet.getPassword(), "password");
                usernamePasswordSize += 2 + passwordBytes.length;
            }
        } else {
            ValidationUtils.requireNull(packet.getPassword(), "password");
        }
        int variablePartSize = variableHeaderSize + clientIdSize + willSize + usernamePasswordSize;
        int fixedHeaderSize = 1 + MqttEncoder.getAndValidateVariablePartLengthSize(variableHeaderSize + variablePartSize);
        ByteBuf buf = byteBufAllocator.buffer(fixedHeaderSize + variablePartSize);
        buf.writeByte(16);
        MqttEncoder.writeVariablePartLength(buf, variablePartSize);
        MqttEncoder.writeStringBytes(buf, PROTOCOL_NAME_BYTES, "protocol name");
        buf.writeByte(4);
        int b = 0;
        if (packet.isCleanSession()) {
            b |= 2;
        }
        if (will != null) {
            b |= 4;
            b |= will.getQosLevel().ordinal() << 3;
            if (will.isRetain()) {
                b |= 0x20;
            }
        }
        if (username != null) {
            b |= 0x40;
        }
        if (password != null) {
            b |= 0x80;
        }
        buf.writeByte(b);
        if (packet.getKeepaliveSeconds() < 0 || packet.getKeepaliveSeconds() > 65535) {
            throw new EncoderException("invalid keepalive seconds");
        }
        buf.writeShort(packet.getKeepaliveSeconds());
        if (packet.getClientId().length() > 65535) {
            throw new EncoderException("invalid client id");
        }
        MqttEncoder.writeStringBytes(buf, clientIdBytes, "client id");
        if (will != null) {
            MqttEncoder.writeStringBytes(buf, willTopicBytes, "will topic");
            MqttEncoder.writeByteArray(buf, willMessage, "will message");
        }
        if (usernameBytes != null) {
            MqttEncoder.writeStringBytes(buf, usernameBytes, "username");
        }
        if (passwordBytes != null) {
            MqttEncoder.writeByteArray(buf, passwordBytes, "password");
        }
        return buf;
    }

    private static ByteBuf encodePingReq(ByteBufAllocator byteBufAllocator, @NonNull MqttPingReqPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        ByteBuf buf = byteBufAllocator.buffer(2);
        buf.writeByte(192);
        buf.writeByte(0);
        return buf;
    }

    private static ByteBuf encodeSubscribe(ByteBufAllocator byteBufAllocator, @NonNull MqttSubscribePacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        LinkedList<byte[]> topicFilterBytesList = new LinkedList<byte[]>();
        int variablePartSize = 2;
        int payloadCount = 0;
        for (MqttTopicAndQosLevel payload : ValidationUtils.requireNonNull(packet.getMqttTopicAndQosLevels(), "topic and qos list")) {
            byte[] topicFilterBytes = ValidationUtils.validateAndEncodeTopicFilter(payload.getTopicFilter());
            if (payload.getQosLevel() == MqttQosLevel.FAILURE) {
                throw new EncoderException("[MQTT-3.3.1-4] PUBLISH Packet MUST NOT have both QoS bits set to 1");
            }
            topicFilterBytesList.add(topicFilterBytes);
            variablePartSize += 2 + topicFilterBytes.length + 1;
            ++payloadCount;
        }
        if (payloadCount == 0) {
            throw new EncoderException("[MQTT-3.8.3-3] The payload of a SUBSCRIBE packet MUST contain at least one Topic Filter / QoS pair");
        }
        int fixedHeaderSize = 1 + MqttEncoder.getAndValidateVariablePartLengthSize(variablePartSize);
        ByteBuf buf = byteBufAllocator.buffer(fixedHeaderSize + variablePartSize);
        buf.writeByte(130);
        MqttEncoder.writeVariablePartLength(buf, variablePartSize);
        buf.writeShort(ValidationUtils.validatePacketId(packet.getPacketId()));
        for (int i = 0; i < packet.getMqttTopicAndQosLevels().size(); ++i) {
            MqttTopicAndQosLevel payload = packet.getMqttTopicAndQosLevels().get(i);
            byte[] topicFilterBytes = (byte[])topicFilterBytesList.get(i);
            MqttEncoder.writeStringBytes(buf, topicFilterBytes, "topic filter");
            buf.writeByte(payload.getQosLevel().ordinal());
        }
        return buf;
    }

    private static ByteBuf encodePublish(ByteBufAllocator byteBufAllocator, @NonNull MqttPublishPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        byte[] topicBytes = ValidationUtils.validateAndEncodeTopicName(packet.getTopic());
        ValidationUtils.requireNonNull(packet.getPayload(), "payload");
        int variablePartSize = 2 + topicBytes.length + packet.getPayload().length;
        if (packet.getQosLevel() == MqttQosLevel.AT_LEAST_ONCE || packet.getQosLevel() == MqttQosLevel.EXACTLY_ONCE) {
            variablePartSize += 2;
        }
        int fixedHeaderSize = 1 + MqttEncoder.getAndValidateVariablePartLengthSize(variablePartSize);
        ByteBuf buf = byteBufAllocator.buffer(fixedHeaderSize + variablePartSize);
        int b = 0;
        b |= 0x30;
        if (packet.isRetain()) {
            b |= 1;
        }
        b |= packet.getQosLevel().ordinal() << 1;
        if (packet.isDupFlag()) {
            b |= 8;
        }
        buf.writeByte(b);
        MqttEncoder.writeVariablePartLength(buf, variablePartSize);
        MqttEncoder.writeStringBytes(buf, topicBytes, "topic name");
        if (packet.getQosLevel() == MqttQosLevel.AT_LEAST_ONCE || packet.getQosLevel() == MqttQosLevel.EXACTLY_ONCE) {
            buf.writeShort(ValidationUtils.validatePacketId(packet.getPacketId()));
        }
        log.debug("payload: {}", (Object)packet.getPayload());
        buf.writeBytes(packet.getPayload());
        return buf;
    }

    private static ByteBuf encodeDisconnect(ByteBufAllocator byteBufAllocator, @NonNull MqttDisconnectPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        ByteBuf buf = byteBufAllocator.buffer(2);
        buf.writeByte(224);
        buf.writeByte(0);
        return buf;
    }

    private static ByteBuf encodeConnAck(ByteBufAllocator byteBufAllocator, @NonNull MqttConnAckPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        ByteBuf buf = byteBufAllocator.buffer(4);
        buf.writeByte(32);
        buf.writeByte(2);
        if (packet.getConnectReturnCode() != MqttConnectReturnCode.CONNECTION_ACCEPTED && packet.isSessionPresent()) {
            throw new EncoderException("[MQTT-3.2.2-4] CONNACK packet containing a non-zero return code it MUST set Session Present to 0");
        }
        if (packet.isSessionPresent()) {
            buf.writeByte(1);
        } else {
            buf.writeByte(0);
        }
        buf.writeByte((int)packet.getConnectReturnCode().byteValue());
        return buf;
    }

    private static ByteBuf encodeSubAck(ByteBufAllocator byteBufAllocator, @NonNull MqttSubAckPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        int variablePartSize = 2 + packet.getQosLevels().size();
        int fixedHeaderSize = 1 + MqttEncoder.getAndValidateVariablePartLengthSize(variablePartSize);
        ByteBuf buf = byteBufAllocator.buffer(fixedHeaderSize + variablePartSize);
        buf.writeByte(144);
        MqttEncoder.writeVariablePartLength(buf, variablePartSize);
        buf.writeShort(packet.getPacketId());
        for (MqttQosLevel qos : packet.getQosLevels()) {
            buf.writeByte(qos.ordinal());
        }
        return buf;
    }

    private static ByteBuf encodeUnsubscribe(ByteBufAllocator byteBufAllocator, @NonNull MqttUnsubscribePacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        int variablePartSize = 2;
        LinkedList<byte[]> topicFilterBytes = new LinkedList<byte[]>();
        for (String topicFilter : packet.getTopicFilters()) {
            byte[] bytes = ValidationUtils.validateAndEncodeString(topicFilter, "topic filter");
            variablePartSize += 2 + bytes.length;
            topicFilterBytes.add(bytes);
        }
        int fixedHeaderSize = 1 + MqttEncoder.getAndValidateVariablePartLengthSize(variablePartSize);
        ByteBuf buf = byteBufAllocator.buffer(fixedHeaderSize + variablePartSize);
        buf.writeByte(162);
        MqttEncoder.writeVariablePartLength(buf, variablePartSize);
        buf.writeShort(packet.getPacketId());
        for (byte[] bytes : topicFilterBytes) {
            MqttEncoder.writeStringBytes(buf, bytes, "topic filter");
        }
        return buf;
    }

    private static ByteBuf encodeUnsubAck(ByteBufAllocator byteBufAllocator, @NonNull MqttUnsubAckPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        ByteBuf buf = byteBufAllocator.buffer(4);
        buf.writeByte(176);
        buf.writeByte(2);
        buf.writeShort(packet.getPacketId());
        return buf;
    }

    private static ByteBuf encodePingResp(ByteBufAllocator byteBufAllocator, @NonNull MqttPingRespPacket packet) {
        if (packet == null) {
            throw new NullPointerException("packet is marked non-null but is null");
        }
        ByteBuf buf = byteBufAllocator.buffer(2);
        buf.writeByte(208);
        buf.writeByte(0);
        return buf;
    }

    private static void writeVariablePartLength(ByteBuf buf, int num) {
        do {
            int digit = num % 128;
            if ((num /= 128) > 0) {
                digit |= 0x80;
            }
            buf.writeByte(digit);
        } while (num > 0);
    }

    private static void writeStringBytes(ByteBuf buf, byte[] stringBytes, @NonNull String name) {
        if (name == null) {
            throw new NullPointerException("name is marked non-null but is null");
        }
        ValidationUtils.requireNonNull(stringBytes, name);
        buf.writeShort(stringBytes.length);
        buf.writeBytes(stringBytes);
    }

    private static void writeByteArray(ByteBuf buf, byte[] bytes, @NonNull String name) {
        if (name == null) {
            throw new NullPointerException("name is marked non-null but is null");
        }
        ValidationUtils.requireNonNull(bytes, name);
        buf.writeShort(bytes.length);
        buf.writeBytes(bytes);
    }

    private static int getAndValidateVariablePartLengthSize(int num) {
        int count = 0;
        do {
            ++count;
        } while ((num /= 128) > 0);
        if (count > 4) {
            throw new IllegalArgumentException("packet too large");
        }
        return count;
    }
}

