package io.shiftleft.console.workspacehandling

import java.nio.file.Path

import better.files.File
import better.files.Dsl._
import io.shiftleft.codepropertygraph.Cpg
import io.shiftleft.semanticcpg.Overlays

object Project {
  val workCpgFileName = "cpg.bin.tmp"
  val persistentCpgFileName = "cpg.bin"
}

case class ProjectFile(inputPath: String, name: String)

/**
  * @param path absolute path to directory holding the project
  * @param cpg reference to loaded CPG or None, if the CPG is not loaded
  * */
case class Project(projectFile: ProjectFile, var path: Path, var cpg: Option[Cpg] = None) {

  import Project._

  def name: String = projectFile.name

  def inputPath: String = projectFile.inputPath

  def appliedOverlays: List[String] = {
    cpg.map(Overlays.appliedOverlays).getOrElse(List())
  }

  def availableOverlays: List[String] = {
    File(path.resolve("overlays")).list.map(_.name).toList
  }

  def overlayDirs: List[File] = {
    val overlayDir = File(path.resolve("overlays"))
    appliedOverlays.map(o => overlayDir / o)
  }

  override def toString: String =
    toTableRow.mkString("\t")

  def toTableRow: List[String] = {
    val cpgLoaded = cpg.isDefined
    val overlays = availableOverlays.mkString(",")
    val inputPath = projectFile.inputPath
    List(name, overlays, inputPath, cpgLoaded.toString)
  }

  /**
    * Close project if it is open and do nothing otherwise.
    * */
  def close: Project = {
    cpg.foreach { c =>
      c.close()
      System.err.println("Turning working copy into new persistent CPG")
      val workingCopy = path.resolve(workCpgFileName)
      val persistent = path.resolve(persistentCpgFileName)
      cp(workingCopy, persistent)
    }
    cpg = None
    this
  }

}
