/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.tinkergraph.structure;

import com.sun.management.GarbageCollectionNotificationInfo;
import java.io.IOException;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryUsage;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.ListenerNotFoundException;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;
import org.apache.tinkerpop.gremlin.tinkergraph.structure.ElementRef;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReferenceManager {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private Map<NotificationEmitter, NotificationListener> gcNotificationListeners = new HashMap<NotificationEmitter, NotificationListener>(2);
    private final float heapUsageThreshold;
    public final int backpressureMillis = 500;
    public final int releaseCount = 100000;
    private int totalReleaseCount;
    private final int cpuCount = Runtime.getRuntime().availableProcessors();
    private final ExecutorService executorService = Executors.newFixedThreadPool(this.cpuCount);
    private final AtomicInteger clearingProcessCount = new AtomicInteger(0);
    private final Comparator<ElementRef> refComparator = (ref1, ref2) -> {
        if (ref1.getSerializationCount() != ref2.getSerializationCount()) {
            return ref1.getSerializationCount() < ref2.getSerializationCount() ? -1 : 1;
        }
        return ref1.getLastDeserializedTime() < ref2.getLastDeserializedTime() ? -1 : 1;
    };
    private final PriorityBlockingQueue<ElementRef> clearableRefs = new PriorityBlockingQueue<ElementRef>(100000, this.refComparator);

    public ReferenceManager(int heapPercentageThreshold) {
        if (heapPercentageThreshold < 0 || heapPercentageThreshold > 100) {
            throw new IllegalArgumentException("heapPercentageThreshold must be between 0 and 100, but is " + heapPercentageThreshold);
        }
        this.heapUsageThreshold = (float)heapPercentageThreshold / 100.0f;
        this.installGCMonitoring();
    }

    public void registerRef(ElementRef ref) {
        this.clearableRefs.add(ref);
    }

    public void applyBackpressureMaybe() {
        if (this.clearingProcessCount.get() > 0) {
            try {
                this.logger.trace("applying 500ms backpressure");
                Thread.sleep(500L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    protected void maybeClearReferences(float heapUsage) {
        if (heapUsage > this.heapUsageThreshold) {
            if (this.clearingProcessCount.get() > 0) {
                this.logger.debug("cleaning in progress, will only queue up more references to clear after that's completed");
            } else if (this.clearableRefs.isEmpty()) {
                this.logger.debug("clearableRefs queue is empty - nothing to clear at the moment");
            } else {
                int releaseCount = Integer.min(this.releaseCount, this.clearableRefs.size());
                this.logger.info("heap usage (after GC) was " + heapUsage + " -> scheduled to clear " + releaseCount + " references (asynchronously)");
                this.asynchronouslyClearReferences(releaseCount);
            }
        }
    }

    protected void asynchronouslyClearReferences(int releaseCount) {
        for (int i = 0; i < this.cpuCount; ++i) {
            int releaseCountPerThread = releaseCount / this.cpuCount;
            this.executorService.submit(() -> {
                int actualReleaseCount = this.safelyClearReferences(releaseCountPerThread);
                this.totalReleaseCount += actualReleaseCount;
                this.logger.info("completed clearing of " + actualReleaseCount + " references");
                this.logger.debug("current clearable queue size: " + this.clearableRefs.size());
                this.logger.debug("references cleared in total: " + this.totalReleaseCount);
            });
        }
    }

    protected int safelyClearReferences(int releaseCount) {
        try {
            this.clearingProcessCount.incrementAndGet();
            int n = this.clearReferences(releaseCount);
            return n;
        }
        catch (Exception e) {
            this.logger.error("error while trying to clear " + releaseCount + " references", (Throwable)e);
        }
        finally {
            this.clearingProcessCount.decrementAndGet();
        }
        return 0;
    }

    protected int clearReferences(int releaseCount) throws IOException {
        this.logger.info("attempting to clear " + releaseCount + " references");
        ElementRef ref = this.clearableRefs.poll();
        int actualReleaseCount = 0;
        while (ref != null && releaseCount > 0) {
            if (ref.isSet()) {
                ref.clear();
                --releaseCount;
                ++actualReleaseCount;
            }
            ref = this.clearableRefs.poll();
        }
        return actualReleaseCount;
    }

    protected void installGCMonitoring() {
        List<GarbageCollectorMXBean> gcbeans = ManagementFactory.getGarbageCollectorMXBeans();
        for (GarbageCollectorMXBean gcbean : gcbeans) {
            NotificationListener listener = this.createNotificationListener();
            NotificationEmitter emitter = (NotificationEmitter)((Object)gcbean);
            emitter.addNotificationListener(listener, null, null);
            this.gcNotificationListeners.put(emitter, listener);
        }
        int heapUsageThresholdPercent = (int)Math.floor(this.heapUsageThreshold * 100.0f);
        this.logger.info("installed GC monitors. will clear references if heap (after GC) is larger than " + heapUsageThresholdPercent + "%");
    }

    private NotificationListener createNotificationListener() {
        HashSet<String> ignoredMemoryAreas = new HashSet<String>(Arrays.asList("Code Cache", "Compressed Class Space", "Metaspace"));
        return (notification, handback) -> {
            if (notification.getType().equals("com.sun.management.gc.notification")) {
                GarbageCollectionNotificationInfo info = GarbageCollectionNotificationInfo.from((CompositeData)notification.getUserData());
                long totalMemUsed = 0L;
                long totalMemMax = 0L;
                for (Map.Entry<String, MemoryUsage> entry : info.getGcInfo().getMemoryUsageAfterGc().entrySet()) {
                    String name = entry.getKey();
                    if (ignoredMemoryAreas.contains(name)) continue;
                    MemoryUsage detail = entry.getValue();
                    totalMemUsed += detail.getUsed();
                    totalMemMax += detail.getMax();
                }
                float heapUsage = (float)totalMemUsed / (float)totalMemMax;
                int heapUsagePercent = (int)Math.floor(heapUsage * 100.0f);
                this.logger.debug("heap usage after GC: " + heapUsagePercent + "%");
                this.maybeClearReferences(heapUsage);
            }
        };
    }

    protected void uninstallGCMonitoring() {
        for (Map.Entry<NotificationEmitter, NotificationListener> entry : this.gcNotificationListeners.entrySet()) {
            try {
                entry.getKey().removeNotificationListener(entry.getValue());
            }
            catch (ListenerNotFoundException e) {
                throw new RuntimeException("unable to remove GC monitor", e);
            }
        }
        this.logger.info("uninstalled GC monitors.");
    }

    public void close() {
        this.uninstallGCMonitoring();
        this.executorService.shutdown();
    }
}

