package io.snice.codecs.codec.gtp;

import io.snice.buffer.Buffer;
import io.snice.codecs.codec.gtp.gtpc.InfoElement;
import io.snice.codecs.codec.gtp.gtpc.v1.Gtp1Message;
import io.snice.codecs.codec.gtp.gtpc.v1.Gtp1Request;
import io.snice.codecs.codec.gtp.gtpc.v2.*;
import io.snice.codecs.codec.gtp.gtpc.v2.messages.tunnel.CreateSessionRequest;
import io.snice.codecs.codec.gtp.gtpc.v2.tliv.Imsi;

import java.util.List;
import java.util.Optional;

import static io.snice.preconditions.PreConditions.assertNotNull;

/**
 *
 */
public interface GtpMessage {

    static GtpMessage frame(final Buffer buffer) throws GtpParseException, IllegalArgumentException {
        assertNotNull(buffer, "The buffer cannot be null");
        if (GtpHeader.getGtpVersion(buffer )== 1) {
            return Gtp1Message.frame(buffer);
        }

        return Gtp2Message.frame(buffer);
    }

    /**
     * Get the entire {@link GtpMessage} as a {@link Buffer}, which you then can use
     * to e.g. write to network socket.
     */
    default Buffer getBuffer() {
        throw new RuntimeException("Not yet implemented");
    }

    default boolean isRequest() {
        return false;
    }

    default boolean isResponse() {
        return false;
    }

    default boolean isGtpVersion1() {
        return getVersion() == 1;
    }

    default boolean isGtpVersion2() {
        return getVersion() == 2;
    }

    default Optional<Imsi> getImsi() {
        return (Optional<Imsi>) toGtp2Message().getInfoElement(Gtp2InfoElement.IMSI);
    }

    default Gtp1Message toGtp1Message() {
        throw new ClassCastException("Unable to cast a " + getClass().getName() + " into a " + Gtp1Message.class.getName());
    }

    default Gtp1Request toGtp1Request() {
        throw new ClassCastException("Unable to cast a " + getClass().getName() + " into a " + Gtp1Request.class.getName());
    }

    default Gtp2Message toGtp2Message() {
        throw new ClassCastException("Unable to cast a " + getClass().getName() + " into a " + Gtp2Message.class.getName());
    }

    default Gtp2Request toGtp2Request() {
        throw new ClassCastException("Unable to cast a " + getClass().getName() + " into a " + Gtp2Request.class.getName());
    }

    default CreateSessionRequest toCreateSessionRequest() {
        throw new ClassCastException("Unable to cast a " + getClass().getName() + " into a " + CreateSessionRequest.class.getName());
    }

    default Gtp2Response toGtp2Response() {
        throw new ClassCastException("Unable to cast a " + getClass().getName() + " into a " + Gtp2Response.class.getName());
    }

    default boolean isEchoRequest() {
        return getMessageTypeDecimal() == Gtp2MessageType.ECHO_REQUEST.getType();
    }

    <T extends GtpHeader> T getHeader();

    List<? extends InfoElement> getInfoElements();

    default int getMessageTypeDecimal() {
        return getHeader().getMessageTypeDecimal();
    }

    default int getVersion() {
        return getHeader().getVersion();
    }

}
