package io.snice.codecs.codec.gtp.gtpc.v1.impl;

import io.snice.buffer.Buffer;
import io.snice.buffer.Buffers;
import io.snice.codecs.codec.gtp.Teid;
import io.snice.codecs.codec.gtp.gtpc.v1.Gtp1Header;
import io.snice.codecs.codec.gtp.gtpc.v1.Gtp1Message;
import io.snice.codecs.codec.gtp.gtpc.v1.Gtp1MessageType;

import java.util.Collections;

import static io.snice.buffer.Buffers.assertNotEmpty;
import static io.snice.preconditions.PreConditions.assertArgument;
import static io.snice.preconditions.PreConditions.assertNotNull;

public abstract class AbstractGtp1MessageBuilder<T extends Gtp1Message> implements Gtp1MessageBuilder<T> {

    private boolean hasBeenBuilt = false;
    private final int index = 0;
    private final Gtp1MessageType type;
    private Teid teid;
    private Buffer seqNo;
    private Buffer payload;

    protected AbstractGtp1MessageBuilder(final Gtp1MessageType type) {
        this.type = type;
    }

    @Override
    public Gtp1MessageBuilder<T> withTeid(final Buffer teid) {
        this.teid = Teid.of(teid);
        return this;
    }

    @Override
    public Gtp1MessageBuilder<T> withTeid(final Teid teid) {
        assertNotNull(teid, "The TEID cannot be null");
        this.teid = teid;
        return this;
    }

    @Override
    public Gtp1MessageBuilder<T> withSeqNo(final Buffer seqNo) {
        this.seqNo = seqNo;
        return this;
    }

    @Override
    public Gtp1MessageBuilder<T> withRandomSeqNo() {
        this.seqNo = Buffers.random(2);
        return this;
    }

    @Override
    public Gtp1MessageBuilder<T> withPayload(final Buffer buffer) {
        assertArgument(type == Gtp1MessageType.G_PDU, "You can only specify a payload for G-PDU messages");
        assertNotEmpty(buffer, "The payload cannot be empty");
        this.payload = buffer;
        return this;
    }

    @Override
    public final T build() {
        if (hasBeenBuilt) {
            throw new IllegalArgumentException("This GTP message has already been built once");
        }
        hasBeenBuilt = true;

        assertNotNull(teid, "The TEID is not optional. You must specify it");
        assertArgument(type == Gtp1MessageType.G_PDU, "Sorry, can only do G-PDU messages right now");
        assertNotEmpty(payload, "Since I can only do G-PDU messages right now, you also have to specify the payload");
        final var header = createHeader();
        final var buffer = Buffers.wrap(header.getBuffer(), payload);
        return internalBuild(type, buffer, header, payload);
    }

    private Gtp1Header createHeader() {
        final var header = Gtp1Header.of(type);
        header.withTeid(teid);

        if (seqNo != null) {
            header.withSequenceNumber(seqNo);
        }
        header.withBodySize(payload.capacity());
        return header.build();
    }

    protected abstract T internalBuild(final Gtp1MessageType type, final Buffer buffer, final Gtp1Header header,
                                       final Buffer payload);
}
