/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.data.task.jedis;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.netflix.spinnaker.clouddriver.core.ClouddriverHostname;
import com.netflix.spinnaker.clouddriver.data.task.DefaultTaskStatus;
import com.netflix.spinnaker.clouddriver.data.task.SagaId;
import com.netflix.spinnaker.clouddriver.data.task.Status;
import com.netflix.spinnaker.clouddriver.data.task.Task;
import com.netflix.spinnaker.clouddriver.data.task.TaskDisplayStatus;
import com.netflix.spinnaker.clouddriver.data.task.TaskRepository;
import com.netflix.spinnaker.clouddriver.data.task.TaskState;
import com.netflix.spinnaker.clouddriver.data.task.jedis.JedisTask;
import com.netflix.spinnaker.kork.exceptions.SystemException;
import com.netflix.spinnaker.kork.jedis.RedisClientDelegate;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.RetryPolicy;
import net.jodah.failsafe.SyncFailsafe;
import net.jodah.failsafe.function.CheckedConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.exceptions.JedisException;

public class RedisTaskRepository
implements TaskRepository {
    private static final Logger log = LoggerFactory.getLogger(RedisTaskRepository.class);
    private static final String RUNNING_TASK_KEY = "kato:tasks";
    private static final String TASK_KEY_MAP = "kato:taskmap";
    private static final TypeReference<Map<String, String>> HISTORY_TYPE = new TypeReference<Map<String, String>>(){};
    private static final TypeReference<Set<SagaId>> SAGA_IDS_TYPE = new TypeReference<Set<SagaId>>(){};
    private static final int TASK_TTL = (int)TimeUnit.HOURS.toSeconds(12L);
    private static final RetryPolicy REDIS_RETRY_POLICY = new RetryPolicy().retryOn(Collections.singletonList(JedisException.class)).withDelay(500L, TimeUnit.MILLISECONDS).withMaxRetries(3);
    private final RedisClientDelegate redisClientDelegate;
    private final Optional<RedisClientDelegate> redisClientDelegatePrevious;
    private final ObjectMapper mapper = new ObjectMapper();

    public RedisTaskRepository(RedisClientDelegate redisClientDelegate, Optional<RedisClientDelegate> redisClientDelegatePrevious) {
        this.redisClientDelegate = redisClientDelegate;
        this.redisClientDelegatePrevious = redisClientDelegatePrevious;
    }

    @Override
    public Task create(String phase, String status) {
        return this.create(phase, status, UUID.randomUUID().toString());
    }

    @Override
    public Task create(String phase, String status, String clientRequestId) {
        String taskKey = this.getClientRequestKey(clientRequestId);
        String taskId = UUID.randomUUID().toString();
        JedisTask task = new JedisTask(taskId, System.currentTimeMillis(), this, ClouddriverHostname.ID, clientRequestId, new HashSet<SagaId>(), false);
        this.addToHistory(DefaultTaskStatus.create(phase, status, TaskState.STARTED), task);
        this.set(taskId, task);
        Long newTask = this.retry(() -> (Long)this.redisClientDelegate.withCommandsClient(client -> client.setnx(taskKey, taskId)), "Registering task with index");
        if (newTask != 0L) {
            return task;
        }
        this.addToHistory(DefaultTaskStatus.create(phase, "Duplicate of " + clientRequestId, TaskState.FAILED), task);
        return this.getByClientRequestId(clientRequestId);
    }

    @Override
    public Task get(String id) {
        boolean oldTask;
        Map taskMap = this.retry(() -> (Map)this.redisClientDelegate.withCommandsClient(client -> client.hgetAll("task:" + id)), String.format("Getting task ID %s", id));
        boolean bl = oldTask = this.redisClientDelegatePrevious.isPresent() && (taskMap == null || taskMap.isEmpty());
        if (oldTask) {
            try {
                taskMap = (Map)this.redisClientDelegatePrevious.get().withCommandsClient(client -> client.hgetAll("task:" + id));
            }
            catch (Exception e) {
                return null;
            }
        }
        if (taskMap.containsKey("id") && taskMap.containsKey("startTimeMs")) {
            HashSet<SagaId> sagaIds;
            if (taskMap.containsKey("sagaIds")) {
                try {
                    sagaIds = (Set)this.mapper.readValue((String)taskMap.get("sagaIds"), SAGA_IDS_TYPE);
                }
                catch (IOException e) {
                    throw new SystemException("Could not deserialize sagaIds key", (Throwable)e);
                }
            } else {
                sagaIds = new HashSet();
            }
            return new JedisTask((String)taskMap.get("id"), Long.parseLong((String)taskMap.get("startTimeMs")), this, (String)taskMap.get("ownerId"), (String)taskMap.get("requestId"), sagaIds, oldTask);
        }
        return null;
    }

    @Override
    public Task getByClientRequestId(String clientRequestId) {
        String clientRequestKey = this.getClientRequestKey(clientRequestId);
        String existingTask = this.retry(() -> (String)this.redisClientDelegate.withCommandsClient(client -> client.get(clientRequestKey)), String.format("Getting task by client request ID %s", clientRequestId));
        if (existingTask == null && this.redisClientDelegatePrevious.isPresent()) {
            try {
                existingTask = (String)this.redisClientDelegatePrevious.get().withCommandsClient(client -> client.get(clientRequestKey));
            }
            catch (Exception e) {
                existingTask = null;
            }
        }
        if (existingTask != null) {
            return this.get(existingTask);
        }
        return null;
    }

    @Override
    public List<Task> list() {
        return this.retry(() -> (List)this.redisClientDelegate.withCommandsClient(client -> client.smembers(RUNNING_TASK_KEY).stream().map(this::get).collect(Collectors.toList())), "Getting all running tasks");
    }

    @Override
    public List<Task> listByThisInstance() {
        return this.list().stream().filter(t -> ClouddriverHostname.ID.equals(t.getOwnerId())).collect(Collectors.toList());
    }

    public void set(String id, JedisTask task) {
        String taskId = "task:" + task.getId();
        HashMap<String, String> data = new HashMap<String, String>();
        data.put("id", task.getId());
        data.put("startTimeMs", Long.toString(task.getStartTimeMs()));
        data.put("ownerId", task.getOwnerId());
        try {
            data.put("sagaIds", this.mapper.writeValueAsString(task.getSagaIds()));
        }
        catch (JsonProcessingException e) {
            throw new SystemException("Failed to serialize saga ids into Task", (Throwable)e);
        }
        this.retry(() -> this.redisClientDelegate.withCommandsClient(client -> {
            client.hmset(taskId, data);
            client.expire(taskId, TASK_TTL);
            client.sadd(RUNNING_TASK_KEY, new String[]{id});
        }), String.format("Writing task %s", id));
    }

    public void addToHistory(DefaultTaskStatus status, JedisTask task) {
        String hist;
        String historyId = "taskHistory:" + task.getId();
        HashMap<String, String> data = new HashMap<String, String>();
        data.put("phase", status.getPhase());
        data.put("status", status.getStatus());
        data.put("state", status.getState().toString());
        try {
            hist = this.mapper.writeValueAsString(data);
        }
        catch (JsonProcessingException e) {
            throw new RuntimeException("Failed converting task history to json", e);
        }
        this.retry(() -> this.redisClientDelegate.withCommandsClient(client -> {
            client.rpush(historyId, new String[]{hist});
            client.expire(historyId, TASK_TTL);
            if (status.isCompleted().booleanValue()) {
                client.srem(RUNNING_TASK_KEY, new String[]{task.getId()});
            }
        }), String.format("Adding status history to task %s: %s", task.getId(), status));
    }

    public List<Status> getHistory(JedisTask task) {
        String historyId = "taskHistory:" + task.getId();
        RedisClientDelegate client = this.clientForTask(task);
        return this.retry(() -> (List)client.withCommandsClient(c -> c.lrange(historyId, 0L, -1L)), String.format("Getting history for task %s", task.getId())).stream().map(h -> {
            Map history;
            try {
                history = (Map)this.mapper.readValue(h, HISTORY_TYPE);
            }
            catch (IOException e) {
                throw new RuntimeException("Could not convert history json to type", e);
            }
            return TaskDisplayStatus.create(DefaultTaskStatus.create((String)history.get("phase"), (String)history.get("status"), TaskState.valueOf((String)history.get("state"))));
        }).collect(Collectors.toList());
    }

    public DefaultTaskStatus currentState(JedisTask task) {
        Map history;
        String historyId = "taskHistory:" + task.getId();
        RedisClientDelegate client = this.clientForTask(task);
        String state = this.retry(() -> (String)client.withCommandsClient(c -> c.lindex(historyId, -1L)), String.format("Getting current state for task %s", task.getId()));
        try {
            history = (Map)this.mapper.readValue(state, HISTORY_TYPE);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed converting task history json to object", e);
        }
        return DefaultTaskStatus.create((String)history.get("phase"), (String)history.get("status"), TaskState.valueOf((String)history.get("state")));
    }

    public void addResultObjects(List<Object> objects, JedisTask task) {
        String resultId = "taskResult:" + task.getId();
        String[] values = objects.stream().map(o -> {
            try {
                return this.mapper.writeValueAsString(o);
            }
            catch (JsonProcessingException e) {
                throw new RuntimeException("Failed to convert object to string", e);
            }
        }).collect(Collectors.toList()).toArray(new String[objects.size()]);
        log.debug("Adding results to task {} (results: {})", (Object)task.getId(), (Object)values);
        this.retry(() -> this.redisClientDelegate.withCommandsClient(client -> {
            client.rpush(resultId, values);
            client.expire(resultId, TASK_TTL);
        }), String.format("Adding results to task %s", task.getId()));
    }

    public List<Object> getResultObjects(JedisTask task) {
        String resultId = "taskResult:" + task.getId();
        return this.retry(() -> (List)this.clientForTask(task).withCommandsClient(client -> client.lrange(resultId, 0L, -1L)), String.format("Getting results for task %s", task.getId())).stream().map(o -> {
            try {
                return (Map)this.mapper.readValue(o, Map.class);
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to convert result object to map", e);
            }
        }).collect(Collectors.toList());
    }

    private String getClientRequestKey(String clientRequestId) {
        return "kato:taskmap:" + clientRequestId;
    }

    private RedisClientDelegate clientForTask(JedisTask task) {
        if (task.getPreviousRedis() && this.redisClientDelegatePrevious.isPresent()) {
            return this.redisClientDelegatePrevious.get();
        }
        return this.redisClientDelegate;
    }

    private <T> T retry(Supplier<T> f, String onRetriesExceededMessage) {
        return this.retry(f, (CheckedConsumer<? extends Throwable>)((CheckedConsumer)failure -> {
            throw new ExcessiveRedisFailureRetries(onRetriesExceededMessage, (Throwable)failure);
        }));
    }

    private <T> T retry(Supplier<T> f, CheckedConsumer<? extends Throwable> retryExceededListener) {
        return (T)((SyncFailsafe)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).onRetriesExceeded(retryExceededListener)).get(f::get);
    }

    private void retry(Runnable f, String onRetriesExceededMessage) {
        this.retry(f, (CheckedConsumer<? extends Throwable>)((CheckedConsumer)failure -> {
            throw new ExcessiveRedisFailureRetries(onRetriesExceededMessage, (Throwable)failure);
        }));
    }

    private void retry(Runnable f, CheckedConsumer<? extends Throwable> retryExceededListener) {
        ((SyncFailsafe)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).onRetriesExceeded(retryExceededListener)).run(f::run);
    }

    private static class ExcessiveRedisFailureRetries
    extends RuntimeException {
        ExcessiveRedisFailureRetries(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

