/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.safety;

import com.google.common.collect.ImmutableMap;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.impl.Preconditions;
import com.netflix.spinnaker.clouddriver.core.services.Front50Service;
import com.netflix.spinnaker.clouddriver.exceptions.TrafficGuardException;
import com.netflix.spinnaker.clouddriver.model.ClusterProvider;
import com.netflix.spinnaker.clouddriver.model.HealthState;
import com.netflix.spinnaker.clouddriver.model.ServerGroup;
import com.netflix.spinnaker.clouddriver.names.NamerRegistry;
import com.netflix.spinnaker.clouddriver.safety.ClusterMatchRule;
import com.netflix.spinnaker.clouddriver.safety.ClusterMatcher;
import com.netflix.spinnaker.kork.dynamicconfig.DynamicConfigService;
import com.netflix.spinnaker.moniker.Moniker;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import retrofit.RetrofitError;

@Component
public class TrafficGuard {
    private static final String MIN_CAPACITY_RATIO = "traffic-guards.min-capacity-ratio";
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final List<ClusterProvider<?>> clusterProviders;
    private final Front50Service front50Service;
    private final Registry registry;
    private final DynamicConfigService dynamicConfigService;
    private final Id savesId;

    @Autowired
    public TrafficGuard(List<ClusterProvider<?>> clusterProviders, Optional<Front50Service> front50Service, Registry registry, DynamicConfigService dynamicConfigService) {
        this.clusterProviders = clusterProviders;
        this.front50Service = front50Service.orElse(null);
        this.registry = registry;
        this.dynamicConfigService = dynamicConfigService;
        this.savesId = registry.createId("trafficGuard.saves");
    }

    public void verifyInstanceTermination(String serverGroupName, List<String> instanceIds, String account, String location, String cloudProvider, String operationDescriptor) {
        throw new UnsupportedOperationException("verifyInstanceTermination method has not been migrated from Orca yet");
    }

    public void verifyTrafficRemoval(String serverGroupName, String account, String location, String cloudProvider, String operationDescriptor) {
        Moniker serverGroupMoniker = NamerRegistry.getDefaultNamer().deriveMoniker((Object)serverGroupName);
        ClusterProvider<?> clusterProvider = this.getClusterProvider(cloudProvider).orElseThrow(() -> new TrafficGuardException(String.format("Could not find ClusterProvider for cloud provider '%s'", cloudProvider)));
        Object cluster = clusterProvider.getCluster(serverGroupMoniker.getApp(), account, serverGroupMoniker.getCluster(), false);
        if (cluster == null) {
            throw new TrafficGuardException(String.format("Could not find cluster '%s' in '%s/%s'", serverGroupMoniker.getCluster(), account, location));
        }
        List<ServerGroup> targetServerGroups = cluster.getServerGroups().stream().filter(it -> it.getRegion().equals(location)).collect(Collectors.toList());
        ServerGroup serverGroupGoingAway = targetServerGroups.stream().filter(it -> serverGroupMoniker.equals((Object)it.getMoniker())).findFirst().orElseThrow(() -> {
            String message = String.format("Could not find server group '%s' in '%s/%s', found [%s]", serverGroupName, account, location, targetServerGroups.stream().map(it -> it.getMoniker().toString()).collect(Collectors.joining(", ")));
            this.log.error("{}\nContext: {}", (Object)message, this.generateContext(targetServerGroups));
            return new TrafficGuardException(message);
        });
        this.verifyTrafficRemoval(serverGroupGoingAway, targetServerGroups, account, operationDescriptor);
    }

    public void verifyTrafficRemoval(ServerGroup serverGroupGoingAway, Collection<ServerGroup> currentServerGroups, String account, String operationDescriptor) {
        this.verifyTrafficRemoval(Collections.singletonList(serverGroupGoingAway), currentServerGroups, account, operationDescriptor);
    }

    public void verifyTrafficRemoval(Collection<ServerGroup> serverGroupsGoingAway, Collection<ServerGroup> currentServerGroups, String account, String operationDescriptor) {
        if (serverGroupsGoingAway == null || serverGroupsGoingAway.isEmpty()) {
            return;
        }
        Preconditions.checkArg((!currentServerGroups.isEmpty() ? 1 : 0) != 0, (String)"currentServerGroups must not be empty");
        ServerGroup someServerGroup = serverGroupsGoingAway.stream().findAny().get();
        String location = someServerGroup.getRegion();
        Preconditions.checkArg((boolean)Stream.concat(serverGroupsGoingAway.stream(), currentServerGroups.stream()).allMatch(sg -> location.equals(sg.getRegion())), (String)("server groups must all be in the same location but some not in " + location));
        String cluster = someServerGroup.getMoniker().getCluster();
        Preconditions.checkArg((boolean)Stream.concat(serverGroupsGoingAway.stream(), currentServerGroups.stream()).allMatch(sg -> cluster.equals(sg.getMoniker().getCluster())), (String)("server groups must all be in the same cluster but some not in " + cluster));
        if (!this.hasDisableLock(someServerGroup.getMoniker(), account, location)) {
            this.log.debug("No traffic guard configured for '{}' in {}/{}", new Object[]{cluster, account, location});
            return;
        }
        Map<String, Integer> capacityByServerGroupName = currentServerGroups.stream().collect(Collectors.toMap(ServerGroup::getName, this::getServerGroupCapacity));
        Set<String> namesOfServerGroupsGoingAway = serverGroupsGoingAway.stream().map(ServerGroup::getName).collect(Collectors.toSet());
        int currentCapacity = capacityByServerGroupName.values().stream().reduce(0, Integer::sum);
        if (currentCapacity == 0) {
            this.log.debug("Bypassing traffic guard check for '{}' in {}/{} with no instances Up. Context: {}", new Object[]{cluster, account, location, this.generateContext(currentServerGroups)});
            return;
        }
        int capacityGoingAway = capacityByServerGroupName.entrySet().stream().filter(entry -> namesOfServerGroupsGoingAway.contains(entry.getKey())).map(Map.Entry::getValue).reduce(0, Integer::sum);
        int futureCapacity = currentCapacity - capacityGoingAway;
        int someDesiredSize = someServerGroup.getCapacity().getDesired();
        if (futureCapacity > 0 && serverGroupsGoingAway.size() > 1 && serverGroupsGoingAway.stream().allMatch(sg -> sg.getCapacity().isPinned()) && serverGroupsGoingAway.stream().allMatch(sg -> sg.getCapacity().getDesired() == someDesiredSize)) {
            this.log.debug("Bypassing traffic guard check for '{}' in {}/{} with pinned server groups of size {}. Context: {}", new Object[]{cluster, account, location, someDesiredSize, this.generateContext(currentServerGroups)});
            return;
        }
        double futureCapacityRatio = (double)futureCapacity / (double)currentCapacity;
        double minCapacityRatio = this.getMinCapacityRatio();
        if (futureCapacityRatio <= minCapacityRatio) {
            String message = this.generateUserFacingMessage(cluster, account, location, operationDescriptor, namesOfServerGroupsGoingAway, futureCapacity, currentCapacity, futureCapacityRatio, minCapacityRatio);
            this.log.debug("{}\nContext: {}", (Object)message, this.generateContext(currentServerGroups));
            this.registry.counter(this.savesId.withTags("application", someServerGroup.getMoniker().getApp(), "account", account)).increment();
            throw new TrafficGuardException(message);
        }
    }

    private String generateUserFacingMessage(String cluster, String account, String location, String operationDescriptor, Set<String> namesOfServerGroupsGoingAway, int futureCapacity, int currentCapacity, double futureCapacityRatio, double minCapacityRatio) {
        String message = String.format("This cluster ('%s' in %s/%s) has traffic guards enabled. %s [%s] would leave the cluster ", cluster, account, location, operationDescriptor, String.join((CharSequence)",", namesOfServerGroupsGoingAway));
        if (futureCapacity == 0) {
            return message + "with no instances up.";
        }
        String withInstances = futureCapacity == 1 ? "with 1 instance up " : String.format("with %d instances up ", futureCapacity);
        return message + withInstances + String.format("(%.1f%% of %d instances currently up). The configured minimum is %.1f%%.", futureCapacityRatio * 100.0, currentCapacity, minCapacityRatio * 100.0);
    }

    private double getMinCapacityRatio() {
        double defaultMinCapacityRatio = 0.0;
        try {
            Double minCapacityRatio = (Double)this.dynamicConfigService.getConfig(Double.class, MIN_CAPACITY_RATIO, (Object)defaultMinCapacityRatio);
            if (minCapacityRatio == null || minCapacityRatio < 0.0 || 0.5 <= minCapacityRatio) {
                this.log.error("Expecting a double value in range [0, 0.5] for {} but got {}", (Object)MIN_CAPACITY_RATIO, (Object)minCapacityRatio);
                return 0.0;
            }
            return minCapacityRatio;
        }
        catch (NumberFormatException e) {
            this.log.error("Expecting a double value in range [0, 0.5] for {}", (Object)MIN_CAPACITY_RATIO, (Object)e);
            return defaultMinCapacityRatio;
        }
    }

    private List<Map> generateContext(Collection<ServerGroup> targetServerGroups) {
        return targetServerGroups.stream().map(tsg -> ImmutableMap.builder().put((Object)"name", (Object)tsg.getName()).put((Object)"disabled", (Object)tsg.isDisabled()).put((Object)"instances", tsg.getInstances()).put((Object)"capacity", (Object)tsg.getCapacity()).build()).collect(Collectors.toList());
    }

    private int getServerGroupCapacity(ServerGroup serverGroup) {
        return (int)serverGroup.getInstances().stream().filter(it -> HealthState.Up.equals((Object)it.getHealthState())).count();
    }

    public boolean hasDisableLock(Moniker clusterMoniker, String account, String location) {
        Map application;
        if (this.front50Service == null) {
            this.log.warn("Front50 has not been configured, no way to check disable lock. Fix this by setting front50.enabled: true");
            return false;
        }
        try {
            application = this.front50Service.getApplication(clusterMoniker.getApp());
        }
        catch (RetrofitError e) {
            if (e.getResponse() != null && Arrays.asList(404, 403).contains(e.getResponse().getStatus())) {
                application = null;
            }
            throw e;
        }
        if (application == null || !application.containsKey("trafficGuards")) {
            return false;
        }
        List trafficGuards = (List)application.get("trafficGuards");
        List<ClusterMatchRule> rules = trafficGuards.stream().filter(guard -> guard.getOrDefault("enabled", true)).map(guard -> new ClusterMatchRule((String)guard.get("account"), (String)guard.get("location"), (String)guard.get("stack"), (String)guard.get("detail"), 1)).collect(Collectors.toList());
        return ClusterMatcher.getMatchingRule(account, location, clusterMoniker, rules) != null;
    }

    private Optional<ClusterProvider<?>> getClusterProvider(String cloudProvider) {
        return this.clusterProviders.stream().filter(it -> it.getCloudProviderId().equals(cloudProvider)).findFirst();
    }
}

