/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.clouddriver.kubernetes.op.job;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.google.gson.stream.JsonReader;
import com.netflix.spinnaker.clouddriver.data.task.Task;
import com.netflix.spinnaker.clouddriver.jobs.JobExecutor;
import com.netflix.spinnaker.clouddriver.jobs.JobRequest;
import com.netflix.spinnaker.clouddriver.jobs.JobResult;
import com.netflix.spinnaker.clouddriver.jobs.local.ReaderConsumer;
import com.netflix.spinnaker.clouddriver.kubernetes.config.KubernetesConfigurationProperties;
import com.netflix.spinnaker.clouddriver.kubernetes.description.JsonPatch;
import com.netflix.spinnaker.clouddriver.kubernetes.description.KubernetesPatchOptions;
import com.netflix.spinnaker.clouddriver.kubernetes.description.KubernetesPodMetric;
import com.netflix.spinnaker.clouddriver.kubernetes.description.manifest.KubernetesKind;
import com.netflix.spinnaker.clouddriver.kubernetes.description.manifest.KubernetesManifest;
import com.netflix.spinnaker.clouddriver.kubernetes.op.job.MetricParser;
import com.netflix.spinnaker.clouddriver.kubernetes.security.KubernetesCredentials;
import com.netflix.spinnaker.clouddriver.kubernetes.security.KubernetesSelectorList;
import com.netflix.spinnaker.kork.annotations.VisibleForTesting;
import com.netflix.spinnaker.kork.resilience4j.Resilience4jHelper;
import io.github.resilience4j.core.IntervalFunction;
import io.github.resilience4j.micrometer.tagged.TaggedRetryMetrics;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import io.github.resilience4j.retry.RetryRegistry;
import io.kubernetes.client.openapi.models.V1DeleteOptions;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import lombok.Generated;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class KubectlJobExecutor {
    private static final Logger log = LoggerFactory.getLogger(KubectlJobExecutor.class);
    private static final String NOT_FOUND_STRING = "(NotFound)";
    private static final String NO_OBJECTS_PASSED_TO_STRING = "error: no objects passed to";
    private static final String NO_OBJECTS_PASSED_TO_APPLY_STRING = "error: no objects passed to apply";
    private static final String NO_OBJECTS_PASSED_TO_CREATE_STRING = "error: no objects passed to create";
    private static final String KUBECTL_COMMAND_OPTION_TOKEN = "--token=";
    private static final String KUBECTL_COMMAND_OPTION_KUBECONFIG = "--kubeconfig=";
    private static final String KUBECTL_COMMAND_OPTION_CONTEXT = "--context=";
    private final JobExecutor jobExecutor;
    private final Gson gson = new Gson();
    private final KubernetesConfigurationProperties kubernetesConfigurationProperties;
    private final Optional<RetryRegistry> retryRegistry;
    private final MeterRegistry meterRegistry;

    @Autowired
    public KubectlJobExecutor(JobExecutor jobExecutor, KubernetesConfigurationProperties kubernetesConfigurationProperties, MeterRegistry meterRegistry) {
        this.jobExecutor = jobExecutor;
        this.kubernetesConfigurationProperties = kubernetesConfigurationProperties;
        this.meterRegistry = meterRegistry;
        this.retryRegistry = this.initializeRetryRegistry(kubernetesConfigurationProperties.getJobExecutor().getRetries());
    }

    private Optional<RetryRegistry> initializeRetryRegistry(KubernetesConfigurationProperties.KubernetesJobExecutorProperties.Retries retriesConfig) {
        if (retriesConfig.isEnabled()) {
            log.info("kubectl retries are enabled");
            RetryConfig.Builder retryConfig = RetryConfig.custom().maxAttempts(retriesConfig.getMaxAttempts());
            if (retriesConfig.isExponentialBackoffEnabled()) {
                retryConfig.intervalFunction(IntervalFunction.ofExponentialBackoff((Duration)Duration.ofMillis(retriesConfig.getExponentialBackOffIntervalMs()), (double)retriesConfig.getExponentialBackoffMultiplier()));
            } else {
                retryConfig.waitDuration(Duration.ofMillis(retriesConfig.getBackOffInMs()));
            }
            retryConfig.ignoreExceptions(new Class[]{NoRetryException.class});
            RetryRegistry retryRegistry = RetryRegistry.of((RetryConfig)retryConfig.build());
            Resilience4jHelper.configureLogging((RetryRegistry)retryRegistry, (String)"Kubectl command", (Logger)log);
            if (this.kubernetesConfigurationProperties.getJobExecutor().getRetries().getMetrics().isEnabled()) {
                TaggedRetryMetrics.ofRetryRegistry((RetryRegistry)retryRegistry).bindTo(this.meterRegistry);
            }
            return Optional.of(retryRegistry);
        }
        log.info("kubectl retries are disabled");
        return Optional.empty();
    }

    public String logs(KubernetesCredentials credentials, String namespace, String podName, String containerName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("logs");
        command.add(podName);
        command.add("-c=" + containerName);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to get logs from " + podName + "/" + containerName + " in " + namespace + ": " + status.getError());
        }
        return (String)status.getOutput();
    }

    public String jobLogs(KubernetesCredentials credentials, String namespace, String jobName, String containerName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        String resource = "job/" + jobName;
        command.add("logs");
        command.add(resource);
        command.add("-c=" + containerName);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to get logs from " + resource + " in " + namespace + ": " + status.getError());
        }
        return (String)status.getOutput();
    }

    public List<String> delete(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesSelectorList labelSelectors, V1DeleteOptions deleteOptions, Task task, String opName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("delete");
        command = this.kubectlLookupInfo(command, kind, name, labelSelectors);
        command.add("--ignore-not-found=true");
        if (deleteOptions.getPropagationPolicy() != null) {
            command.add("--cascade=" + deleteOptions.getPropagationPolicy());
        }
        if (deleteOptions.getGracePeriodSeconds() != null) {
            command.add("--grace-period=" + deleteOptions.getGracePeriodSeconds());
        }
        Object id = !Strings.isNullOrEmpty((String)name) ? String.valueOf(kind) + "/" + name : labelSelectors.toString();
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        this.persistKubectlJobOutput(credentials, status, (String)id, task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to delete " + (String)id + " from " + namespace + ": " + status.getError());
        }
        if (Strings.isNullOrEmpty((String)((String)status.getOutput())) || ((String)status.getOutput()).equals("No output from command.") || ((String)status.getOutput()).startsWith("No resources found")) {
            return new ArrayList<String>();
        }
        return Arrays.stream(((String)status.getOutput()).split("\n")).map(m -> m.substring(m.indexOf("\"") + 1)).map(m -> m.substring(0, m.lastIndexOf("\""))).collect(Collectors.toList());
    }

    public Void scale(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, int replicas, Task task, String opName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("scale");
        command = this.kubectlLookupInfo(command, kind, name, null);
        command.add("--replicas=" + replicas);
        String resource = String.valueOf(kind) + "/" + name;
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        this.persistKubectlJobOutput(credentials, status, resource, task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to scale " + resource + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public List<Integer> historyRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        String resource = String.valueOf(kind) + "/" + name;
        command.add("rollout");
        command.add("history");
        command.add(resource);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to get rollout history of " + resource + " from " + namespace + ": " + status.getError());
        }
        String stdout = (String)status.getOutput();
        if (Strings.isNullOrEmpty((String)stdout)) {
            return new ArrayList<Integer>();
        }
        List splitOutput = Arrays.stream(stdout.split("\n")).collect(Collectors.toList());
        if (splitOutput.size() <= 2) {
            return new ArrayList<Integer>();
        }
        splitOutput = splitOutput.subList(2, splitOutput.size());
        return splitOutput.stream().map(l -> l.split("[ \t]")).filter(l -> ((String[])l).length > 0).map(l -> l[0]).map(Integer::valueOf).collect(Collectors.toList());
    }

    public Void undoRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, int revision) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        String resource = String.valueOf(kind) + "/" + name;
        command.add("rollout");
        command.add("undo");
        command.add(resource);
        command.add("--to-revision=" + revision);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to undo rollout " + resource + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public Void pauseRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        String resource = String.valueOf(kind) + "/" + name;
        command.add("rollout");
        command.add("pause");
        command.add(resource);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to pause rollout " + resource + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public Void resumeRollout(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, Task task, String opName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        String resource = String.valueOf(kind) + "/" + name;
        command.add("rollout");
        command.add("resume");
        command.add(resource);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        this.persistKubectlJobOutput(credentials, status, resource, task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to resume rollout " + resource + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    public Void rollingRestart(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, Task task, String opName) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        String resource = String.valueOf(kind) + "/" + name;
        command.add("rollout");
        command.add("restart");
        command.add(resource);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        this.persistKubectlJobOutput(credentials, status, resource, task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to complete rolling restart of " + resource + " from " + namespace + ": " + status.getError());
        }
        return null;
    }

    @Nullable
    public KubernetesManifest get(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        log.debug("Getting information for {} of Kind {} in namespace {}", new Object[]{name, kind.toString(), namespace});
        List<String> command = this.kubectlNamespacedGet(credentials, (List<KubernetesKind>)ImmutableList.of((Object)kind), namespace);
        command.add(name);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (status.getError().contains(NOT_FOUND_STRING)) {
                return null;
            }
            throw new KubectlException("Failed to get: " + name + " of kind: " + String.valueOf(kind) + " from namespace: " + namespace + ": " + status.getError());
        }
        try {
            return (KubernetesManifest)this.gson.fromJson((String)status.getOutput(), KubernetesManifest.class);
        }
        catch (JsonSyntaxException e) {
            throw new KubectlException("Failed to parse kubectl output for: " + name + " of kind: " + String.valueOf(kind) + " in namespace: " + namespace + ": " + e.getMessage(), e);
        }
    }

    @Nonnull
    public ImmutableList<KubernetesManifest> eventsFor(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name) {
        log.debug("Getting events for {} of Kind {} in namespace {}", new Object[]{name, kind.toString(), namespace});
        List<String> command = this.kubectlNamespacedGet(credentials, (List<KubernetesKind>)ImmutableList.of((Object)KubernetesKind.EVENT), namespace);
        command.add("--field-selector");
        command.add(String.format("involvedObject.name=%s,involvedObject.kind=%s", name, StringUtils.capitalize((String)kind.toString())));
        JobResult<ImmutableList<KubernetesManifest>> status = this.executeKubectlCommand(credentials, command, this.parseManifestList());
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Failed to read events for: " + String.valueOf(kind) + "/" + name + " from " + namespace + ": " + status.getError());
        }
        if (status.getError().contains("No resources found")) {
            return ImmutableList.of();
        }
        return (ImmutableList)status.getOutput();
    }

    @Nonnull
    public ImmutableList<KubernetesManifest> list(KubernetesCredentials credentials, List<KubernetesKind> kinds, String namespace, KubernetesSelectorList selectors) {
        JobResult<ImmutableList<KubernetesManifest>> status;
        log.debug("Getting list of kinds {} in namespace {}", kinds, (Object)namespace);
        List<String> command = this.kubectlNamespacedGet(credentials, kinds, namespace);
        if (selectors.isNotEmpty()) {
            log.debug("with selectors: {}", (Object)selectors.toString());
            command.add("-l=" + selectors.toString());
        }
        if ((status = this.executeKubectlCommand(credentials, command, this.parseManifestList())).getResult() != JobResult.Result.SUCCESS) {
            boolean permissionError = StringUtils.containsIgnoreCase((CharSequence)status.getError(), (CharSequence)"forbidden");
            if (permissionError) {
                log.warn(status.getError());
            } else {
                throw new KubectlException("Failed to read " + String.valueOf(kinds) + " from " + namespace + ": " + status.getError());
            }
        }
        if (status.getError().contains("No resources found")) {
            return ImmutableList.of();
        }
        return (ImmutableList)status.getOutput();
    }

    public KubernetesManifest deploy(KubernetesCredentials credentials, KubernetesManifest manifest, Task task, String opName, KubernetesSelectorList labelSelectors, String ... cmdArgs) {
        log.info("Deploying manifest {}", (Object)manifest.getFullResourceName());
        List<String> command = this.kubectlAuthPrefix(credentials);
        command.add("apply");
        command.addAll(List.of(cmdArgs));
        command.add("-o");
        command.add("json");
        command.add("-f");
        command.add("-");
        this.addLabelSelectors(command, labelSelectors);
        JobResult<String> status = this.executeKubectlCommand(credentials, command, Optional.of(manifest));
        this.persistKubectlJobOutput(credentials, status, manifest.getFullResourceName(), task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (labelSelectors.isNotEmpty() && status.getError().contains(NO_OBJECTS_PASSED_TO_APPLY_STRING)) {
                return null;
            }
            throw new KubectlException("Deploy failed for manifest: " + manifest.getFullResourceName() + ". Error: " + status.getError());
        }
        return this.getKubernetesManifestFromJobResult(status, manifest);
    }

    public KubernetesManifest replace(KubernetesCredentials credentials, KubernetesManifest manifest, Task task, String opName) {
        log.info("Replacing manifest {}", (Object)manifest.getFullResourceName());
        List<String> command = this.kubectlAuthPrefix(credentials);
        command.add("replace");
        command.add("-o");
        command.add("json");
        command.add("-f");
        command.add("-");
        JobResult<String> status = this.executeKubectlCommand(credentials, command, Optional.of(manifest));
        this.persistKubectlJobOutput(credentials, status, manifest.getFullResourceName(), task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (status.getError().contains(NOT_FOUND_STRING)) {
                throw new KubectlNotFoundException("Replace failed for manifest: " + manifest.getFullResourceName() + ". Error: " + status.getError());
            }
            throw new KubectlException("Replace failed for manifest: " + manifest.getFullResourceName() + ". Error: " + status.getError());
        }
        return this.getKubernetesManifestFromJobResult(status, manifest);
    }

    public KubernetesManifest create(KubernetesCredentials credentials, KubernetesManifest manifest, Task task, String opName, KubernetesSelectorList labelSelectors) {
        log.info("Creating manifest {}", (Object)manifest.getFullResourceName());
        List<String> command = this.kubectlAuthPrefix(credentials);
        command.add("create");
        command.add("-o");
        command.add("json");
        command.add("-f");
        command.add("-");
        this.addLabelSelectors(command, labelSelectors);
        JobResult<String> status = this.executeKubectlCommand(credentials, command, Optional.of(manifest));
        this.persistKubectlJobOutput(credentials, status, manifest.getFullResourceName(), task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (labelSelectors.isNotEmpty() && status.getError().contains(NO_OBJECTS_PASSED_TO_CREATE_STRING)) {
                return null;
            }
            throw new KubectlException("Create failed for manifest: " + manifest.getFullResourceName() + ". Error: " + status.getError());
        }
        return this.getKubernetesManifestFromJobResult(status, manifest);
    }

    private KubernetesManifest getKubernetesManifestFromJobResult(JobResult<String> status, KubernetesManifest inputManifest) {
        try {
            return (KubernetesManifest)this.gson.fromJson((String)status.getOutput(), KubernetesManifest.class);
        }
        catch (JsonSyntaxException e) {
            throw new KubectlException("Failed to parse kubectl output for manifest: " + inputManifest.getName() + ". Error: " + e.getMessage(), e);
        }
    }

    private List<String> kubectlAuthPrefix(KubernetesCredentials credentials) {
        ArrayList<String> command = new ArrayList<String>();
        if (!Strings.isNullOrEmpty((String)credentials.getKubectlExecutable())) {
            command.add(credentials.getKubectlExecutable());
        } else {
            command.add(this.kubernetesConfigurationProperties.getKubectl().getExecutable());
        }
        if (credentials.getKubectlRequestTimeoutSeconds() != null) {
            command.add("--request-timeout=" + credentials.getKubectlRequestTimeoutSeconds());
        }
        if (credentials.isDebug()) {
            command.add("-v");
            command.add("9");
        }
        if (!credentials.isServiceAccount()) {
            String context;
            String kubeconfigFile;
            if (credentials.getOAuthServiceAccount() != null && !credentials.getOAuthServiceAccount().isEmpty()) {
                command.add(KUBECTL_COMMAND_OPTION_TOKEN + this.getOAuthToken(credentials));
            }
            if (!Strings.isNullOrEmpty((String)(kubeconfigFile = credentials.getKubeconfigFile()))) {
                command.add(KUBECTL_COMMAND_OPTION_KUBECONFIG + kubeconfigFile);
            }
            if (!Strings.isNullOrEmpty((String)(context = credentials.getContext()))) {
                command.add(KUBECTL_COMMAND_OPTION_CONTEXT + context);
            }
        }
        return command;
    }

    private List<String> kubectlLookupInfo(List<String> command, KubernetesKind kind, String name, KubernetesSelectorList labelSelectors) {
        if (!Strings.isNullOrEmpty((String)name)) {
            command.add(String.valueOf(kind) + "/" + name);
        } else {
            command.add(kind.toString());
        }
        this.addLabelSelectors(command, labelSelectors);
        return command;
    }

    private List<String> kubectlNamespacedAuthPrefix(KubernetesCredentials credentials, String namespace) {
        List<String> command = this.kubectlAuthPrefix(credentials);
        if (!Strings.isNullOrEmpty((String)namespace)) {
            command.add("--namespace=" + namespace);
        }
        return command;
    }

    private List<String> kubectlNamespacedGet(KubernetesCredentials credentials, List<KubernetesKind> kind, String namespace) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("-o");
        command.add("json");
        command.add("get");
        command.add(kind.stream().map(KubernetesKind::toString).collect(Collectors.joining(",")));
        return command;
    }

    private String getOAuthToken(KubernetesCredentials credentials) {
        ArrayList<String> command = new ArrayList<String>();
        command.add(this.kubernetesConfigurationProperties.getOAuth().getExecutable());
        command.add("fetch");
        command.add("--json");
        command.add(credentials.getOAuthServiceAccount());
        command.addAll(credentials.getOAuthScopes());
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            throw new KubectlException("Could not fetch OAuth token: " + status.getError());
        }
        return (String)status.getOutput();
    }

    public ImmutableList<KubernetesPodMetric> topPod(KubernetesCredentials credentials, String namespace, @Nonnull String pod) {
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("top");
        command.add("po");
        if (!pod.isEmpty()) {
            command.add(pod);
        }
        command.add("--containers");
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            if (status.getError().toLowerCase().contains("not available") || status.getError().toLowerCase().contains("not found")) {
                log.warn(String.format("Error fetching metrics for account %s: %s", credentials.getAccountName(), status.getError()));
                return ImmutableList.of();
            }
            throw new KubectlException("Could not read metrics: " + status.getError());
        }
        ImmutableSetMultimap<String, KubernetesPodMetric.ContainerMetric> metrics = MetricParser.parseMetrics((String)status.getOutput());
        return (ImmutableList)metrics.asMap().entrySet().stream().map(podMetrics -> KubernetesPodMetric.builder().podName((String)podMetrics.getKey()).namespace(namespace).containerMetrics((Iterable)podMetrics.getValue()).build()).collect(ImmutableList.toImmutableList());
    }

    public Void patch(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesPatchOptions options, List<JsonPatch> patches, Task task, String opName) {
        return this.patch(credentials, kind, namespace, name, options, this.gson.toJson(patches), task, opName);
    }

    public Void patch(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesPatchOptions options, KubernetesManifest manifest, Task task, String opName) {
        return this.patch(credentials, kind, namespace, name, options, this.gson.toJson((Object)manifest), task, opName);
    }

    private Void patch(KubernetesCredentials credentials, KubernetesKind kind, String namespace, String name, KubernetesPatchOptions options, String patchBody, Task task, String opName) {
        String mergeStrategy;
        List<String> command = this.kubectlNamespacedAuthPrefix(credentials, namespace);
        command.add("patch");
        command.add(kind.toString());
        command.add(name);
        if (options.isRecord()) {
            command.add("--record");
        }
        if (!Strings.isNullOrEmpty((String)(mergeStrategy = options.getMergeStrategy().toString()))) {
            command.add("--type");
            command.add(mergeStrategy);
        }
        command.add("--patch");
        command.add(patchBody);
        JobResult<String> status = this.executeKubectlCommand(credentials, command);
        this.persistKubectlJobOutput(credentials, status, String.valueOf(kind) + "/" + name, task, opName);
        if (status.getResult() != JobResult.Result.SUCCESS) {
            String errMsg = status.getError();
            if (Strings.isNullOrEmpty((String)errMsg)) {
                errMsg = (String)status.getOutput();
            }
            if (errMsg.contains("not patched")) {
                log.warn("No change occurred after patching {} {}:{}, ignoring", new Object[]{kind, namespace, name});
                return null;
            }
            throw new KubectlException("Patch failed for: " + name + " in namespace: " + namespace + ": " + errMsg);
        }
        return null;
    }

    private ReaderConsumer<ImmutableList<KubernetesManifest>> parseManifestList() {
        return r -> {
            try {
                JsonReader reader = new JsonReader((Reader)r);
                try {
                    try {
                        reader.beginObject();
                    }
                    catch (EOFException e) {
                        ImmutableList immutableList = ImmutableList.of();
                        reader.close();
                        return immutableList;
                    }
                    ImmutableList.Builder manifestList = new ImmutableList.Builder();
                    while (reader.hasNext()) {
                        if (reader.nextName().equals("items")) {
                            reader.beginArray();
                            while (reader.hasNext()) {
                                KubernetesManifest manifest = (KubernetesManifest)this.gson.fromJson(reader, KubernetesManifest.class);
                                manifestList.add((Object)manifest);
                            }
                            reader.endArray();
                            continue;
                        }
                        reader.skipValue();
                    }
                    reader.endObject();
                    ImmutableList immutableList = manifestList.build();
                    return immutableList;
                }
                finally {
                    try {
                        reader.close();
                    }
                    catch (Throwable throwable2) {
                        Throwable throwable;
                        throwable.addSuppressed(throwable2);
                    }
                }
            }
            catch (JsonSyntaxException | IllegalStateException e) {
                throw new KubectlException("Failed to parse kubectl output: " + e.getMessage(), e);
            }
        };
    }

    private JobResult<String> executeKubectlCommand(KubernetesCredentials credentials, List<String> command) {
        return this.executeKubectlCommand(credentials, command, Optional.empty());
    }

    private JobResult<String> executeKubectlCommand(KubernetesCredentials credentials, List<String> command, Optional<KubernetesManifest> manifest) {
        if (this.retryRegistry.isEmpty()) {
            return this.jobExecutor.runJob(this.createJobRequest(command, manifest));
        }
        JobResult.JobResultBuilder finalResult = JobResult.builder();
        KubectlActionIdentifier identifier = new KubectlActionIdentifier(credentials, command, manifest);
        Retry retryContext = this.retryRegistry.get().retry(identifier.getRetryInstanceName());
        try {
            return (JobResult)retryContext.executeSupplier(() -> {
                JobResult result = this.jobExecutor.runJob(this.createJobRequest(command, manifest));
                return this.processJobResult(identifier, result, finalResult);
            });
        }
        catch (KubectlException | NoRetryException e) {
            return finalResult.build();
        }
    }

    private <T> JobResult<T> executeKubectlCommand(KubernetesCredentials credentials, List<String> command, ReaderConsumer<T> readerConsumer) {
        if (this.retryRegistry.isEmpty()) {
            return this.jobExecutor.runJob(new JobRequest(command), readerConsumer);
        }
        JobResult.JobResultBuilder finalResult = JobResult.builder();
        KubectlActionIdentifier identifier = new KubectlActionIdentifier(credentials, command);
        Retry retryContext = this.retryRegistry.get().retry(identifier.getRetryInstanceName());
        try {
            return (JobResult)retryContext.executeSupplier(() -> {
                JobResult result = this.jobExecutor.runJob(new JobRequest(command), readerConsumer);
                return this.processJobResult(identifier, result, finalResult);
            });
        }
        catch (KubectlException | NoRetryException e) {
            return finalResult.build();
        }
    }

    @VisibleForTesting
    JobRequest createJobRequest(List<String> command, Optional<KubernetesManifest> manifest) {
        if (manifest.isPresent()) {
            String manifestAsJson = this.gson.toJson((Object)manifest.get());
            return new JobRequest(command, (InputStream)new ByteArrayInputStream(manifestAsJson.getBytes(StandardCharsets.UTF_8)));
        }
        return new JobRequest(command);
    }

    @VisibleForTesting
    <T> JobResult<T> processJobResult(KubectlActionIdentifier identifier, JobResult<T> result, JobResult.JobResultBuilder<T> finalResult) {
        if (result.getResult() == JobResult.Result.SUCCESS) {
            return result;
        }
        finalResult.error(result.getError()).killed(result.isKilled()).output(result.getOutput()).result(result.getResult());
        throw this.convertKubectlJobResultToException(identifier.getKubectlAction(), result);
    }

    private <T> RuntimeException convertKubectlJobResultToException(String identifier, JobResult<T> result) {
        if (this.kubernetesConfigurationProperties.getJobExecutor().getRetries().getRetryableErrorMessages().stream().anyMatch(errorMessage -> result.getError().contains((CharSequence)errorMessage))) {
            return new KubectlException(identifier + " failed. Error: " + result.getError());
        }
        if (result.isKilled()) {
            return new KubectlException("retrying " + identifier + " since the job " + String.valueOf(result) + " was killed");
        }
        String message = "Not retrying " + identifier + " as retries are not enabled for error: " + result.getError();
        log.warn(message);
        return new NoRetryException(message);
    }

    private void persistKubectlJobOutput(KubernetesCredentials credentials, JobResult<String> status, String manifestName, Task task, String taskName) {
        if (this.kubernetesConfigurationProperties.getJobExecutor().isPersistTaskOutput() && (this.kubernetesConfigurationProperties.getJobExecutor().isEnableTaskOutputForAllAccounts() || credentials.isDebug())) {
            task.updateOutput(manifestName, taskName, (String)status.getOutput(), status.getError());
        }
    }

    private void addLabelSelectors(List<String> command, KubernetesSelectorList labelSelectors) {
        if (labelSelectors != null && !labelSelectors.isEmpty()) {
            command.add("-l=" + String.valueOf(labelSelectors));
        }
    }

    @Generated
    public Optional<RetryRegistry> getRetryRegistry() {
        return this.retryRegistry;
    }

    static class NoRetryException
    extends RuntimeException {
        NoRetryException(String message) {
            super(message);
        }
    }

    public static class KubectlException
    extends RuntimeException {
        public KubectlException(String message) {
            super(message);
        }

        public KubectlException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class KubectlNotFoundException
    extends KubectlException {
        public KubectlNotFoundException(String message) {
            super(message);
        }
    }

    static class KubectlActionIdentifier {
        KubernetesCredentials credentials;
        List<String> command;
        String namespace;
        String resource;

        public KubectlActionIdentifier(KubernetesCredentials credentials, List<String> command, String namespace, String resource) {
            this.credentials = credentials;
            this.command = command;
            this.namespace = namespace;
            this.resource = resource;
        }

        public KubectlActionIdentifier(KubernetesCredentials credentials, List<String> command) {
            this(credentials, command, "", "");
        }

        public KubectlActionIdentifier(KubernetesCredentials credentials, List<String> command, Optional<KubernetesManifest> manifest) {
            this(credentials, command);
            if (manifest.isPresent()) {
                this.namespace = manifest.get().getNamespace();
                this.resource = manifest.get().getFullResourceName();
            }
        }

        public String getKubectlAction() {
            List commandToLog = this.command.stream().filter(s -> !s.contains(KubectlJobExecutor.KUBECTL_COMMAND_OPTION_TOKEN) && !s.contains(KubectlJobExecutor.KUBECTL_COMMAND_OPTION_KUBECONFIG) && !s.contains(KubectlJobExecutor.KUBECTL_COMMAND_OPTION_CONTEXT)).collect(Collectors.toList());
            String identifier = "command: '" + String.join((CharSequence)" ", commandToLog) + "' in account: " + this.credentials.getAccountName();
            if (!this.namespace.isEmpty()) {
                identifier = identifier + " in namespace: " + this.namespace;
            }
            if (!this.resource.isEmpty()) {
                identifier = identifier + " for resource: " + this.resource;
            }
            return identifier;
        }

        public String getRetryInstanceName() {
            return this.credentials.getAccountName();
        }
    }
}

