package io.stigg.api.client;

import com.apollographql.apollo3.ApolloClient;
import com.apollographql.apollo3.api.Mutation;
import com.apollographql.apollo3.api.Query;
import com.apollographql.apollo3.network.http.DefaultHttpEngine;
import com.apollographql.apollo3.rx2.Rx2Apollo;
import io.stigg.api.operations.GetEntitlementQuery;
import io.stigg.api.operations.GetEntitlementsQuery;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import org.jetbrains.annotations.NotNull;

public class Stigg {
  private final ClientConfig config;
  private final ApolloClient apolloClient;
  private final EdgeApiClient edgeApiClient;

  private Stigg(ClientConfig config) {
    this.config = config;

    var httpClient =
        new OkHttpClient.Builder()
            .connectTimeout(config.getRequestConnectTimeoutMs(), TimeUnit.MILLISECONDS)
            .readTimeout(config.getRequestReadTimeoutMs(), TimeUnit.MILLISECONDS)
            .retryOnConnectionFailure(true)
            .build();

    var httpEngine =
        new DefaultHttpEngine(
            (@NotNull Request request) ->
                new FailsafeCallAdapter(NetworkRequestUtils.toFailsafeCall(httpClient, request)));

    apolloClient =
        new ApolloClient.Builder()
            .serverUrl(config.getApiUrl())
            .addHttpHeader(NetworkRequestUtils.AUTH_HEADER, config.getApiKey())
            .httpEngine(httpEngine)
            .build();

    edgeApiClient = new EdgeApiClient(config, httpClient);

    Runtime.getRuntime().addShutdownHook(new Thread(this::tryCloseClient));
  }

  public void close() {
    tryCloseClient();
  }

  private void tryCloseClient() {
    try {
      if (apolloClient != null) {
        apolloClient.close();
      }
    } catch (Exception ignored) {
    }
  }

  @SuppressWarnings("unchecked")
  public <D extends Query.Data> D query(Query<D> query) {
    if (!this.config.getDisableEdge()) {
      if (query instanceof GetEntitlementsQuery) {
        return (D) this.edgeApiClient.getEntitlements((GetEntitlementsQuery) query);
      } else if (query instanceof GetEntitlementQuery) {
        return (D) this.edgeApiClient.getEntitlement((GetEntitlementQuery) query);
      }
    }

    var call = apolloClient.query(query);
    var result = Rx2Apollo.single(call).blockingGet();
    return result.dataAssertNoErrors();
  }

  public <D extends Mutation.Data> D mutation(Mutation<D> mutation) {
    var call = apolloClient.mutation(mutation);
    var result = Rx2Apollo.single(call).blockingGet();
    return result.dataAssertNoErrors();
  }

  public static Stigg createClient(@Nonnull String apiKey) {
    Objects.requireNonNull(apiKey, "apiKey cannot be null");
    return createClient(ClientConfig.builder().apiKey(apiKey).build());
  }

  public static Stigg createClient(@Nonnull ClientConfig config) {
    return new Stigg(config);
  }
}
