package io.stigg.api.client;

import com.apollographql.apollo3.api.CustomScalarAdapters;
import com.apollographql.apollo3.api.Operation;
import com.apollographql.apollo3.api.Operations;
import io.stigg.api.operations.GetEntitlementQuery;
import io.stigg.api.operations.GetEntitlementsQuery;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import okhttp3.OkHttpClient;
import okhttp3.Request;

class EdgeApiClient {

  private static String API_VERSION_HEADER = "X-API-VERSION";
  private static String API_VERSION_VALUE = "1";

  private final ClientConfig config;
  private final OkHttpClient httpClient;

  EdgeApiClient(ClientConfig config, OkHttpClient httpClient) {
    this.config = config;
    this.httpClient = httpClient;
  }

  GetEntitlementsQuery.Data getEntitlements(GetEntitlementsQuery query) {
    var params = query.query;
    var customerId = params.customerId;
    var resourceId = params.resourceId.getOrNull();

    var requestParams = new ArrayList<String>();
    if (resourceId != null) {
      requestParams.add("resourceId=" + resourceId);
    }

    var url = this.composeUrl("/v1/c/" + customerId + "/entitlements.json", requestParams);

    return this.exchange(query, url);
  }

  GetEntitlementQuery.Data getEntitlement(GetEntitlementQuery query) {
    var params = query.query;
    var customerId = params.customerId;
    var featureId = params.featureId;
    var resourceId = params.resourceId.getOrNull();
    var options = params.options.getOrNull();
    var requestedUsage = (options != null) ? options.requestedUsage.getOrNull() : null;

    var requestParams = new ArrayList<String>();
    requestParams.add("featureId=" + featureId);
    if (resourceId != null) {
      requestParams.add("resourceId=" + resourceId);
    }
    if (requestedUsage != null) {
      requestParams.add("requestedUsage=" + requestedUsage);
    }

    var url = this.composeUrl("/v1/c/" + customerId + "/entitlements.json", requestParams);

    return this.exchange(query, url);
  }

  private <D extends Operation.Data> D exchange(Operation<D> operation, String url) {
    Request request =
        new Request.Builder()
            .url(url)
            .header(NetworkRequestUtils.AUTH_HEADER, this.config.getApiKey())
            .header(API_VERSION_HEADER, API_VERSION_VALUE)
            .build();

    var call = NetworkRequestUtils.toFailsafeCall(httpClient, request);

    try (var response = call.execute()) {
      var body = response.body();
      assert body != null;
      var res = Operations.parseJsonResponse(operation, body.string(), CustomScalarAdapters.Empty);
      return res.dataAssertNoErrors();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private String composeUrl(String path, List<String> requestParams) {
    var base = this.composeUrl(path);
    return requestParams.isEmpty() ? base : base + "?" + String.join("&", requestParams);
  }

  private String composeUrl(String path) {
    return this.config.getEdgeApiUrl() + path;
  }
}
