/*
 * Decompiled with CFR 0.152.
 */
package dlshade.org.apache.bookkeeper.bookie.storage.ldb;

import dlshade.com.google.common.collect.Lists;
import dlshade.com.google.protobuf.ByteString;
import dlshade.org.apache.bookkeeper.bookie.BookieImpl;
import dlshade.org.apache.bookkeeper.bookie.DefaultEntryLogger;
import dlshade.org.apache.bookkeeper.bookie.Journal;
import dlshade.org.apache.bookkeeper.bookie.LedgerDirsManager;
import dlshade.org.apache.bookkeeper.bookie.storage.EntryLogScanner;
import dlshade.org.apache.bookkeeper.bookie.storage.ldb.ArrayUtil;
import dlshade.org.apache.bookkeeper.bookie.storage.ldb.DbLedgerStorageDataFormats;
import dlshade.org.apache.bookkeeper.bookie.storage.ldb.KeyValueStorage;
import dlshade.org.apache.bookkeeper.bookie.storage.ldb.KeyValueStorageFactory;
import dlshade.org.apache.bookkeeper.bookie.storage.ldb.KeyValueStorageRocksDB;
import dlshade.org.apache.bookkeeper.conf.ServerConfiguration;
import dlshade.org.apache.bookkeeper.util.DiskChecker;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.CopyOption;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LedgersIndexRebuildOp {
    private static final Logger LOG = LoggerFactory.getLogger(LedgersIndexRebuildOp.class);
    private final ServerConfiguration conf;
    private final boolean verbose;
    private static final String LedgersSubPath = "ledgers";

    public LedgersIndexRebuildOp(ServerConfiguration conf, boolean verbose) {
        this.conf = conf;
        this.verbose = verbose;
    }

    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_WOULD_HAVE_BEEN_A_NPE"})
    public boolean initiate() {
        LOG.info("Starting ledger index rebuilding");
        File[] indexDirs = this.conf.getIndexDirs();
        if (indexDirs == null) {
            indexDirs = this.conf.getLedgerDirs();
        }
        if (indexDirs.length != this.conf.getLedgerDirs().length) {
            LOG.error("ledger and index dirs size not matched");
            return false;
        }
        for (int i = 0; i < indexDirs.length; ++i) {
            File indexDir = indexDirs[i];
            File ledgerDir = this.conf.getLedgerDirs()[i];
            String timestamp = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ").format(new Date());
            String indexBasePath = BookieImpl.getCurrentDirectory(indexDir).toString();
            String tempLedgersSubPath = "ledgers.TEMP-" + timestamp;
            Path indexTempPath = FileSystems.getDefault().getPath(indexBasePath, tempLedgersSubPath);
            Path indexCurrentPath = FileSystems.getDefault().getPath(indexBasePath, LedgersSubPath);
            LOG.info("Starting scan phase (scans journal and entry log files)");
            try {
                HashSet<Long> ledgers = new HashSet<Long>();
                this.scanJournals(ledgers);
                File[] lDirs = new File[]{ledgerDir};
                this.scanEntryLogFiles(ledgers, lDirs);
                LOG.info("Scan complete, found {} ledgers. Starting to build a new ledgers index", (Object)ledgers.size());
                try (KeyValueStorage newIndex = KeyValueStorageRocksDB.factory.newKeyValueStorage(indexBasePath, tempLedgersSubPath, KeyValueStorageFactory.DbConfigType.Default, this.conf);){
                    LOG.info("Created ledgers index at temp location {}", (Object)indexTempPath);
                    for (Long ledgerId : ledgers) {
                        DbLedgerStorageDataFormats.LedgerData ledgerData = DbLedgerStorageDataFormats.LedgerData.newBuilder().setExists(true).setFenced(true).setMasterKey(ByteString.EMPTY).build();
                        byte[] ledgerArray = new byte[16];
                        ArrayUtil.setLong(ledgerArray, 0, ledgerId);
                        newIndex.put(ledgerArray, ledgerData.toByteArray());
                    }
                    newIndex.sync();
                }
            }
            catch (Throwable t) {
                LOG.error("Error during rebuild, the original index remains unchanged", t);
                this.delete(indexTempPath);
                return false;
            }
            try {
                Path prevPath = FileSystems.getDefault().getPath(indexBasePath, "ledgers.PREV-" + timestamp);
                LOG.info("Moving original index from original location: {} up to back-up location: {}", (Object)indexCurrentPath, (Object)prevPath);
                Files.move(indexCurrentPath, prevPath, new CopyOption[0]);
                LOG.info("Moving rebuilt index from: {} to: {}", (Object)indexTempPath, (Object)indexCurrentPath);
                Files.move(indexTempPath, indexCurrentPath, new CopyOption[0]);
                LOG.info("Original index has been replaced with the new index. The original index has been moved to {}", (Object)prevPath);
                continue;
            }
            catch (IOException e) {
                LOG.error("Could not replace original index with rebuilt index. To return to the original state, ensure the original index is in its original location", (Throwable)e);
                return false;
            }
        }
        return true;
    }

    private void scanEntryLogFiles(final Set<Long> ledgers, File[] lDirs) throws IOException {
        DefaultEntryLogger entryLogger = new DefaultEntryLogger(this.conf, new LedgerDirsManager(this.conf, lDirs, new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold())));
        Set<Long> entryLogs = entryLogger.getEntryLogsSet();
        int totalEntryLogs = entryLogs.size();
        int completedEntryLogs = 0;
        LOG.info("Scanning {} entry logs", (Object)totalEntryLogs);
        for (long entryLogId : entryLogs) {
            entryLogger.scanEntryLog(entryLogId, new EntryLogScanner(){

                @Override
                public void process(long ledgerId, long offset, ByteBuf entry) throws IOException {
                    if (ledgers.add(ledgerId) && LedgersIndexRebuildOp.this.verbose) {
                        LOG.info("Found ledger {} in entry log", (Object)ledgerId);
                    }
                }

                @Override
                public boolean accept(long ledgerId) {
                    return true;
                }
            });
            LOG.info("Completed scanning of log {}.log -- {} / {}", new Object[]{Long.toHexString(entryLogId), ++completedEntryLogs, totalEntryLogs});
        }
    }

    private void scanJournals(Set<Long> ledgers) throws IOException {
        for (Journal journal : this.getJournals(this.conf)) {
            List<Long> journalIds = Journal.listJournalIds(journal.getJournalDirectory(), new Journal.JournalIdFilter(){

                @Override
                public boolean accept(long journalId) {
                    return true;
                }
            });
            for (Long journalId : journalIds) {
                this.scanJournal(journal, journalId, ledgers);
            }
        }
    }

    private List<Journal> getJournals(ServerConfiguration conf) throws IOException {
        ArrayList<Journal> journals = Lists.newArrayListWithCapacity(conf.getJournalDirs().length);
        int idx = 0;
        for (File journalDir : conf.getJournalDirs()) {
            journals.add(new Journal(idx++, new File(journalDir, "current"), conf, new LedgerDirsManager(conf, conf.getLedgerDirs(), new DiskChecker(conf.getDiskUsageThreshold(), conf.getDiskUsageWarnThreshold()))));
        }
        return journals;
    }

    private void scanJournal(Journal journal, long journalId, final Set<Long> ledgers) throws IOException {
        LOG.info("Scanning journal " + journalId + " (" + Long.toHexString(journalId) + ".txn)");
        journal.scanJournal(journalId, 0L, new Journal.JournalScanner(){

            @Override
            public void process(int journalVersion, long offset, ByteBuffer entry) {
                ByteBuf buf = Unpooled.wrappedBuffer((ByteBuffer)entry);
                long ledgerId = buf.readLong();
                if (ledgers.add(ledgerId) && LedgersIndexRebuildOp.this.verbose) {
                    LOG.info("Found ledger {} in journal", (Object)ledgerId);
                }
            }
        }, false);
    }

    private void delete(Path path) {
        try {
            Files.delete(path);
        }
        catch (IOException e) {
            LOG.warn("Unable to delete {}", (Object)path.toAbsolutePath(), (Object)e);
        }
    }
}

