/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.admin.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.pulsar.broker.admin.AdminResource;
import org.apache.pulsar.broker.web.RestException;
import org.apache.pulsar.client.admin.Namespaces;
import org.apache.pulsar.shade.com.fasterxml.jackson.core.JsonGenerationException;
import org.apache.pulsar.shade.com.fasterxml.jackson.databind.JsonMappingException;
import org.apache.pulsar.shade.com.google.common.collect.Lists;
import org.apache.pulsar.shade.com.google.common.collect.Maps;
import org.apache.pulsar.shade.io.swagger.annotations.ApiOperation;
import org.apache.pulsar.shade.io.swagger.annotations.ApiParam;
import org.apache.pulsar.shade.io.swagger.annotations.ApiResponse;
import org.apache.pulsar.shade.io.swagger.annotations.ApiResponses;
import org.apache.pulsar.shade.io.swagger.annotations.Example;
import org.apache.pulsar.shade.io.swagger.annotations.ExampleProperty;
import org.apache.pulsar.shade.javax.ws.rs.DELETE;
import org.apache.pulsar.shade.javax.ws.rs.GET;
import org.apache.pulsar.shade.javax.ws.rs.POST;
import org.apache.pulsar.shade.javax.ws.rs.PUT;
import org.apache.pulsar.shade.javax.ws.rs.Path;
import org.apache.pulsar.shade.javax.ws.rs.PathParam;
import org.apache.pulsar.shade.javax.ws.rs.container.AsyncResponse;
import org.apache.pulsar.shade.javax.ws.rs.container.Suspended;
import org.apache.pulsar.shade.javax.ws.rs.core.Response;
import org.apache.pulsar.shade.org.apache.bookkeeper.util.ZkUtils;
import org.apache.pulsar.shade.org.apache.pulsar.common.naming.NamedEntity;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.BrokerNamespaceIsolationData;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.ClusterData;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.FailureDomain;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.NamespaceIsolationData;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.impl.NamespaceIsolationPolicies;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.impl.NamespaceIsolationPolicyImpl;
import org.apache.pulsar.shade.org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.shade.org.apache.pulsar.common.util.ObjectMapperFactory;
import org.apache.pulsar.shade.org.apache.zookeeper.CreateMode;
import org.apache.pulsar.shade.org.apache.zookeeper.KeeperException;
import org.apache.pulsar.shade.org.apache.zookeeper.ZooDefs;
import org.apache.pulsar.shade.org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClustersBase
extends AdminResource {
    private static final Logger log = LoggerFactory.getLogger(ClustersBase.class);

    @GET
    @ApiOperation(value="Get the list of all the Pulsar clusters.", response=String.class, responseContainer="Set")
    @ApiResponses(value={@ApiResponse(code=200, message="Return a list of clusters."), @ApiResponse(code=500, message="Internal server error.")})
    public Set<String> getClusters() throws Exception {
        try {
            Set<String> clusters = this.clustersListCache().get().stream().filter(cluster -> !"global".equals(cluster)).collect(Collectors.toSet());
            return clusters;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters list", (Object)this.clientAppId(), (Object)e);
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}")
    @ApiOperation(value="Get the configuration for the specified cluster.", response=ClusterData.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=200, message="Return the cluster data.", response=ClusterData.class), @ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public ClusterData getCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        try {
            return this.clustersCache().get(ClustersBase.path("clusters", cluster)).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "Cluster does not exist"));
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster {}", new Object[]{this.clientAppId(), cluster, e});
            if (e instanceof RestException) {
                throw (RestException)e;
            }
            throw new RestException(e);
        }
    }

    @PUT
    @Path(value="/{cluster}")
    @ApiOperation(value="Create a new cluster.", notes="This operation requires Pulsar superuser privileges, and the name cannot contain the '/' characters.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been created."), @ApiResponse(code=403, message="You don't have admin permission to create the cluster."), @ApiResponse(code=409, message="Cluster already exists."), @ApiResponse(code=412, message="Cluster name is not valid."), @ApiResponse(code=500, message="Internal server error.")})
    public void createCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The cluster data", required=true, examples=@Example(value={@ExampleProperty(mediaType="application/json", value="{\n   'serviceUrl': 'http://pulsar.example.com:8080',\n   'brokerServiceUrl': 'pulsar://pulsar.example.com:6651',\n}")})) ClusterData clusterData) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        try {
            NamedEntity.checkName(cluster);
            this.zkCreate(ClustersBase.path("clusters", cluster), ClustersBase.jsonMapper().writeValueAsBytes(clusterData));
            log.info("[{}] Created cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (KeeperException.NodeExistsException e) {
            log.warn("[{}] Failed to create already existing cluster {}", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.CONFLICT, "Cluster already exists");
        }
        catch (IllegalArgumentException e) {
            log.warn("[{}] Failed to create cluster with invalid name {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Cluster name is not valid");
        }
        catch (Exception e) {
            log.error("[{}] Failed to create cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}")
    @ApiOperation(value="Update the configuration for a cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been updated."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void updateCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The cluster data", required=true, examples=@Example(value={@ExampleProperty(mediaType="application/json", value="{\n   'serviceUrl': 'http://pulsar.example.com:8080',\n   'brokerServiceUrl': 'pulsar://pulsar.example.com:6651'\n}")})) ClusterData clusterData) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        try {
            String clusterPath = ClustersBase.path("clusters", cluster);
            Stat nodeStat = new Stat();
            byte[] content = this.globalZk().getData(clusterPath, null, nodeStat);
            ClusterData currentClusterData = null;
            if (content.length > 0) {
                currentClusterData = ClustersBase.jsonMapper().readValue(content, ClusterData.class);
                currentClusterData.update(clusterData);
            } else {
                currentClusterData = clusterData;
            }
            this.globalZk().setData(clusterPath, ClustersBase.jsonMapper().writeValueAsBytes(currentClusterData), nodeStat.getVersion());
            this.globalZkCache().invalidate(clusterPath);
            log.info("[{}] Updated cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (KeeperException.NoNodeException e) {
            log.warn("[{}] Failed to update cluster {}: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}/peers")
    @ApiOperation(value="Update peer-cluster-list for a cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been updated."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=412, message="Peer cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void setPeerClusterNames(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The list of peer cluster names", required=true, examples=@Example(value={@ExampleProperty(mediaType="application/json", value="[\n   'cluster-a',\n   'cluster-b'\n]")})) LinkedHashSet<String> peerClusterNames) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        if (peerClusterNames != null && !peerClusterNames.isEmpty()) {
            for (String peerCluster : peerClusterNames) {
                try {
                    if (cluster.equalsIgnoreCase(peerCluster)) {
                        throw new RestException(Response.Status.PRECONDITION_FAILED, cluster + " itself can't be part of peer-list");
                    }
                    this.clustersCache().get(ClustersBase.path("clusters", peerCluster)).orElseThrow(() -> new RestException(Response.Status.PRECONDITION_FAILED, "Peer cluster " + peerCluster + " does not exist"));
                }
                catch (RestException e) {
                    log.warn("[{}] Peer cluster doesn't exist from {}, {}", new Object[]{this.clientAppId(), peerClusterNames, e.getMessage()});
                    throw e;
                }
                catch (Exception e) {
                    log.warn("[{}] Failed to validate peer-cluster list {}, {}", new Object[]{this.clientAppId(), peerClusterNames, e.getMessage()});
                    throw new RestException(e);
                }
            }
        }
        try {
            String clusterPath = ClustersBase.path("clusters", cluster);
            Stat nodeStat = new Stat();
            byte[] content = this.globalZk().getData(clusterPath, null, nodeStat);
            ClusterData currentClusterData = ClustersBase.jsonMapper().readValue(content, ClusterData.class);
            currentClusterData.setPeerClusterNames(peerClusterNames);
            this.globalZk().setData(clusterPath, ClustersBase.jsonMapper().writeValueAsBytes(currentClusterData), nodeStat.getVersion());
            this.globalZkCache().invalidate(clusterPath);
            log.info("[{}] Successfully added peer-cluster {} for {}", new Object[]{this.clientAppId(), peerClusterNames, cluster});
        }
        catch (KeeperException.NoNodeException e) {
            log.warn("[{}] Failed to update cluster {}: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/peers")
    @ApiOperation(value="Get the peer-cluster data for the specified cluster.", response=String.class, responseContainer="Set", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public Set<String> getPeerCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        try {
            String clusterPath = ClustersBase.path("clusters", cluster);
            byte[] content = this.globalZk().getData(clusterPath, null, null);
            ClusterData clusterData = ClustersBase.jsonMapper().readValue(content, ClusterData.class);
            return clusterData.getPeerClusterNames();
        }
        catch (KeeperException.NoNodeException e) {
            log.warn("[{}] Failed to get cluster {}: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @DELETE
    @Path(value="/{cluster}")
    @ApiOperation(value="Delete an existing cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=204, message="Cluster has been deleted."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=412, message="Cluster is not empty."), @ApiResponse(code=500, message="Internal server error.")})
    public void deleteCluster(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        this.validateSuperUserAccess();
        this.validatePoliciesReadOnlyAccess();
        boolean isClusterUsed = false;
        try {
            for (String property : this.globalZk().getChildren(ClustersBase.path("policies"), false)) {
                if (this.globalZk().exists(ClustersBase.path("policies", property, cluster), false) == null) continue;
                if (this.globalZk().getChildren(ClustersBase.path("policies", property, cluster), false).isEmpty()) continue;
                isClusterUsed = true;
                break;
            }
            String path = ClustersBase.path("clusters", cluster, "namespaceIsolationPolicies");
            Optional<NamespaceIsolationPolicies> nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(path);
            if (nsIsolationPolicies.isPresent()) {
                if (nsIsolationPolicies.get().getPolicies().isEmpty()) {
                    this.globalZk().delete(path, -1);
                    this.namespaceIsolationPoliciesCache().invalidate(path);
                } else {
                    isClusterUsed = true;
                }
            }
        }
        catch (Exception e) {
            log.error("[{}] Failed to get cluster usage {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        if (isClusterUsed) {
            log.warn("[{}] Failed to delete cluster {} - Cluster not empty", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Cluster not empty");
        }
        try {
            String clusterPath = ClustersBase.path("clusters", cluster);
            this.deleteFailureDomain(clusterPath);
            this.globalZk().delete(clusterPath, -1);
            this.globalZkCache().invalidate(clusterPath);
            log.info("[{}] Deleted cluster {}", (Object)this.clientAppId(), (Object)cluster);
        }
        catch (KeeperException.NoNodeException e) {
            log.warn("[{}] Failed to delete cluster {} - Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "Cluster does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to delete cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    private void deleteFailureDomain(String clusterPath) {
        try {
            String failureDomain = ClustersBase.joinPath(clusterPath, "failureDomain");
            if (this.globalZk().exists(failureDomain, false) == null) {
                return;
            }
            for (String domain : this.globalZk().getChildren(failureDomain, false)) {
                String domainPath = ClustersBase.joinPath(failureDomain, domain);
                this.globalZk().delete(domainPath, -1);
            }
            this.globalZk().delete(failureDomain, -1);
            this.failureDomainCache().clear();
            this.failureDomainListCache().clear();
        }
        catch (Exception e) {
            log.warn("Failed to delete failure-domain under cluster {}", (Object)clusterPath);
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies")
    @ApiOperation(value="Get the namespace isolation policies assigned to the cluster.", response=NamespaceIsolationData.class, responseContainer="Map", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public Map<String, NamespaceIsolationData> getNamespaceIsolationPolicies(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) throws Exception {
        this.validateSuperUserAccess();
        if (!this.clustersCache().get(ClustersBase.path("clusters", cluster)).isPresent()) {
            throw new RestException(Response.Status.NOT_FOUND, "Cluster " + cluster + " does not exist.");
        }
        try {
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(ClustersBase.path("clusters", cluster, "namespaceIsolationPolicies")).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist"));
            return nsIsolationPolicies.getPolicies();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters/{}/namespaceIsolationPolicies", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Get the single namespace isolation policy assigned to the cluster.", response=NamespaceIsolationData.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Policy doesn't exist."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public NamespaceIsolationData getNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The name of the namespace isolation policy", required=true) @PathParam(value="policyName") String policyName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(ClustersBase.path("clusters", cluster, "namespaceIsolationPolicies")).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist"));
            if (!nsIsolationPolicies.getPolicies().containsKey(policyName)) {
                log.info("[{}] Cannot find NamespaceIsolationPolicy {} for cluster {}", new Object[]{this.clientAppId(), policyName, cluster});
                throw new RestException(Response.Status.NOT_FOUND, "Cannot find NamespaceIsolationPolicy " + policyName + " for cluster " + cluster);
            }
            return nsIsolationPolicies.getPolicies().get(policyName);
        }
        catch (RestException re) {
            throw re;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get clusters/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies/brokers")
    @ApiOperation(value="Get list of brokers with namespace-isolation policies attached to them.", response=BrokerNamespaceIsolationData.class, responseContainer="set", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Namespace-isolation policies not found."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public List<BrokerNamespaceIsolationData> getBrokersWithNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) {
        Map<String, NamespaceIsolationData> nsPolicies;
        Set<String> availableBrokers;
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        String nsIsolationPoliciesPath = AdminResource.path("clusters", cluster, "namespaceIsolationPolicies");
        try {
            availableBrokers = this.pulsar().getLoadManager().get().getAvailableBrokers();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get list of brokers in cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        try {
            Optional<NamespaceIsolationPolicies> nsPoliciesResult = this.namespaceIsolationPoliciesCache().get(nsIsolationPoliciesPath);
            if (!nsPoliciesResult.isPresent()) {
                throw new RestException(Response.Status.NOT_FOUND, "namespace-isolation policies not found for " + cluster);
            }
            nsPolicies = nsPoliciesResult.get().getPolicies();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get namespace isolation-policies {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        return availableBrokers.stream().map(broker -> {
            BrokerNamespaceIsolationData brokerIsolationData = new BrokerNamespaceIsolationData();
            brokerIsolationData.brokerName = broker;
            if (nsPolicies != null) {
                nsPolicies.forEach((name, policyData) -> {
                    NamespaceIsolationPolicyImpl nsPolicyImpl = new NamespaceIsolationPolicyImpl((NamespaceIsolationData)policyData);
                    if (nsPolicyImpl.isPrimaryBroker((String)broker) || nsPolicyImpl.isSecondaryBroker((String)broker)) {
                        if (brokerIsolationData.namespaceRegex == null) {
                            brokerIsolationData.namespaceRegex = Lists.newArrayList();
                        }
                        brokerIsolationData.namespaceRegex.addAll(policyData.namespaces);
                        if (nsPolicyImpl.isPrimaryBroker((String)broker)) {
                            brokerIsolationData.isPrimary = true;
                        }
                    }
                });
            }
            return brokerIsolationData;
        }).collect(Collectors.toList());
    }

    @GET
    @Path(value="/{cluster}/namespaceIsolationPolicies/brokers/{broker}")
    @ApiOperation(value="Get a broker with namespace-isolation policies attached to it.", response=BrokerNamespaceIsolationData.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Namespace-isolation policies/ Broker not found."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public BrokerNamespaceIsolationData getBrokerWithNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The broker name (<broker-hostname>:<web-service-port>)", required=true, example="broker1:8080") @PathParam(value="broker") String broker) {
        Map<String, NamespaceIsolationData> nsPolicies;
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        String nsIsolationPoliciesPath = AdminResource.path("clusters", cluster, "namespaceIsolationPolicies");
        try {
            Optional<NamespaceIsolationPolicies> nsPoliciesResult = this.namespaceIsolationPoliciesCache().get(nsIsolationPoliciesPath);
            if (!nsPoliciesResult.isPresent()) {
                throw new RestException(Response.Status.NOT_FOUND, "namespace-isolation policies not found for " + cluster);
            }
            nsPolicies = nsPoliciesResult.get().getPolicies();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get namespace isolation-policies {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        BrokerNamespaceIsolationData brokerIsolationData = new BrokerNamespaceIsolationData();
        brokerIsolationData.brokerName = broker;
        if (nsPolicies != null) {
            nsPolicies.forEach((name, policyData) -> {
                NamespaceIsolationPolicyImpl nsPolicyImpl = new NamespaceIsolationPolicyImpl((NamespaceIsolationData)policyData);
                boolean isPrimary = nsPolicyImpl.isPrimaryBroker(broker);
                if (isPrimary || nsPolicyImpl.isSecondaryBroker(broker)) {
                    if (brokerIsolationData.namespaceRegex == null) {
                        brokerIsolationData.namespaceRegex = Lists.newArrayList();
                    }
                    brokerIsolationData.namespaceRegex.addAll(policyData.namespaces);
                    brokerIsolationData.isPrimary = isPrimary;
                    brokerIsolationData.policyName = name;
                }
            });
        }
        return brokerIsolationData;
    }

    @POST
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Set namespace isolation policy.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=400, message="Namespace isolation policy data is invalid."), @ApiResponse(code=403, message="Don't have admin permission or policies are read-only."), @ApiResponse(code=404, message="Namespace isolation policy doesn't exist."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void setNamespaceIsolationPolicy(@Suspended AsyncResponse asyncResponse, @ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The namespace isolation policy name", required=true) @PathParam(value="policyName") String policyName, @ApiParam(value="The namespace isolation policy data", required=true) NamespaceIsolationData policyData) {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validatePoliciesReadOnlyAccess();
        String jsonInput = null;
        try {
            policyData.validate();
            jsonInput = ObjectMapperFactory.create().writeValueAsString(policyData);
            String nsIsolationPolicyPath = ClustersBase.path("clusters", cluster, "namespaceIsolationPolicies");
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(nsIsolationPolicyPath).orElseGet(() -> {
                try {
                    this.createZnodeIfNotExist(nsIsolationPolicyPath, Optional.of(Collections.emptyMap()));
                    return new NamespaceIsolationPolicies();
                }
                catch (InterruptedException | KeeperException e) {
                    throw new RestException(e);
                }
            });
            nsIsolationPolicies.setPolicy(policyName, policyData);
            this.globalZk().setData(nsIsolationPolicyPath, ClustersBase.jsonMapper().writeValueAsBytes(nsIsolationPolicies.getPolicies()), -1);
            this.namespaceIsolationPoliciesCache().invalidate(nsIsolationPolicyPath);
            if (!this.pulsar().getConfiguration().isEnableNamespaceIsolationUpdateOnTime()) {
                asyncResponse.resume(Response.noContent().build());
                return;
            }
            this.filterAndUnloadMatchedNameSpaces(asyncResponse, policyData);
        }
        catch (IllegalArgumentException iae) {
            log.info("[{}] Failed to update clusters/{}/namespaceIsolationPolicies/{}. Input data is invalid", new Object[]{this.clientAppId(), cluster, policyName, iae});
            asyncResponse.resume(new RestException(Response.Status.BAD_REQUEST, "Invalid format of input policy data. policy: " + policyName + "; data: " + jsonInput));
        }
        catch (KeeperException.NoNodeException nne) {
            log.warn("[{}] Failed to update clusters/{}/namespaceIsolationPolicies: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            asyncResponse.resume(new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist"));
        }
        catch (Exception e) {
            log.error("[{}] Failed to update clusters/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, policyName, e});
            asyncResponse.resume(new RestException(e));
        }
    }

    private void filterAndUnloadMatchedNameSpaces(AsyncResponse asyncResponse, NamespaceIsolationData policyData) throws Exception {
        Namespaces namespaces = this.pulsar().getAdminClient().namespaces();
        ArrayList nssToUnload = Lists.newArrayList();
        this.pulsar().getAdminClient().tenants().getTenantsAsync().whenComplete((tenants, ex) -> {
            if (ex != null) {
                log.error("[{}] Failed to get tenants when setNamespaceIsolationPolicy.", (Object)this.clientAppId(), ex);
                return;
            }
            AtomicInteger tenantsNumber = new AtomicInteger(tenants.size());
            tenants.forEach(tenant -> namespaces.getNamespacesAsync((String)tenant).whenComplete((nss, e) -> {
                int leftTenantsToHandle = tenantsNumber.decrementAndGet();
                if (ex != null) {
                    log.error("[{}] Failed to get namespaces for tenant {} when setNamespaceIsolationPolicy.", new Object[]{this.clientAppId(), tenant, ex});
                    if (leftTenantsToHandle == 0) {
                        this.unloadMatchedNamespacesList(asyncResponse, nssToUnload, namespaces);
                    }
                    return;
                }
                AtomicInteger nssNumber = new AtomicInteger(nss.size());
                nss.forEach(namespaceName -> {
                    int leftNssToHandle = nssNumber.decrementAndGet();
                    if (policyData.namespaces.stream().anyMatch(nsnameRegex -> namespaceName.matches((String)nsnameRegex))) {
                        nssToUnload.add(namespaceName);
                    }
                    if (leftNssToHandle == 0 && leftTenantsToHandle == 0) {
                        this.unloadMatchedNamespacesList(asyncResponse, nssToUnload, namespaces);
                    }
                });
            }));
        });
    }

    private void unloadMatchedNamespacesList(AsyncResponse asyncResponse, List<String> nssToUnload, Namespaces namespaces) {
        if (nssToUnload.size() == 0) {
            asyncResponse.resume(Response.noContent().build());
            return;
        }
        List futures = nssToUnload.stream().map(namespaceName -> namespaces.unloadAsync((String)namespaceName)).collect(Collectors.toList());
        FutureUtil.waitForAll(futures).whenComplete((result, exception) -> {
            if (exception != null) {
                log.error("[{}] Failed to unload namespace while setNamespaceIsolationPolicy.", (Object)this.clientAppId(), exception);
                asyncResponse.resume(new RestException((Throwable)exception));
                return;
            }
            try {
                this.pulsar().getLoadManager().get().writeLoadReportOnZookeeper(true);
            }
            catch (Exception e) {
                log.warn("[{}] Failed to writeLoadReportOnZookeeper.", (Object)this.clientAppId(), (Object)e);
            }
            asyncResponse.resume(Response.noContent().build());
        });
    }

    private boolean createZnodeIfNotExist(String path, Optional<Object> value) throws KeeperException, InterruptedException {
        if (this.globalZk().exists(path, false) == null) {
            try {
                ZkUtils.createFullPathOptimistic(this.globalZk(), path, value.isPresent() ? ClustersBase.jsonMapper().writeValueAsBytes(value.get()) : null, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
                return true;
            }
            catch (KeeperException.NodeExistsException nee) {
                if (log.isDebugEnabled()) {
                    log.debug("Other broker preempted the full path [{}] already. Continue...", (Object)path);
                }
            }
            catch (JsonGenerationException jsonGenerationException) {
            }
            catch (JsonMappingException jsonMappingException) {
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return false;
    }

    @DELETE
    @Path(value="/{cluster}/namespaceIsolationPolicies/{policyName}")
    @ApiOperation(value="Delete namespace isolation policy.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission or policies are read only."), @ApiResponse(code=404, message="Namespace isolation policy doesn't exist."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void deleteNamespaceIsolationPolicy(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The namespace isolation policy name", required=true) @PathParam(value="policyName") String policyName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validatePoliciesReadOnlyAccess();
        try {
            String nsIsolationPolicyPath = ClustersBase.path("clusters", cluster, "namespaceIsolationPolicies");
            NamespaceIsolationPolicies nsIsolationPolicies = this.namespaceIsolationPoliciesCache().get(nsIsolationPolicyPath).orElseGet(() -> {
                try {
                    this.createZnodeIfNotExist(nsIsolationPolicyPath, Optional.of(Collections.emptyMap()));
                    return new NamespaceIsolationPolicies();
                }
                catch (InterruptedException | KeeperException e) {
                    throw new RestException(e);
                }
            });
            nsIsolationPolicies.deletePolicy(policyName);
            this.globalZk().setData(nsIsolationPolicyPath, ClustersBase.jsonMapper().writeValueAsBytes(nsIsolationPolicies.getPolicies()), -1);
            this.namespaceIsolationPoliciesCache().invalidate(nsIsolationPolicyPath);
        }
        catch (KeeperException.NoNodeException nne) {
            log.warn("[{}] Failed to update brokers/{}/namespaceIsolationPolicies: Does not exist", (Object)this.clientAppId(), (Object)cluster);
            throw new RestException(Response.Status.NOT_FOUND, "NamespaceIsolationPolicies for cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update brokers/{}/namespaceIsolationPolicies/{}", new Object[]{this.clientAppId(), cluster, policyName, e});
            throw new RestException(e);
        }
    }

    @POST
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Set the failure domain of the cluster.", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission."), @ApiResponse(code=404, message="Failure domain doesn't exist."), @ApiResponse(code=409, message="Broker already exists in another domain."), @ApiResponse(code=412, message="Cluster doesn't exist."), @ApiResponse(code=500, message="Internal server error.")})
    public void setFailureDomain(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The failure domain name", required=true) @PathParam(value="domainName") String domainName, @ApiParam(value="The configuration data of a failure domain", required=true) FailureDomain domain) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        this.validateBrokerExistsInOtherDomain(cluster, domainName, domain);
        try {
            String domainPath = ClustersBase.joinPath(this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT, domainName);
            if (this.createZnodeIfNotExist(domainPath, Optional.ofNullable(domain))) {
                this.failureDomainListCache().clear();
            } else {
                this.globalZk().setData(domainPath, ClustersBase.jsonMapper().writeValueAsBytes(domain), -1);
                this.failureDomainCache().invalidate(domainPath);
            }
        }
        catch (KeeperException.NoNodeException nne) {
            log.warn("[{}] Failed to update domain {}. clusters {}  Does not exist", new Object[]{this.clientAppId(), cluster, domainName});
            throw new RestException(Response.Status.NOT_FOUND, "Domain " + domainName + " for cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to update clusters/{}/domainName/{}", new Object[]{this.clientAppId(), cluster, domainName, e});
            throw new RestException(e);
        }
    }

    @GET
    @Path(value="/{cluster}/failureDomains")
    @ApiOperation(value="Get the cluster failure domains.", response=FailureDomain.class, responseContainer="Map", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=500, message="Internal server error")})
    public Map<String, FailureDomain> getFailureDomains(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster) throws Exception {
        this.validateSuperUserAccess();
        HashMap<String, FailureDomain> domains = Maps.newHashMap();
        try {
            String failureDomainRootPath = this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT;
            for (String domainName : this.failureDomainListCache().get()) {
                try {
                    Optional<FailureDomain> domain = this.failureDomainCache().get(ClustersBase.joinPath(failureDomainRootPath, domainName));
                    domain.ifPresent(failureDomain -> domains.put(domainName, (FailureDomain)failureDomain));
                }
                catch (Exception e) {
                    log.warn("Failed to get domain {}", (Object)domainName, (Object)e);
                }
            }
        }
        catch (KeeperException.NoNodeException e) {
            log.warn("[{}] Failure-domain is not configured for cluster {}", new Object[]{this.clientAppId(), cluster, e});
            return Collections.emptyMap();
        }
        catch (Exception e) {
            log.error("[{}] Failed to get failure-domains for cluster {}", new Object[]{this.clientAppId(), cluster, e});
            throw new RestException(e);
        }
        return domains;
    }

    @GET
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Get a domain in a cluster", response=FailureDomain.class, notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission"), @ApiResponse(code=404, message="FailureDomain doesn't exist"), @ApiResponse(code=412, message="Cluster doesn't exist"), @ApiResponse(code=500, message="Internal server error")})
    public FailureDomain getDomain(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The failure domain name", required=true) @PathParam(value="domainName") String domainName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            String failureDomainRootPath = this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT;
            return this.failureDomainCache().get(ClustersBase.joinPath(failureDomainRootPath, domainName)).orElseThrow(() -> new RestException(Response.Status.NOT_FOUND, "Domain " + domainName + " for cluster " + cluster + " does not exist"));
        }
        catch (RestException re) {
            throw re;
        }
        catch (Exception e) {
            log.error("[{}] Failed to get domain {} for cluster {}", new Object[]{this.clientAppId(), domainName, cluster, e});
            throw new RestException(e);
        }
    }

    @DELETE
    @Path(value="/{cluster}/failureDomains/{domainName}")
    @ApiOperation(value="Delete the failure domain of the cluster", notes="This operation requires Pulsar superuser privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="Don't have admin permission or policy is read only"), @ApiResponse(code=404, message="FailureDomain doesn't exist"), @ApiResponse(code=412, message="Cluster doesn't exist"), @ApiResponse(code=500, message="Internal server error")})
    public void deleteFailureDomain(@ApiParam(value="The cluster name", required=true) @PathParam(value="cluster") String cluster, @ApiParam(value="The failure domain name", required=true) @PathParam(value="domainName") String domainName) throws Exception {
        this.validateSuperUserAccess();
        this.validateClusterExists(cluster);
        try {
            String domainPath = ClustersBase.joinPath(this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT, domainName);
            this.globalZk().delete(domainPath, -1);
            this.failureDomainCache().invalidate(domainPath);
            this.failureDomainListCache().clear();
        }
        catch (KeeperException.NoNodeException nne) {
            log.warn("[{}] Domain {} does not exist in {}", new Object[]{this.clientAppId(), domainName, cluster});
            throw new RestException(Response.Status.NOT_FOUND, "Domain-name " + domainName + " or cluster " + cluster + " does not exist");
        }
        catch (Exception e) {
            log.error("[{}] Failed to delete domain {} in cluster {}", new Object[]{this.clientAppId(), domainName, cluster, e});
            throw new RestException(e);
        }
    }

    private void validateBrokerExistsInOtherDomain(String cluster, String inputDomainName, FailureDomain inputDomain) {
        if (inputDomain != null && inputDomain.brokers != null) {
            try {
                String failureDomainRootPath = this.pulsar().getConfigurationCache().CLUSTER_FAILURE_DOMAIN_ROOT;
                for (String domainName : this.failureDomainListCache().get()) {
                    if (inputDomainName.equals(domainName)) continue;
                    try {
                        Optional<FailureDomain> domain = this.failureDomainCache().get(ClustersBase.joinPath(failureDomainRootPath, domainName));
                        if (!domain.isPresent() || domain.get().brokers == null) continue;
                        List duplicateBrokers = ((Stream)domain.get().brokers.stream().parallel()).filter(inputDomain.brokers::contains).collect(Collectors.toList());
                        if (duplicateBrokers.isEmpty()) continue;
                        throw new RestException(Response.Status.CONFLICT, duplicateBrokers + " already exists in " + domainName);
                    }
                    catch (Exception e) {
                        if (e instanceof RestException) {
                            throw e;
                        }
                        log.warn("Failed to get domain {}", (Object)domainName, (Object)e);
                    }
                }
            }
            catch (KeeperException.NoNodeException e) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Domain is not configured for cluster", (Object)this.clientAppId(), (Object)e);
                }
            }
            catch (Exception e) {
                log.error("[{}] Failed to get domains for cluster {}", (Object)this.clientAppId(), (Object)e);
                throw new RestException(e);
            }
        }
    }
}

