/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.persistent;

import com.google.common.base.MoreObjects;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.pulsar.broker.service.BrokerService;
import org.apache.pulsar.broker.service.persistent.DispatchRateLimiter;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.common.policies.data.Policies;
import org.apache.pulsar.common.policies.data.SubscribeRate;
import org.apache.pulsar.common.util.RateLimiter;
import org.apache.pulsar.common.util.Runnables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SubscribeRateLimiter {
    private final String topicName;
    private final BrokerService brokerService;
    private ConcurrentHashMap<ConsumerIdentifier, RateLimiter> subscribeRateLimiter;
    private final ScheduledExecutorService executorService;
    private ScheduledFuture<?> resetTask;
    private SubscribeRate subscribeRate;
    private static final Logger log = LoggerFactory.getLogger(SubscribeRateLimiter.class);

    public SubscribeRateLimiter(PersistentTopic topic) {
        this.topicName = topic.getName();
        this.brokerService = topic.getBrokerService();
        this.subscribeRateLimiter = new ConcurrentHashMap();
        this.executorService = this.brokerService.pulsar().getExecutor();
        this.subscribeRate = topic.getSubscribeRate();
        if (SubscribeRateLimiter.isSubscribeRateEnabled(this.subscribeRate)) {
            this.resetTask = this.createTask();
            log.info("[{}] configured subscribe-dispatch rate at broker {}", (Object)this.topicName, (Object)this.subscribeRate);
        }
    }

    public long getAvailableSubscribeRateLimit(ConsumerIdentifier consumerIdentifier) {
        return this.subscribeRateLimiter.get(consumerIdentifier) == null ? -1L : this.subscribeRateLimiter.get(consumerIdentifier).getAvailablePermits();
    }

    public synchronized boolean tryAcquire(ConsumerIdentifier consumerIdentifier) {
        this.addSubscribeLimiterIfAbsent(consumerIdentifier);
        return this.subscribeRateLimiter.get(consumerIdentifier) == null || this.subscribeRateLimiter.get(consumerIdentifier).tryAcquire();
    }

    public boolean subscribeAvailable(ConsumerIdentifier consumerIdentifier) {
        return this.subscribeRateLimiter.get(consumerIdentifier) == null || this.subscribeRateLimiter.get(consumerIdentifier).getAvailablePermits() > 0L;
    }

    private synchronized void addSubscribeLimiterIfAbsent(ConsumerIdentifier consumerIdentifier) {
        if (this.subscribeRateLimiter.get(consumerIdentifier) != null || !SubscribeRateLimiter.isSubscribeRateEnabled(this.subscribeRate)) {
            return;
        }
        this.updateSubscribeRate(consumerIdentifier, this.subscribeRate);
    }

    private synchronized void removeSubscribeLimiter(ConsumerIdentifier consumerIdentifier) {
        if (this.subscribeRateLimiter.get(consumerIdentifier) != null) {
            this.subscribeRateLimiter.get(consumerIdentifier).close();
            this.subscribeRateLimiter.remove(consumerIdentifier);
        }
    }

    private synchronized void updateSubscribeRate(ConsumerIdentifier consumerIdentifier, SubscribeRate subscribeRate) {
        long ratePerConsumer = subscribeRate.subscribeThrottlingRatePerConsumer;
        long ratePeriod = subscribeRate.ratePeriodInSecond;
        if (ratePerConsumer > 0L) {
            if (this.subscribeRateLimiter.get(consumerIdentifier) == null) {
                this.subscribeRateLimiter.put(consumerIdentifier, RateLimiter.builder().scheduledExecutorService(this.brokerService.pulsar().getExecutor()).permits(ratePerConsumer).rateTime(ratePeriod).timeUnit(TimeUnit.SECONDS).build());
            } else {
                this.subscribeRateLimiter.get(consumerIdentifier).setRate(ratePerConsumer, ratePeriod, TimeUnit.SECONDS, null);
            }
        } else {
            this.removeSubscribeLimiter(consumerIdentifier);
        }
    }

    public void onSubscribeRateUpdate(SubscribeRate subscribeRate) {
        this.subscribeRate = subscribeRate;
        this.stopResetTask();
        for (ConsumerIdentifier consumerIdentifier : this.subscribeRateLimiter.keySet()) {
            if (!SubscribeRateLimiter.isSubscribeRateEnabled(this.subscribeRate)) {
                this.removeSubscribeLimiter(consumerIdentifier);
                continue;
            }
            this.updateSubscribeRate(consumerIdentifier, subscribeRate);
        }
        if (SubscribeRateLimiter.isSubscribeRateEnabled(this.subscribeRate)) {
            this.resetTask = this.createTask();
            log.info("[{}] configured subscribe-dispatch rate at broker {}", (Object)this.topicName, (Object)subscribeRate);
        }
    }

    public SubscribeRate getPoliciesSubscribeRate() {
        return SubscribeRateLimiter.getPoliciesSubscribeRate(this.brokerService, this.topicName);
    }

    public static SubscribeRate getPoliciesSubscribeRate(BrokerService brokerService, String topicName) {
        String cluster = brokerService.pulsar().getConfiguration().getClusterName();
        Optional<Policies> policies = DispatchRateLimiter.getPolicies(brokerService, topicName);
        return SubscribeRateLimiter.getPoliciesSubscribeRate(cluster, policies, topicName);
    }

    public static SubscribeRate getPoliciesSubscribeRate(String cluster, Optional<Policies> policies, String topicName) {
        return policies.map(p -> {
            if (p.clusterSubscribeRate != null) {
                SubscribeRate subscribeRate = (SubscribeRate)p.clusterSubscribeRate.get(cluster);
                return SubscribeRateLimiter.isSubscribeRateEnabled(subscribeRate) ? subscribeRate : null;
            }
            return null;
        }).orElse(null);
    }

    public long getSubscribeRatePerConsumer(ConsumerIdentifier consumerIdentifier) {
        return this.subscribeRateLimiter.get(consumerIdentifier) != null ? this.subscribeRateLimiter.get(consumerIdentifier).getRate() : -1L;
    }

    public static boolean isSubscribeRateEnabled(SubscribeRate subscribeRate) {
        return subscribeRate.subscribeThrottlingRatePerConsumer > 0 && subscribeRate.ratePeriodInSecond > 0;
    }

    public void close() {
        this.closeAndClearRateLimiters();
        this.stopResetTask();
    }

    private ScheduledFuture<?> createTask() {
        return this.executorService.scheduleAtFixedRate(Runnables.catchingAndLoggingThrowables(this::closeAndClearRateLimiters), this.subscribeRate.ratePeriodInSecond, this.subscribeRate.ratePeriodInSecond, TimeUnit.SECONDS);
    }

    private void stopResetTask() {
        if (this.resetTask != null) {
            this.resetTask.cancel(false);
        }
    }

    private synchronized void closeAndClearRateLimiters() {
        this.subscribeRateLimiter.values().forEach(rateLimiter -> {
            if (rateLimiter != null) {
                rateLimiter.close();
            }
        });
        this.subscribeRateLimiter.clear();
    }

    public SubscribeRate getSubscribeRate() {
        return this.subscribeRate;
    }

    public static class ConsumerIdentifier {
        private String host;
        private String consumerName;
        private long consumerId;

        public ConsumerIdentifier(String host, String consumerName, long consumerId) {
            this.host = host;
            this.consumerName = consumerName;
            this.consumerId = consumerId;
        }

        public int hashCode() {
            return Objects.hash(this.host, this.consumerName, this.consumerId);
        }

        public boolean equals(Object obj) {
            if (obj instanceof ConsumerIdentifier) {
                ConsumerIdentifier consumer = (ConsumerIdentifier)obj;
                return Objects.equals(this.host, consumer.host) && Objects.equals(this.consumerName, consumer.consumerName) && Objects.equals(this.consumerId, consumer.consumerId);
            }
            return false;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("host", (Object)this.host).add("consumerName", (Object)this.consumerName).add("consumerId", this.consumerId).toString();
        }
    }
}

