/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.execution.buffer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import io.prestosql.memory.context.LocalMemoryContext;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
class OutputBufferMemoryManager {
    private final long maxBufferedBytes;
    private final AtomicLong bufferedBytes = new AtomicLong();
    private final AtomicLong peakMemoryUsage = new AtomicLong();
    @GuardedBy(value="this")
    private boolean closed;
    @GuardedBy(value="this")
    private SettableFuture<?> bufferBlockedFuture;
    @GuardedBy(value="this")
    private ListenableFuture<?> blockedOnMemory = Futures.immediateFuture(null);
    private final AtomicBoolean blockOnFull = new AtomicBoolean(true);
    private final Supplier<LocalMemoryContext> systemMemoryContextSupplier;
    private final Executor notificationExecutor;

    public OutputBufferMemoryManager(long maxBufferedBytes, Supplier<LocalMemoryContext> systemMemoryContextSupplier, Executor notificationExecutor) {
        Objects.requireNonNull(systemMemoryContextSupplier, "systemMemoryContextSupplier is null");
        Preconditions.checkArgument((maxBufferedBytes > 0L ? 1 : 0) != 0, (Object)"maxBufferedBytes must be > 0");
        this.maxBufferedBytes = maxBufferedBytes;
        this.systemMemoryContextSupplier = Suppliers.memoize(systemMemoryContextSupplier::get);
        this.notificationExecutor = Objects.requireNonNull(notificationExecutor, "notificationExecutor is null");
        this.bufferBlockedFuture = SettableFuture.create();
        this.bufferBlockedFuture.set(null);
    }

    public synchronized void updateMemoryUsage(long bytesAdded) {
        Optional<LocalMemoryContext> systemMemoryContext = this.getSystemMemoryContext();
        if (this.closed || !systemMemoryContext.isPresent()) {
            return;
        }
        long currentBufferedBytes = this.bufferedBytes.addAndGet(bytesAdded);
        this.peakMemoryUsage.accumulateAndGet(currentBufferedBytes, Math::max);
        this.blockedOnMemory = systemMemoryContext.get().setBytes(currentBufferedBytes);
        if (!(this.isBufferFull() || this.isBlockedOnMemory() || this.bufferBlockedFuture.isDone())) {
            SettableFuture<?> future = this.bufferBlockedFuture;
            this.notificationExecutor.execute(() -> future.set(null));
            return;
        }
        this.blockedOnMemory.addListener(this::onMemoryAvailable, this.notificationExecutor);
    }

    public synchronized ListenableFuture<?> getBufferBlockedFuture() {
        if ((this.isBufferFull() || this.isBlockedOnMemory()) && this.bufferBlockedFuture.isDone()) {
            this.bufferBlockedFuture = SettableFuture.create();
        }
        return this.bufferBlockedFuture;
    }

    public synchronized void setNoBlockOnFull() {
        this.blockOnFull.set(false);
        SettableFuture<?> future = this.bufferBlockedFuture;
        this.notificationExecutor.execute(() -> future.set(null));
    }

    public long getBufferedBytes() {
        return this.bufferedBytes.get();
    }

    public double getUtilization() {
        return (double)this.bufferedBytes.get() / (double)this.maxBufferedBytes;
    }

    public synchronized boolean isOverutilized() {
        return this.isBufferFull();
    }

    private synchronized boolean isBufferFull() {
        return this.bufferedBytes.get() > this.maxBufferedBytes && this.blockOnFull.get();
    }

    private synchronized boolean isBlockedOnMemory() {
        return !this.blockedOnMemory.isDone();
    }

    @VisibleForTesting
    synchronized void onMemoryAvailable() {
        if (this.bufferedBytes.get() > this.maxBufferedBytes) {
            return;
        }
        SettableFuture<?> future = this.bufferBlockedFuture;
        this.notificationExecutor.execute(() -> future.set(null));
    }

    public long getPeakMemoryUsage() {
        return this.peakMemoryUsage.get();
    }

    public synchronized void close() {
        this.updateMemoryUsage(-this.bufferedBytes.get());
        this.getSystemMemoryContext().ifPresent(LocalMemoryContext::close);
        this.closed = true;
    }

    private Optional<LocalMemoryContext> getSystemMemoryContext() {
        try {
            return Optional.of(this.systemMemoryContextSupplier.get());
        }
        catch (RuntimeException runtimeException) {
            return Optional.empty();
        }
    }
}

