/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.client.impl;

import io.netty.buffer.ByteBuf;
import java.net.InetSocketAddress;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.pulsar.client.api.PulsarClientException;
import org.apache.pulsar.client.api.SchemaSerializationException;
import org.apache.pulsar.client.impl.Backoff;
import org.apache.pulsar.client.impl.BackoffBuilder;
import org.apache.pulsar.client.impl.ClientCnx;
import org.apache.pulsar.client.impl.LookupService;
import org.apache.pulsar.client.impl.PulsarClientImpl;
import org.apache.pulsar.client.impl.PulsarServiceNameResolver;
import org.apache.pulsar.client.impl.ServiceNameResolver;
import org.apache.pulsar.common.api.proto.CommandGetTopicsOfNamespace;
import org.apache.pulsar.common.api.proto.CommandLookupTopicResponse;
import org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.common.partition.PartitionedTopicMetadata;
import org.apache.pulsar.common.protocol.Commands;
import org.apache.pulsar.common.protocol.schema.BytesSchemaVersion;
import org.apache.pulsar.common.schema.SchemaInfo;
import org.apache.pulsar.common.util.FutureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BinaryProtoLookupService
implements LookupService {
    private final PulsarClientImpl client;
    private final ServiceNameResolver serviceNameResolver;
    private final boolean useTls;
    private final ExecutorService executor;
    private final String listenerName;
    private final int maxLookupRedirects;
    private static final Logger log = LoggerFactory.getLogger(BinaryProtoLookupService.class);

    public BinaryProtoLookupService(PulsarClientImpl client, String serviceUrl, boolean useTls, ExecutorService executor) throws PulsarClientException {
        this(client, serviceUrl, null, useTls, executor);
    }

    public BinaryProtoLookupService(PulsarClientImpl client, String serviceUrl, String listenerName, boolean useTls, ExecutorService executor) throws PulsarClientException {
        this.client = client;
        this.useTls = useTls;
        this.executor = executor;
        this.maxLookupRedirects = client.getConfiguration().getMaxLookupRedirects();
        this.serviceNameResolver = new PulsarServiceNameResolver();
        this.listenerName = listenerName;
        this.updateServiceUrl(serviceUrl);
    }

    @Override
    public void updateServiceUrl(String serviceUrl) throws PulsarClientException {
        this.serviceNameResolver.updateServiceUrl(serviceUrl);
    }

    @Override
    public CompletableFuture<Pair<InetSocketAddress, InetSocketAddress>> getBroker(TopicName topicName) {
        return this.findBroker(this.serviceNameResolver.resolveHost(), false, topicName, 0);
    }

    @Override
    public CompletableFuture<PartitionedTopicMetadata> getPartitionedTopicMetadata(TopicName topicName) {
        return this.getPartitionedTopicMetadata(this.serviceNameResolver.resolveHost(), topicName);
    }

    private CompletableFuture<Pair<InetSocketAddress, InetSocketAddress>> findBroker(InetSocketAddress socketAddress, boolean authoritative, TopicName topicName, int redirectCount) {
        CompletableFuture<Pair<InetSocketAddress, InetSocketAddress>> addressFuture = new CompletableFuture<Pair<InetSocketAddress, InetSocketAddress>>();
        if (this.maxLookupRedirects > 0 && redirectCount > this.maxLookupRedirects) {
            addressFuture.completeExceptionally((Throwable)new PulsarClientException.LookupException("Too many redirects: " + this.maxLookupRedirects));
            return addressFuture;
        }
        ((CompletableFuture)this.client.getCnxPool().getConnection(socketAddress).thenAccept(clientCnx -> {
            long requestId = this.client.newRequestId();
            ByteBuf request = Commands.newLookup((String)topicName.toString(), (String)this.listenerName, (boolean)authoritative, (long)requestId);
            clientCnx.newLookup(request, requestId).whenComplete((r, t) -> {
                if (t != null) {
                    log.warn("[{}] failed to send lookup request : {}", (Object)topicName.toString(), (Object)t.getMessage());
                    if (log.isDebugEnabled()) {
                        log.warn("[{}] Lookup response exception: {}", (Object)topicName.toString(), t);
                    }
                    addressFuture.completeExceptionally((Throwable)t);
                } else {
                    URI uri = null;
                    try {
                        if (this.useTls) {
                            uri = new URI(r.brokerUrlTls);
                        } else {
                            String serviceUrl = r.brokerUrl;
                            uri = new URI(serviceUrl);
                        }
                        InetSocketAddress responseBrokerAddress = InetSocketAddress.createUnresolved(uri.getHost(), uri.getPort());
                        if (r.redirect) {
                            ((CompletableFuture)this.findBroker(responseBrokerAddress, r.authoritative, topicName, redirectCount + 1).thenAccept(addressFuture::complete)).exceptionally(lookupException -> {
                                Throwable cause = FutureUtil.unwrapCompletionException((Throwable)lookupException);
                                if (redirectCount > 0) {
                                    if (log.isDebugEnabled()) {
                                        log.debug("[{}] lookup redirection failed ({}) : {}", new Object[]{topicName, redirectCount, cause.getMessage()});
                                    }
                                } else {
                                    log.warn("[{}] lookup failed : {}", new Object[]{topicName, cause.getMessage(), cause});
                                }
                                addressFuture.completeExceptionally(cause);
                                return null;
                            });
                        } else if (r.proxyThroughServiceUrl) {
                            addressFuture.complete(Pair.of((Object)responseBrokerAddress, (Object)socketAddress));
                        } else {
                            addressFuture.complete(Pair.of((Object)responseBrokerAddress, (Object)responseBrokerAddress));
                        }
                    }
                    catch (Exception parseUrlException) {
                        log.warn("[{}] invalid url {} : {}", new Object[]{topicName.toString(), uri, parseUrlException.getMessage(), parseUrlException});
                        addressFuture.completeExceptionally(parseUrlException);
                    }
                }
                this.client.getCnxPool().releaseConnection((ClientCnx)((Object)clientCnx));
            });
        })).exceptionally(connectionException -> {
            addressFuture.completeExceptionally(FutureUtil.unwrapCompletionException((Throwable)connectionException));
            return null;
        });
        return addressFuture;
    }

    private CompletableFuture<PartitionedTopicMetadata> getPartitionedTopicMetadata(InetSocketAddress socketAddress, TopicName topicName) {
        CompletableFuture<PartitionedTopicMetadata> partitionFuture = new CompletableFuture<PartitionedTopicMetadata>();
        ((CompletableFuture)this.client.getCnxPool().getConnection(socketAddress).thenAccept(clientCnx -> {
            long requestId = this.client.newRequestId();
            ByteBuf request = Commands.newPartitionMetadataRequest((String)topicName.toString(), (long)requestId);
            clientCnx.newLookup(request, requestId).whenComplete((r, t) -> {
                if (t != null) {
                    log.warn("[{}] failed to get Partitioned metadata : {}", new Object[]{topicName.toString(), t.getMessage(), t});
                    partitionFuture.completeExceptionally((Throwable)t);
                } else {
                    try {
                        partitionFuture.complete(new PartitionedTopicMetadata(r.partitions));
                    }
                    catch (Exception e) {
                        partitionFuture.completeExceptionally((Throwable)new PulsarClientException.LookupException(String.format("Failed to parse partition-response redirect=%s, topic=%s, partitions with %s, error message %s", r.redirect, topicName.toString(), r.partitions, e.getMessage())));
                    }
                }
                this.client.getCnxPool().releaseConnection((ClientCnx)((Object)clientCnx));
            });
        })).exceptionally(connectionException -> {
            partitionFuture.completeExceptionally(FutureUtil.unwrapCompletionException((Throwable)connectionException));
            return null;
        });
        return partitionFuture;
    }

    @Override
    public CompletableFuture<Optional<SchemaInfo>> getSchema(TopicName topicName) {
        return this.getSchema(topicName, null);
    }

    @Override
    public CompletableFuture<Optional<SchemaInfo>> getSchema(TopicName topicName, byte[] version) {
        CompletableFuture<Optional<SchemaInfo>> schemaFuture = new CompletableFuture<Optional<SchemaInfo>>();
        if (version != null && version.length == 0) {
            schemaFuture.completeExceptionally((Throwable)new SchemaSerializationException("Empty schema version"));
            return schemaFuture;
        }
        InetSocketAddress socketAddress = this.serviceNameResolver.resolveHost();
        ((CompletableFuture)this.client.getCnxPool().getConnection(socketAddress).thenAccept(clientCnx -> {
            long requestId = this.client.newRequestId();
            ByteBuf request = Commands.newGetSchema((long)requestId, (String)topicName.toString(), Optional.ofNullable(BytesSchemaVersion.of((byte[])version)));
            clientCnx.sendGetSchema(request, requestId).whenComplete((r, t) -> {
                if (t != null) {
                    log.warn("[{}] failed to get schema : {}", new Object[]{topicName.toString(), t.getMessage(), t});
                    schemaFuture.completeExceptionally((Throwable)t);
                } else {
                    schemaFuture.complete((Optional<SchemaInfo>)r);
                }
                this.client.getCnxPool().releaseConnection((ClientCnx)((Object)clientCnx));
            });
        })).exceptionally(ex -> {
            schemaFuture.completeExceptionally(FutureUtil.unwrapCompletionException((Throwable)ex));
            return null;
        });
        return schemaFuture;
    }

    @Override
    public String getServiceUrl() {
        return this.serviceNameResolver.getServiceUrl();
    }

    @Override
    public CompletableFuture<List<String>> getTopicsUnderNamespace(NamespaceName namespace, CommandGetTopicsOfNamespace.Mode mode) {
        CompletableFuture<List<String>> topicsFuture = new CompletableFuture<List<String>>();
        AtomicLong opTimeoutMs = new AtomicLong(this.client.getConfiguration().getOperationTimeoutMs());
        Backoff backoff = new BackoffBuilder().setInitialTime(100L, TimeUnit.MILLISECONDS).setMandatoryStop(opTimeoutMs.get() * 2L, TimeUnit.MILLISECONDS).setMax(1L, TimeUnit.MINUTES).create();
        this.getTopicsUnderNamespace(this.serviceNameResolver.resolveHost(), namespace, backoff, opTimeoutMs, topicsFuture, mode);
        return topicsFuture;
    }

    private void getTopicsUnderNamespace(InetSocketAddress socketAddress, NamespaceName namespace, Backoff backoff, AtomicLong remainingTime, CompletableFuture<List<String>> topicsFuture, CommandGetTopicsOfNamespace.Mode mode) {
        ((CompletableFuture)this.client.getCnxPool().getConnection(socketAddress).thenAccept(clientCnx -> {
            long requestId = this.client.newRequestId();
            ByteBuf request = Commands.newGetTopicsOfNamespaceRequest((String)namespace.toString(), (long)requestId, (CommandGetTopicsOfNamespace.Mode)mode);
            clientCnx.newGetTopicsOfNamespace(request, requestId).whenComplete((r, t) -> {
                if (t != null) {
                    topicsFuture.completeExceptionally((Throwable)t);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("[namespace: {}] Success get topics list in request: {}", (Object)namespace.toString(), (Object)requestId);
                    }
                    ArrayList result = new ArrayList();
                    r.forEach(topic -> {
                        String filtered = TopicName.get((String)topic).getPartitionedTopicName();
                        if (!result.contains(filtered)) {
                            result.add(filtered);
                        }
                    });
                    topicsFuture.complete(result);
                }
                this.client.getCnxPool().releaseConnection((ClientCnx)((Object)clientCnx));
            });
        })).exceptionally(e -> {
            long nextDelay = Math.min(backoff.next(), remainingTime.get());
            if (nextDelay <= 0L) {
                topicsFuture.completeExceptionally((Throwable)new PulsarClientException.TimeoutException(String.format("Could not get topics of namespace %s within configured timeout", namespace.toString())));
                return null;
            }
            ((ScheduledExecutorService)this.executor).schedule(() -> {
                log.warn("[namespace: {}] Could not get connection while getTopicsUnderNamespace -- Will try again in {} ms", (Object)namespace, (Object)nextDelay);
                remainingTime.addAndGet(-nextDelay);
                this.getTopicsUnderNamespace(socketAddress, namespace, backoff, remainingTime, topicsFuture, mode);
            }, nextDelay, TimeUnit.MILLISECONDS);
            return null;
        });
    }

    @Override
    public void close() throws Exception {
    }

    public static class LookupDataResult {
        public final String brokerUrl;
        public final String brokerUrlTls;
        public final int partitions;
        public final boolean authoritative;
        public final boolean proxyThroughServiceUrl;
        public final boolean redirect;

        public LookupDataResult(CommandLookupTopicResponse result) {
            this.brokerUrl = result.hasBrokerServiceUrl() ? result.getBrokerServiceUrl() : null;
            this.brokerUrlTls = result.hasBrokerServiceUrlTls() ? result.getBrokerServiceUrlTls() : null;
            this.authoritative = result.isAuthoritative();
            this.redirect = result.hasResponse() && result.getResponse() == CommandLookupTopicResponse.LookupType.Redirect;
            this.proxyThroughServiceUrl = result.isProxyThroughServiceUrl();
            this.partitions = -1;
        }

        public LookupDataResult(int partitions) {
            this.partitions = partitions;
            this.brokerUrl = null;
            this.brokerUrlTls = null;
            this.authoritative = false;
            this.proxyThroughServiceUrl = false;
            this.redirect = false;
        }
    }
}

