/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.execution.buffer;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import io.airlift.units.DataSize;
import io.prestosql.execution.buffer.BufferInfo;
import io.prestosql.execution.buffer.BufferResult;
import io.prestosql.execution.buffer.OutputBuffers;
import io.prestosql.execution.buffer.PageBufferInfo;
import io.prestosql.execution.buffer.SerializedPage;
import io.prestosql.execution.buffer.SerializedPageReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
class ClientBuffer {
    private final String taskInstanceId;
    private final OutputBuffers.OutputBufferId bufferId;
    private final AtomicLong rowsAdded = new AtomicLong();
    private final AtomicLong pagesAdded = new AtomicLong();
    private final AtomicLong bufferedBytes = new AtomicLong();
    @GuardedBy(value="this")
    private final AtomicLong currentSequenceId = new AtomicLong();
    @GuardedBy(value="this")
    private final LinkedList<SerializedPageReference> pages = new LinkedList();
    @GuardedBy(value="this")
    private boolean noMorePages;
    @GuardedBy(value="this")
    private final AtomicBoolean destroyed = new AtomicBoolean();
    @GuardedBy(value="this")
    private PendingRead pendingRead;

    public ClientBuffer(String taskInstanceId, OutputBuffers.OutputBufferId bufferId) {
        this.taskInstanceId = Objects.requireNonNull(taskInstanceId, "taskInstanceId is null");
        this.bufferId = Objects.requireNonNull(bufferId, "bufferId is null");
    }

    public BufferInfo getInfo() {
        boolean destroyed = this.destroyed.get();
        long sequenceId = this.currentSequenceId.get();
        int bufferedPages = destroyed ? 0 : Math.max(Math.toIntExact(this.pagesAdded.get() - sequenceId), 0);
        PageBufferInfo pageBufferInfo = new PageBufferInfo(this.bufferId.getId(), bufferedPages, this.bufferedBytes.get(), this.rowsAdded.get(), this.pagesAdded.get());
        return new BufferInfo(this.bufferId, destroyed, bufferedPages, sequenceId, pageBufferInfo);
    }

    public boolean isDestroyed() {
        boolean destroyed = this.destroyed.get();
        return destroyed;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void destroy() {
        PendingRead pendingRead;
        ImmutableList removedPages;
        ClientBuffer clientBuffer = this;
        synchronized (clientBuffer) {
            removedPages = ImmutableList.copyOf(this.pages);
            this.pages.clear();
            this.bufferedBytes.getAndSet(0L);
            this.noMorePages = true;
            this.destroyed.set(true);
            pendingRead = this.pendingRead;
            this.pendingRead = null;
        }
        removedPages.forEach(SerializedPageReference::dereferencePage);
        if (pendingRead != null) {
            pendingRead.completeResultFutureWithEmpty();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void enqueuePages(Collection<SerializedPageReference> pages) {
        PendingRead pendingRead;
        ClientBuffer clientBuffer = this;
        synchronized (clientBuffer) {
            if (this.noMorePages) {
                return;
            }
            this.addPages(pages);
            pendingRead = this.pendingRead;
            this.pendingRead = null;
        }
        if (pendingRead != null) {
            this.processRead(pendingRead);
        }
    }

    private synchronized void addPages(Collection<SerializedPageReference> pages) {
        pages.forEach(SerializedPageReference::addReference);
        this.pages.addAll(pages);
        long rowCount = pages.stream().mapToLong(SerializedPageReference::getPositionCount).sum();
        this.rowsAdded.addAndGet(rowCount);
        this.pagesAdded.addAndGet(pages.size());
        long bytesAdded = pages.stream().mapToLong(SerializedPageReference::getRetainedSizeInBytes).sum();
        this.bufferedBytes.addAndGet(bytesAdded);
    }

    public ListenableFuture<BufferResult> getPages(long sequenceId, DataSize maxSize) {
        return this.getPages(sequenceId, maxSize, Optional.empty());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public ListenableFuture<BufferResult> getPages(long sequenceId, DataSize maxSize, Optional<PagesSupplier> pagesSupplier) {
        SettableFuture<BufferResult> settableFuture;
        this.acknowledgePages(sequenceId);
        pagesSupplier.ifPresent(supplier -> this.loadPagesIfNecessary((PagesSupplier)supplier, maxSize));
        PendingRead oldPendingRead = null;
        try {
            ClientBuffer clientBuffer = this;
            // MONITORENTER : clientBuffer
            oldPendingRead = this.pendingRead;
            this.pendingRead = null;
            if (!this.pages.isEmpty() || this.noMorePages || sequenceId != this.currentSequenceId.get()) {
                ListenableFuture listenableFuture = Futures.immediateFuture((Object)this.processRead(sequenceId, maxSize));
                // MONITOREXIT : clientBuffer
                if (oldPendingRead == null) return listenableFuture;
                oldPendingRead.completeResultFutureWithEmpty();
                return listenableFuture;
            }
            this.pendingRead = new PendingRead(this.taskInstanceId, sequenceId, maxSize);
            settableFuture = this.pendingRead.getResultFuture();
        }
        catch (Throwable throwable) {
            if (oldPendingRead == null) throw throwable;
            oldPendingRead.completeResultFutureWithEmpty();
            throw throwable;
        }
        if (oldPendingRead == null) return settableFuture;
        oldPendingRead.completeResultFutureWithEmpty();
        return settableFuture;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setNoMorePages() {
        PendingRead pendingRead;
        ClientBuffer clientBuffer = this;
        synchronized (clientBuffer) {
            if (this.noMorePages) {
                return;
            }
            this.noMorePages = true;
            pendingRead = this.pendingRead;
            this.pendingRead = null;
        }
        if (pendingRead != null) {
            this.processRead(pendingRead);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void loadPagesIfNecessary(PagesSupplier pagesSupplier) {
        DataSize maxSize;
        Objects.requireNonNull(pagesSupplier, "pagesSupplier is null");
        ClientBuffer clientBuffer = this;
        synchronized (clientBuffer) {
            if (this.pendingRead == null) {
                return;
            }
            maxSize = this.pendingRead.getMaxSize();
        }
        boolean dataAddedOrNoMorePages = this.loadPagesIfNecessary(pagesSupplier, maxSize);
        if (dataAddedOrNoMorePages) {
            PendingRead pendingRead;
            ClientBuffer clientBuffer2 = this;
            synchronized (clientBuffer2) {
                pendingRead = this.pendingRead;
            }
            if (pendingRead != null) {
                this.processRead(pendingRead);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean loadPagesIfNecessary(PagesSupplier pagesSupplier, DataSize maxSize) {
        boolean dataAddedOrNoMorePages;
        List<SerializedPageReference> pageReferences;
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot load pages while holding a lock on this");
        ClientBuffer clientBuffer = this;
        synchronized (clientBuffer) {
            if (this.noMorePages) {
                return false;
            }
            if (!this.pages.isEmpty()) {
                return false;
            }
            pageReferences = pagesSupplier.getPages(maxSize);
            this.addPages(pageReferences);
            if (!pagesSupplier.mayHaveMorePages()) {
                this.noMorePages = true;
            }
            dataAddedOrNoMorePages = !pageReferences.isEmpty() || this.noMorePages;
        }
        pageReferences.forEach(SerializedPageReference::dereferencePage);
        return dataAddedOrNoMorePages;
    }

    private void processRead(PendingRead pendingRead) {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (Object)"Cannot process pending read while holding a lock on this");
        if (pendingRead.getResultFuture().isDone()) {
            return;
        }
        BufferResult bufferResult = this.processRead(pendingRead.getSequenceId(), pendingRead.getMaxSize());
        pendingRead.getResultFuture().set((Object)bufferResult);
    }

    private synchronized BufferResult processRead(long sequenceId, DataSize maxSize) {
        if (sequenceId < this.currentSequenceId.get()) {
            return BufferResult.emptyResults(this.taskInstanceId, sequenceId, false);
        }
        if (this.pages.isEmpty() && this.noMorePages) {
            return BufferResult.emptyResults(this.taskInstanceId, this.currentSequenceId.get(), true);
        }
        Verify.verify((sequenceId == this.currentSequenceId.get() ? 1 : 0) != 0, (String)"Invalid sequence id", (Object[])new Object[0]);
        long maxBytes = maxSize.toBytes();
        ArrayList<SerializedPage> result = new ArrayList<SerializedPage>();
        long bytes = 0L;
        for (SerializedPageReference page : this.pages) {
            if (!result.isEmpty() && (bytes += page.getRetainedSizeInBytes()) > maxBytes) break;
            result.add(page.getSerializedPage());
        }
        return new BufferResult(this.taskInstanceId, sequenceId, sequenceId + (long)result.size(), false, result);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void acknowledgePages(long sequenceId) {
        Preconditions.checkArgument((sequenceId >= 0L ? 1 : 0) != 0, (Object)"Invalid sequence id");
        ArrayList<SerializedPageReference> removedPages = new ArrayList<SerializedPageReference>();
        ClientBuffer clientBuffer = this;
        synchronized (clientBuffer) {
            if (this.destroyed.get()) {
                return;
            }
            long oldCurrentSequenceId = this.currentSequenceId.get();
            if (sequenceId < oldCurrentSequenceId) {
                return;
            }
            int pagesToRemove = Math.toIntExact(sequenceId - oldCurrentSequenceId);
            Preconditions.checkArgument((pagesToRemove <= this.pages.size() ? 1 : 0) != 0, (Object)"Invalid sequence id");
            long bytesRemoved = 0L;
            for (int i = 0; i < pagesToRemove; ++i) {
                SerializedPageReference removedPage = this.pages.removeFirst();
                removedPages.add(removedPage);
                bytesRemoved += removedPage.getRetainedSizeInBytes();
            }
            Verify.verify((boolean)this.currentSequenceId.compareAndSet(oldCurrentSequenceId, oldCurrentSequenceId + (long)pagesToRemove));
            Verify.verify((this.bufferedBytes.addAndGet(-bytesRemoved) >= 0L ? 1 : 0) != 0);
        }
        removedPages.forEach(SerializedPageReference::dereferencePage);
    }

    public String toString() {
        long sequenceId = this.currentSequenceId.get();
        boolean destroyed = this.destroyed.get();
        return MoreObjects.toStringHelper((Object)this).add("bufferId", (Object)this.bufferId).add("sequenceId", sequenceId).add("destroyed", destroyed).toString();
    }

    public static interface PagesSupplier {
        public List<SerializedPageReference> getPages(DataSize var1);

        public boolean mayHaveMorePages();
    }

    @Immutable
    private static class PendingRead {
        private final String taskInstanceId;
        private final long sequenceId;
        private final DataSize maxSize;
        private final SettableFuture<BufferResult> resultFuture = SettableFuture.create();

        private PendingRead(String taskInstanceId, long sequenceId, DataSize maxSize) {
            this.taskInstanceId = Objects.requireNonNull(taskInstanceId, "taskInstanceId is null");
            this.sequenceId = sequenceId;
            this.maxSize = maxSize;
        }

        public long getSequenceId() {
            return this.sequenceId;
        }

        public DataSize getMaxSize() {
            return this.maxSize;
        }

        public SettableFuture<BufferResult> getResultFuture() {
            return this.resultFuture;
        }

        public void completeResultFutureWithEmpty() {
            this.resultFuture.set((Object)BufferResult.emptyResults(this.taskInstanceId, this.sequenceId, false));
        }
    }
}

