/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.server.ui;

import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;
import com.google.common.hash.Hashing;
import com.google.common.io.ByteStreams;
import com.google.common.net.HostAndPort;
import com.google.common.primitives.Ints;
import io.airlift.http.client.HttpUriBuilder;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtException;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.prestosql.server.security.PasswordAuthenticatorManager;
import io.prestosql.server.ui.WebUiAuthenticationManager;
import io.prestosql.server.ui.WebUiConfig;
import io.prestosql.spi.security.AccessDeniedException;
import io.prestosql.spi.security.BasicPrincipal;
import io.prestosql.spi.security.Identity;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.security.SecureRandom;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import javax.inject.Inject;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;

public class FormWebUiAuthenticationManager
implements WebUiAuthenticationManager {
    private static final String PRESTO_UI_AUDIENCE = "presto-ui";
    private static final String PRESTO_UI_COOKIE = "Presto-UI-Token";
    private static final String LOGIN_FORM = "/ui/login.html";
    private static final String DISABLED_LOCATION = "/ui/disabled.html";
    private static final String UI_LOCATION = "/ui/";
    private final Function<String, String> jwtParser;
    private final Function<String, String> jwtGenerator;
    private final PasswordAuthenticatorManager passwordAuthenticatorManager;

    @Inject
    public FormWebUiAuthenticationManager(WebUiConfig config, PasswordAuthenticatorManager passwordAuthenticatorManager) {
        byte[] hmac;
        if (config.getSharedSecret().isPresent()) {
            hmac = Hashing.sha256().hashString((CharSequence)config.getSharedSecret().get(), StandardCharsets.UTF_8).asBytes();
        } else {
            hmac = new byte[32];
            new SecureRandom().nextBytes(hmac);
        }
        this.jwtParser = jwt -> FormWebUiAuthenticationManager.parseJwt(hmac, jwt);
        long sessionTimeoutNanos = config.getSessionTimeout().roundTo(TimeUnit.NANOSECONDS);
        this.jwtGenerator = username -> FormWebUiAuthenticationManager.generateJwt(hmac, username, sessionTimeoutNanos);
        this.passwordAuthenticatorManager = Objects.requireNonNull(passwordAuthenticatorManager, "passwordAuthenticatorManager is null");
    }

    @Override
    public void handleUiRequest(HttpServletRequest request, HttpServletResponse response, FilterChain nextFilter) throws IOException, ServletException {
        if (request.getPathInfo() == null || request.getPathInfo().equals("/")) {
            FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getUiLocation(request));
            return;
        }
        if (FormWebUiAuthenticationManager.isPublic(request)) {
            nextFilter.doFilter((ServletRequest)request, (ServletResponse)response);
            return;
        }
        if (request.getPathInfo().equals("/ui/login")) {
            this.handleLoginRequest(request, response);
            return;
        }
        if (request.getPathInfo().equals("/ui/logout")) {
            this.handleLogoutRequest(request, response);
            return;
        }
        Optional<String> username = this.getAuthenticatedUsername(request);
        if (username.isPresent()) {
            if (request.getPathInfo().equals(LOGIN_FORM)) {
                FormWebUiAuthenticationManager.sendRedirectFromSuccessfulLogin(request, response, request.getQueryString());
                return;
            }
            nextFilter.doFilter(FormWebUiAuthenticationManager.withUsername(request, username.get()), (ServletResponse)response);
            return;
        }
        FormWebUiAuthenticationManager.getAuthenticationCookie(request).ifPresent(ignored -> response.addCookie(FormWebUiAuthenticationManager.getDeleteCookie(request)));
        try (ServletInputStream inputStream = request.getInputStream();){
            ByteStreams.exhaust((InputStream)inputStream);
        }
        if (request.getPathInfo().startsWith("/ui/api/")) {
            response.setHeader("WWW-Authenticate", "Presto-Form-Login");
            response.setStatus(401);
            return;
        }
        if (!this.isAuthenticationEnabled(request)) {
            FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getRedirectLocation(request, DISABLED_LOCATION));
            return;
        }
        if (request.getPathInfo().equals(LOGIN_FORM)) {
            nextFilter.doFilter((ServletRequest)request, (ServletResponse)response);
            return;
        }
        FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getRedirectLocation(request, LOGIN_FORM, FormWebUiAuthenticationManager.encodeCurrentLocationForLoginRedirect(request)));
    }

    private static String encodeCurrentLocationForLoginRedirect(HttpServletRequest request) {
        String path = request.getPathInfo();
        if (!Strings.isNullOrEmpty((String)request.getQueryString())) {
            path = path + "?" + request.getQueryString();
        }
        if (path.equals("/ui") || path.equals(UI_LOCATION)) {
            return null;
        }
        return path;
    }

    private void handleLoginRequest(HttpServletRequest request, HttpServletResponse response) {
        if (!this.isAuthenticationEnabled(request)) {
            FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getRedirectLocation(request, DISABLED_LOCATION));
            return;
        }
        Optional<String> username = this.checkLoginCredentials(request);
        if (username.isPresent()) {
            response.addCookie(this.createAuthenticationCookie(request, username.get()));
            FormWebUiAuthenticationManager.sendRedirectFromSuccessfulLogin(request, response, request.getParameter("redirectPath"));
            return;
        }
        FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getLoginFormLocation(request));
    }

    private static void sendRedirectFromSuccessfulLogin(HttpServletRequest request, HttpServletResponse response, String redirectPath) {
        try {
            URI redirectUri = new URI((String)MoreObjects.firstNonNull((Object)Strings.emptyToNull((String)redirectPath), (Object)UI_LOCATION));
            FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getRedirectLocation(request, redirectUri.getPath(), redirectUri.getQuery()));
        }
        catch (URISyntaxException ignored) {
            FormWebUiAuthenticationManager.sendRedirect(response, UI_LOCATION);
        }
    }

    private Optional<String> checkLoginCredentials(HttpServletRequest request) {
        String username = Strings.emptyToNull((String)request.getParameter("username"));
        if (username == null) {
            return Optional.empty();
        }
        if (!FormWebUiAuthenticationManager.isHttps(request)) {
            return Optional.of(username);
        }
        String password = Strings.emptyToNull((String)request.getParameter("password"));
        try {
            this.passwordAuthenticatorManager.getAuthenticator().createAuthenticatedPrincipal(username, password);
            return Optional.of(username);
        }
        catch (AccessDeniedException e) {
            return Optional.empty();
        }
    }

    private void handleLogoutRequest(HttpServletRequest request, HttpServletResponse response) {
        response.addCookie(FormWebUiAuthenticationManager.getDeleteCookie(request));
        if (this.isAuthenticationEnabled(request)) {
            FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getLoginFormLocation(request));
            return;
        }
        FormWebUiAuthenticationManager.sendRedirect(response, FormWebUiAuthenticationManager.getRedirectLocation(request, DISABLED_LOCATION));
    }

    private Optional<String> getAuthenticatedUsername(HttpServletRequest request) {
        Optional<Cookie> cookie = FormWebUiAuthenticationManager.getAuthenticationCookie(request);
        if (cookie.isPresent()) {
            try {
                return Optional.of(this.jwtParser.apply(cookie.get().getValue()));
            }
            catch (JwtException e) {
                return Optional.empty();
            }
            catch (RuntimeException e) {
                throw new RuntimeException("Authentication error", e);
            }
        }
        return Optional.empty();
    }

    private static ServletRequest withUsername(HttpServletRequest request, String username) {
        Objects.requireNonNull(username, "username is null");
        final BasicPrincipal principal = new BasicPrincipal(username);
        request.setAttribute("presto.authenticated-identity", (Object)Identity.forUser((String)username).withPrincipal((Principal)principal).build());
        return new HttpServletRequestWrapper(request){

            public Principal getUserPrincipal() {
                return principal;
            }
        };
    }

    private Cookie createAuthenticationCookie(HttpServletRequest request, String userName) {
        String jwt = this.jwtGenerator.apply(userName);
        Cookie cookie = new Cookie(PRESTO_UI_COOKIE, jwt);
        cookie.setSecure(FormWebUiAuthenticationManager.isHttps(request));
        cookie.setHttpOnly(true);
        cookie.setPath("/ui");
        return cookie;
    }

    private static Cookie getDeleteCookie(HttpServletRequest request) {
        Cookie cookie = new Cookie(PRESTO_UI_COOKIE, "delete");
        cookie.setMaxAge(0);
        cookie.setSecure(FormWebUiAuthenticationManager.isHttps(request));
        cookie.setHttpOnly(true);
        return cookie;
    }

    private static Optional<Cookie> getAuthenticationCookie(HttpServletRequest request) {
        return Arrays.stream((Object[])MoreObjects.firstNonNull((Object)request.getCookies(), (Object)new Cookie[0])).filter(cookie -> cookie.getName().equals(PRESTO_UI_COOKIE)).findFirst();
    }

    private static boolean isPublic(HttpServletRequest request) {
        String pathInfo = request.getPathInfo();
        return pathInfo.equals(DISABLED_LOCATION) || pathInfo.startsWith("/ui/vendor") || pathInfo.startsWith("/ui/assets");
    }

    private static void sendRedirect(HttpServletResponse response, String location) {
        response.setHeader("Location", location);
        response.setStatus(303);
    }

    private static String getLoginFormLocation(HttpServletRequest request) {
        return FormWebUiAuthenticationManager.getRedirectLocation(request, LOGIN_FORM);
    }

    private static String getUiLocation(HttpServletRequest request) {
        return FormWebUiAuthenticationManager.getRedirectLocation(request, UI_LOCATION);
    }

    private static String getRedirectLocation(HttpServletRequest request, String path) {
        return FormWebUiAuthenticationManager.getRedirectLocation(request, path, null);
    }

    static String getRedirectLocation(HttpServletRequest request, String path, String queryParameter) {
        HttpUriBuilder builder = FormWebUiAuthenticationManager.toUriBuilderWithForwarding(request);
        builder.replacePath(path);
        if (queryParameter != null) {
            builder.addParameter(queryParameter, new String[0]);
        }
        return builder.toString();
    }

    private static boolean isHttps(HttpServletRequest request) {
        return "https".equals(MoreObjects.firstNonNull((Object)Strings.emptyToNull((String)request.getHeader("X-Forwarded-Proto")), (Object)request.getScheme()));
    }

    private boolean isAuthenticationEnabled(HttpServletRequest request) {
        return !FormWebUiAuthenticationManager.isHttps(request) || this.passwordAuthenticatorManager.isLoaded();
    }

    private static String generateJwt(byte[] hmac, String username, long sessionTimeoutNanos) {
        return Jwts.builder().signWith(SignatureAlgorithm.HS256, hmac).setSubject(username).setExpiration(Date.from(ZonedDateTime.now().plusNanos(sessionTimeoutNanos).toInstant())).setAudience(PRESTO_UI_AUDIENCE).compact();
    }

    private static String parseJwt(byte[] hmac, String jwt) {
        return ((Claims)Jwts.parser().setSigningKey(hmac).requireAudience(PRESTO_UI_AUDIENCE).parseClaimsJws(jwt).getBody()).getSubject();
    }

    private static HttpUriBuilder toUriBuilderWithForwarding(HttpServletRequest request) {
        HttpUriBuilder builder;
        if (Strings.isNullOrEmpty((String)request.getHeader("X-Forwarded-Proto")) && Strings.isNullOrEmpty((String)request.getHeader("X-Forwarded-Host"))) {
            builder = HttpUriBuilder.uriBuilder().scheme(request.getScheme()).hostAndPort(HostAndPort.fromString((String)request.getHeader("Host")));
        } else {
            builder = HttpUriBuilder.uriBuilder().scheme((String)MoreObjects.firstNonNull((Object)Strings.emptyToNull((String)request.getHeader("X-Forwarded-Proto")), (Object)request.getScheme())).hostAndPort(HostAndPort.fromString((String)((String)MoreObjects.firstNonNull((Object)Strings.emptyToNull((String)request.getHeader("X-Forwarded-Host")), (Object)request.getHeader("Host")))));
            Optional.ofNullable(Strings.emptyToNull((String)request.getHeader("X-Forwarded-Port"))).map(Ints::tryParse).ifPresent(arg_0 -> ((HttpUriBuilder)builder).port(arg_0));
        }
        return builder;
    }
}

