/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.mledger.util;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import io.netty.util.IllegalReferenceCountException;
import io.netty.util.Recycler;
import io.netty.util.ReferenceCounted;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.NavigableMap;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.StampedLock;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RangeCache<Key extends Comparable<Key>, Value extends ValueWithKeyValidation<Key>> {
    private static final Logger log = LoggerFactory.getLogger(RangeCache.class);
    private final ConcurrentNavigableMap<Key, EntryWrapper<Key, Value>> entries;
    private AtomicLong size = new AtomicLong(0L);
    private final Weighter<Value> weighter;
    private final TimestampExtractor<Value> timestampExtractor;

    public RangeCache() {
        this(new DefaultWeighter(), x -> System.nanoTime());
    }

    public RangeCache(Weighter<Value> weighter, TimestampExtractor<Value> timestampExtractor) {
        this.entries = new ConcurrentSkipListMap<Key, EntryWrapper<Key, Value>>();
        this.weighter = weighter;
        this.timestampExtractor = timestampExtractor;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean put(Key key, Value value) {
        value.retain();
        try {
            if (!value.matchesKey(key)) {
                throw new IllegalArgumentException("Value '" + value + "' does not match key '" + key + "'");
            }
            long entrySize = this.weighter.getSize(value);
            EntryWrapper<Key, Value> newWrapper = EntryWrapper.create(key, value, entrySize);
            if (this.entries.putIfAbsent(key, newWrapper) == null) {
                this.size.addAndGet(entrySize);
                boolean bl = true;
                return bl;
            }
            newWrapper.recycle();
            boolean bl = false;
            return bl;
        }
        finally {
            value.release();
        }
    }

    public boolean exists(Key key) {
        return key != null ? this.entries.containsKey(key) : true;
    }

    public Value get(Key key) {
        return this.getValue(key, (EntryWrapper)this.entries.get(key));
    }

    private Value getValue(Key key, EntryWrapper<Key, Value> valueWrapper) {
        if (valueWrapper == null) {
            return null;
        }
        ValueWithKeyValidation value = (ValueWithKeyValidation)valueWrapper.getValue(key);
        if (value == null) {
            return null;
        }
        try {
            value.retain();
        }
        catch (IllegalReferenceCountException e) {
            return null;
        }
        if (value.refCnt() > 1 && value.matchesKey(key)) {
            return (Value)value;
        }
        value.release();
        return null;
    }

    public Collection<Value> getRange(Key first, Key last) {
        ArrayList<Value> values = new ArrayList<Value>();
        for (Map.Entry entry : this.entries.subMap((Object)first, true, (Object)last, true).entrySet()) {
            Value value = this.getValue((Comparable)entry.getKey(), (EntryWrapper)entry.getValue());
            if (value == null) continue;
            values.add(value);
        }
        return values;
    }

    public Pair<Integer, Long> removeRange(Key first, Key last, boolean lastInclusive) {
        RemovalCounters counters = RemovalCounters.create();
        NavigableMap subMap = this.entries.subMap((Object)first, true, (Object)last, lastInclusive);
        for (Map.Entry entry : subMap.entrySet()) {
            this.removeEntry(entry, counters, true);
        }
        return this.handleRemovalResult(counters);
    }

    private RemoveEntryResult removeEntry(Map.Entry<Key, EntryWrapper<Key, Value>> entry, RemovalCounters counters, boolean skipInvalid) {
        return this.removeEntry(entry, counters, skipInvalid, x -> true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private RemoveEntryResult removeEntry(Map.Entry<Key, EntryWrapper<Key, Value>> entry, RemovalCounters counters, boolean skipInvalid, Predicate<Value> removeCondition) {
        Comparable key = (Comparable)entry.getKey();
        EntryWrapper<Key, Value> entryWrapper = entry.getValue();
        ValueWithKeyValidation value = (ValueWithKeyValidation)entryWrapper.getValue(key);
        if (value == null) {
            EntryWrapper removed;
            if (!skipInvalid && (removed = (EntryWrapper)this.entries.remove(key)) != null) {
                log.info("Key {} does not match the entry's value wrapper's key {}, removed entry by key without releasing the value", (Object)key, entryWrapper.getKey());
                counters.entryRemoved(removed.getSize());
                return RemoveEntryResult.ENTRY_REMOVED;
            }
            return RemoveEntryResult.CONTINUE_LOOP;
        }
        try {
            value.retain();
        }
        catch (IllegalReferenceCountException e) {
            if (!skipInvalid && this.entries.remove(key, entryWrapper)) {
                log.info("Value was already released for key {}, removed entry without releasing the value", (Object)key);
                counters.entryRemoved(entryWrapper.getSize());
                return RemoveEntryResult.ENTRY_REMOVED;
            }
            return RemoveEntryResult.CONTINUE_LOOP;
        }
        if (!value.matchesKey(key)) {
            log.warn("Unexpected race condition. Value {} does not match the key {}. Removing entry.", (Object)value, (Object)key);
        }
        try {
            if (!removeCondition.test((Object)value)) {
                RemoveEntryResult e = RemoveEntryResult.BREAK_LOOP;
                return e;
            }
            if (!skipInvalid) {
                boolean entryRemoved = this.entries.remove(key, entryWrapper);
                if (entryRemoved) {
                    counters.entryRemoved(entryWrapper.getSize());
                    if (value.refCnt() > 1) {
                        entryWrapper.recycle();
                        value.release();
                    } else {
                        log.info("Unexpected refCnt {} for key {}, removed entry without releasing the value", (Object)value.refCnt(), (Object)key);
                    }
                }
            } else if (skipInvalid && value.refCnt() > 1 && this.entries.remove(key, entryWrapper)) {
                counters.entryRemoved(entryWrapper.getSize());
                entryWrapper.recycle();
                value.release();
            }
        }
        finally {
            value.release();
        }
        return RemoveEntryResult.ENTRY_REMOVED;
    }

    private Pair<Integer, Long> handleRemovalResult(RemovalCounters counters) {
        this.size.addAndGet(-counters.removedSize);
        Pair result = Pair.of((Object)counters.removedEntries, (Object)counters.removedSize);
        counters.recycle();
        return result;
    }

    public Pair<Integer, Long> evictLeastAccessedEntries(long minSize) {
        Map.Entry entry;
        Preconditions.checkArgument((minSize > 0L ? 1 : 0) != 0);
        RemovalCounters counters = RemovalCounters.create();
        while (counters.removedSize < minSize && !Thread.currentThread().isInterrupted() && (entry = this.entries.firstEntry()) != null) {
            this.removeEntry(entry, counters, false);
        }
        return this.handleRemovalResult(counters);
    }

    public Pair<Integer, Long> evictLEntriesBeforeTimestamp(long maxTimestamp) {
        Map.Entry entry;
        RemovalCounters counters = RemovalCounters.create();
        while (!Thread.currentThread().isInterrupted() && (entry = this.entries.firstEntry()) != null && this.removeEntry(entry, counters, false, value -> this.timestampExtractor.getTimestamp(value) <= maxTimestamp) != RemoveEntryResult.BREAK_LOOP) {
        }
        return this.handleRemovalResult(counters);
    }

    protected long getNumberOfEntries() {
        return this.entries.size();
    }

    public long getSize() {
        return this.size.get();
    }

    public Pair<Integer, Long> clear() {
        Map.Entry entry;
        RemovalCounters counters = RemovalCounters.create();
        while (!Thread.currentThread().isInterrupted() && (entry = this.entries.firstEntry()) != null) {
            this.removeEntry(entry, counters, false);
        }
        return this.handleRemovalResult(counters);
    }

    private static class DefaultWeighter<Value>
    implements Weighter<Value> {
        private DefaultWeighter() {
        }

        @Override
        public long getSize(Value value) {
            return 1L;
        }
    }

    public static interface TimestampExtractor<ValueT> {
        public long getTimestamp(ValueT var1);
    }

    public static interface Weighter<ValueT> {
        public long getSize(ValueT var1);
    }

    public static interface ValueWithKeyValidation<T>
    extends ReferenceCounted {
        public boolean matchesKey(T var1);
    }

    private static class EntryWrapper<K, V> {
        private final Recycler.Handle<EntryWrapper> recyclerHandle;
        private static final Recycler<EntryWrapper> RECYCLER = new Recycler<EntryWrapper>(){

            protected EntryWrapper newObject(Recycler.Handle<EntryWrapper> recyclerHandle) {
                return new EntryWrapper(recyclerHandle);
            }
        };
        private final StampedLock lock = new StampedLock();
        private K key;
        private V value;
        long size;

        private EntryWrapper(Recycler.Handle<EntryWrapper> recyclerHandle) {
            this.recyclerHandle = recyclerHandle;
        }

        static <K, V> EntryWrapper<K, V> create(K key, V value, long size) {
            EntryWrapper entryWrapper = (EntryWrapper)RECYCLER.get();
            long stamp = entryWrapper.lock.writeLock();
            entryWrapper.key = key;
            entryWrapper.value = value;
            entryWrapper.size = size;
            entryWrapper.lock.unlockWrite(stamp);
            return entryWrapper;
        }

        K getKey() {
            long stamp = this.lock.tryOptimisticRead();
            K localKey = this.key;
            if (!this.lock.validate(stamp)) {
                stamp = this.lock.readLock();
                localKey = this.key;
                this.lock.unlockRead(stamp);
            }
            return localKey;
        }

        V getValue(K key) {
            long stamp = this.lock.tryOptimisticRead();
            K localKey = this.key;
            V localValue = this.value;
            if (!this.lock.validate(stamp)) {
                stamp = this.lock.readLock();
                localKey = this.key;
                localValue = this.value;
                this.lock.unlockRead(stamp);
            }
            if (localKey != key) {
                return null;
            }
            return localValue;
        }

        long getSize() {
            long stamp = this.lock.tryOptimisticRead();
            long localSize = this.size;
            if (!this.lock.validate(stamp)) {
                stamp = this.lock.readLock();
                localSize = this.size;
                this.lock.unlockRead(stamp);
            }
            return localSize;
        }

        void recycle() {
            this.key = null;
            this.value = null;
            this.size = 0L;
            this.recyclerHandle.recycle((Object)this);
        }
    }

    private static class RemovalCounters {
        private final Recycler.Handle<RemovalCounters> recyclerHandle;
        private static final Recycler<RemovalCounters> RECYCLER = new Recycler<RemovalCounters>(){

            protected RemovalCounters newObject(Recycler.Handle<RemovalCounters> recyclerHandle) {
                return new RemovalCounters(recyclerHandle);
            }
        };
        int removedEntries;
        long removedSize;

        private RemovalCounters(Recycler.Handle<RemovalCounters> recyclerHandle) {
            this.recyclerHandle = recyclerHandle;
        }

        static <T> RemovalCounters create() {
            RemovalCounters results = (RemovalCounters)RECYCLER.get();
            results.removedEntries = 0;
            results.removedSize = 0L;
            return results;
        }

        void recycle() {
            this.removedEntries = 0;
            this.removedSize = 0L;
            this.recyclerHandle.recycle((Object)this);
        }

        public void entryRemoved(long size) {
            this.removedSize += size;
            ++this.removedEntries;
        }
    }

    static enum RemoveEntryResult {
        ENTRY_REMOVED,
        CONTINUE_LOOP,
        BREAK_LOOP;

    }
}

