var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
import { html, nothing, svg, unsafeCSS } from 'lit';
import { customElement, property, state } from 'lit/decorators.js';
import { createRef, ref } from 'lit/directives/ref.js';
import { MutantStatus } from 'mutation-testing-report-schema/api';
import { findDiffIndices, gte, highlightCode, transformHighlightedLines } from '../../lib/code-helpers';
import { createCustomEvent } from '../../lib/custom-events';
import { escapeHtml, getContextClassForStatus, getEmojiForStatus, scrollToCodeFragmentIfNeeded } from '../../lib/html-helpers';
import { prismjs, tailwind } from '../../style';
import style from './file.scss';
import { renderDots, renderLine } from './util';
import { RealTimeElement } from '../real-time-element';
const diffOldClass = 'diff-old';
const diffNewClass = 'diff-new';
export let FileComponent = class FileComponent extends RealTimeElement {
    constructor() {
        super(...arguments);
        this.filters = [];
        this.selectedMutantStates = [];
        this.lines = [];
        this.mutants = [];
        this.codeRef = createRef();
        this.filtersChanged = (event) => {
            // Pending is not filterable, but they should still be shown to the user.
            this.selectedMutantStates = event.detail.concat([MutantStatus.Pending]);
        };
        this.codeClicked = (ev) => {
            ev.stopPropagation();
            if (ev.target instanceof Element) {
                let maybeMutantTarget = ev.target;
                const mutantsInScope = [];
                for (; maybeMutantTarget instanceof Element; maybeMutantTarget = maybeMutantTarget.parentElement) {
                    const mutantId = maybeMutantTarget.getAttribute('mutant-id');
                    const mutant = this.mutants.find(({ id }) => id.toString() === mutantId);
                    if (mutant) {
                        mutantsInScope.push(mutant);
                    }
                }
                const index = (this.selectedMutant ? mutantsInScope.indexOf(this.selectedMutant) : -1) + 1;
                if (mutantsInScope[index]) {
                    this.toggleMutant(mutantsInScope[index]);
                    clearSelection();
                }
                else if (this.selectedMutant) {
                    this.toggleMutant(this.selectedMutant);
                    clearSelection();
                }
            }
        };
        this.nextMutant = () => {
            const index = this.selectedMutant ? (this.mutants.indexOf(this.selectedMutant) + 1) % this.mutants.length : 0;
            if (this.mutants[index]) {
                this.toggleMutant(this.mutants[index]);
            }
        };
        this.previousMutant = () => {
            const index = this.selectedMutant
                ? (this.mutants.indexOf(this.selectedMutant) + this.mutants.length - 1) % this.mutants.length
                : this.mutants.length - 1;
            if (this.mutants[index]) {
                this.toggleMutant(this.mutants[index]);
            }
        };
    }
    render() {
        const mutantLineMap = new Map();
        for (const mutant of this.mutants) {
            let mutants = mutantLineMap.get(mutant.location.start.line);
            if (!mutants) {
                mutants = [];
                mutantLineMap.set(mutant.location.start.line, mutants);
            }
            mutants.push(mutant);
        }
        const renderFinalMutants = (lastLine) => {
            return this.renderMutantDots([...mutantLineMap.entries()].filter(([line]) => line > lastLine).flatMap(([, mutants]) => mutants));
        };
        return html `
      <mte-state-filter
        allow-toggle-all
        .filters="${this.filters}"
        @filters-changed="${this.filtersChanged}"
        @next=${this.nextMutant}
        @previous=${this.previousMutant}
      ></mte-state-filter>
      <pre
        @click="${this.codeClicked}"
        id="report-code-block"
        class="line-numbers ${this.selectedMutantStates.map((state) => `mte-selected-${state}`).join(' ')} flex rounded-md py-4"
      >
        <code ${ref(this.codeRef)} class="flex language-${this.model.language}">
          <table>${this.lines.map((line, lineIndex) => {
            const lineNr = lineIndex + 1;
            const mutantDots = this.renderMutantDots(mutantLineMap.get(lineNr));
            const finalMutants = this.lines.length === lineNr ? renderFinalMutants(lineNr) : nothing;
            return renderLine(line, renderDots(mutantDots, finalMutants));
        })}</table>
          </code>
          </pre>
    `;
    }
    renderMutantDots(mutants) {
        return mutants?.length
            ? mutants.map((mutant) => svg `<svg mutant-id="${mutant.id}" class="mutant-dot ${this.selectedMutant?.id === mutant.id ? 'selected' : mutant.status}" height="10" width="12">
          <title>${title(mutant)}</title>
          <circle cx="5" cy="5" r="5" />
          </svg>`)
            : nothing;
    }
    toggleMutant(mutant) {
        this.removeCurrentDiff();
        if (this.selectedMutant === mutant) {
            this.selectedMutant = undefined;
            this.dispatchEvent(createCustomEvent('mutant-selected', { selected: false, mutant }));
            return;
        }
        this.selectedMutant = mutant;
        const lines = this.codeRef.value.querySelectorAll('tr.line');
        for (let i = mutant.location.start.line - 1; i < mutant.location.end.line; i++) {
            lines.item(i).classList.add(diffOldClass);
        }
        const mutatedLines = this.highlightedReplacementRows(mutant);
        const mutantEndRow = lines.item(mutant.location.end.line - 1);
        mutantEndRow.insertAdjacentHTML('afterend', mutatedLines);
        scrollToCodeFragmentIfNeeded(mutantEndRow);
        this.dispatchEvent(createCustomEvent('mutant-selected', { selected: true, mutant }));
    }
    removeCurrentDiff() {
        const oldDiffLines = this.codeRef.value.querySelectorAll(`.${diffOldClass}`);
        oldDiffLines.forEach((oldDiffLine) => oldDiffLine.classList.remove(diffOldClass));
        const newDiffLines = this.codeRef.value.querySelectorAll(`.${diffNewClass}`);
        newDiffLines.forEach((newDiffLine) => newDiffLine.remove());
    }
    reactivate() {
        super.reactivate();
        this.updateFileRepresentation();
    }
    update(changes) {
        if (changes.has('model') && this.model) {
            this.updateFileRepresentation();
        }
        if ((changes.has('model') && this.model) || changes.has('selectedMutantStates')) {
            this.mutants = this.model.mutants
                .filter((mutant) => this.selectedMutantStates.includes(mutant.status))
                .sort((m1, m2) => (gte(m1.location.start, m2.location.start) ? 1 : -1));
            if (this.selectedMutant &&
                !this.mutants.includes(this.selectedMutant) &&
                changes.has('selectedMutantStates') &&
                // This extra check is to allow mutants that have been opened before, to stay open when a realtime update comes through
                this.selectedMutantsHaveChanged(changes.get('selectedMutantStates'))) {
                this.toggleMutant(this.selectedMutant);
            }
        }
        super.update(changes);
    }
    updateFileRepresentation() {
        this.filters = [
            MutantStatus.Killed,
            MutantStatus.Survived,
            MutantStatus.NoCoverage,
            MutantStatus.Ignored,
            MutantStatus.Timeout,
            MutantStatus.CompileError,
            MutantStatus.RuntimeError,
        ]
            .filter((status) => this.model.mutants.some((mutant) => mutant.status === status))
            .map((status) => ({
            enabled: [...this.selectedMutantStates, MutantStatus.Survived, MutantStatus.NoCoverage, MutantStatus.Timeout].includes(status),
            count: this.model.mutants.filter((m) => m.status === status).length,
            status,
            label: html `${getEmojiForStatus(status)} ${status}`,
            context: getContextClassForStatus(status),
        }));
        const highlightedSource = highlightCode(this.model.source, this.model.name);
        const startedMutants = new Set();
        const mutantsToPlace = new Set(this.model.mutants);
        this.lines = transformHighlightedLines(highlightedSource, function* (position) {
            // End previously opened mutants
            for (const mutant of startedMutants) {
                if (gte(position, mutant.location.end)) {
                    startedMutants.delete(mutant);
                    yield { elementName: 'span', id: mutant.id, isClosing: true };
                }
            }
            // Open new mutants
            for (const mutant of mutantsToPlace) {
                if (gte(position, mutant.location.start)) {
                    startedMutants.add(mutant);
                    mutantsToPlace.delete(mutant);
                    yield {
                        elementName: 'span',
                        id: mutant.id,
                        attributes: {
                            class: escapeHtml(`mutant border-none ${mutant.status}`),
                            title: escapeHtml(title(mutant)),
                            'mutant-id': escapeHtml(mutant.id.toString()),
                        },
                    };
                }
            }
        });
    }
    selectedMutantsHaveChanged(changedMutantStates) {
        if (changedMutantStates.length !== this.selectedMutantStates.length) {
            return true;
        }
        return !changedMutantStates.every((state, index) => this.selectedMutantStates[index] === state);
    }
    highlightedReplacementRows(mutant) {
        const mutatedLines = mutant.getMutatedLines().trimEnd();
        const originalLines = mutant.getOriginalLines().trimEnd();
        const [focusFrom, focusTo] = findDiffIndices(originalLines, mutatedLines);
        const lines = transformHighlightedLines(highlightCode(mutatedLines, this.model.name), function* ({ offset }) {
            if (offset === focusFrom) {
                yield { elementName: 'span', id: 'diff-focus', attributes: { class: 'diff-focus' } };
            }
            else if (offset === focusTo) {
                yield { elementName: 'span', id: 'diff-focus', isClosing: true };
            }
            return;
        });
        const lineStart = `<tr class="${diffNewClass}"><td class="empty-line-number"></td><td class="line-marker"></td><td class="code">`;
        const lineEnd = '</td></tr>';
        return lines.map((line) => `${lineStart}${line}${lineEnd}`).join('');
    }
};
FileComponent.styles = [prismjs, tailwind, unsafeCSS(style)];
__decorate([
    state()
], FileComponent.prototype, "filters", void 0);
__decorate([
    property()
], FileComponent.prototype, "model", void 0);
__decorate([
    state()
], FileComponent.prototype, "selectedMutantStates", void 0);
__decorate([
    state()
], FileComponent.prototype, "selectedMutant", void 0);
__decorate([
    state()
], FileComponent.prototype, "lines", void 0);
__decorate([
    state()
], FileComponent.prototype, "mutants", void 0);
FileComponent = __decorate([
    customElement('mte-file')
], FileComponent);
function title(mutant) {
    return `${mutant.mutatorName} ${mutant.status}`;
}
function clearSelection() {
    window.getSelection()?.removeAllRanges();
}
//# sourceMappingURL=file.component.js.map