package io.taig.taigless.hashing

import java.nio.charset.{Charset, StandardCharsets}
import java.util.Base64

import cats.syntax.all._
import cats.{Functor, MonadThrow}

abstract class Hashing[F[_], A, B] {
  def hash(payload: A): F[B]

  def verify(payload: A, hash: B): F[Boolean]
}

object Hashing {
  def toBase64[F[_]: Functor, A](hashing: Hashing[F, A, Array[Byte]]): Hashing[F, A, String] =
    new Hashing[F, A, String] {
      override def hash(payload: A): F[String] = hashing.hash(payload).map(Base64.getEncoder.encodeToString)

      override def verify(payload: A, hash: String): F[Boolean] =
        hashing.verify(payload, Base64.getDecoder.decode(hash))
    }

  def toHex[F[_]: MonadThrow, A](hashing: Hashing[F, A, Array[Byte]]): Hashing[F, A, String] =
    new Hashing[F, A, String] {
      override def hash(payload: A): F[String] = hashing.hash(payload).map(Hex.fromBytes)

      override def verify(payload: A, hash: String): F[Boolean] =
        Hex.toBytes(hash).liftTo[F](new IllegalArgumentException("Invalid hex representation")).flatMap { hash =>
          hashing.verify(payload, hash)
        }
    }

  def fromText[F[_], A](
      hashing: Hashing[F, Array[Byte], A],
      charset: Charset = StandardCharsets.UTF_8
  ): Hashing[F, String, A] = new Hashing[F, String, A] {
    override def hash(payload: String): F[A] = hashing.hash(payload.getBytes(charset))

    override def verify(payload: String, hash: A): F[Boolean] = hashing.verify(payload.getBytes(charset), hash)
  }

  def toText[F[_]: Functor, A](
      hashing: Hashing[F, A, Array[Byte]],
      charset: Charset = StandardCharsets.UTF_8
  ): Hashing[F, A, String] = new Hashing[F, A, String] {
    override def hash(payload: A): F[String] = hashing.hash(payload).map(new String(_, charset))

    override def verify(payload: A, hash: String): F[Boolean] = hashing.verify(payload, hash.getBytes(charset))
  }

  def fromTextToText[F[_]: Functor](
      hashing: Hashing[F, Array[Byte], Array[Byte]],
      charset: Charset = StandardCharsets.UTF_8
  ): Hashing[F, String, String] = toText(fromText(hashing, charset), charset)

  def fromTextToHex[F[_]: MonadThrow](
      hashing: Hashing[F, Array[Byte], Array[Byte]],
      charset: Charset = StandardCharsets.UTF_8
  ): Hashing[F, String, String] = toHex(fromText(hashing, charset))
}
