package io.taig.taigless

import cats.arrow.Profunctor
import cats.{Contravariant, Functor, Invariant, Semigroupal}
import cats.implicits._

package object twiddler {

  /** Infix alias for `(A, B)` that provides convenient syntax for left-associated HLists.
    *
    * @see https://github.com/tpolecat/skunk/blob/96ac7ee38f65633abae137d3baf6e1aeae53ab8d/modules/core/src/main/scala/package.scala
    */
  type ~[+A, +B] = (A, B)

  /** Companion providing unapply for `~` such that `(x ~ y ~ z) match { case a ~ b ~ c => ... }`.
    *
    * @see https://github.com/tpolecat/skunk/blob/96ac7ee38f65633abae137d3baf6e1aeae53ab8d/modules/core/src/main/scala/package.scala
    */
  object ~ {
    def unapply[A, B](t: A ~ B): Some[A ~ B] = Some(t)
  }

  implicit final class TwiddlerOps[A](val a: A) extends AnyVal {
    def ~[B](b: B): A ~ B = (a, b)
  }

  implicit final class SemigroupalOps[F[_], A](val fa: F[A]) extends AnyVal {
    def <~>[B](fb: F[B])(implicit F: Semigroupal[F]): F[A ~ B] = fa.product(fb)
  }

  implicit final class FunctorOps[F[_], A](val fa: F[A]) extends AnyVal {
    def gmap[B](implicit F: Functor[F], twiddler: Twiddler.Aux[B, A]): F[B] = fa.map(twiddler.from)
  }

  implicit final class ContravariantOps[F[_], A](val fa: F[A]) extends AnyVal {
    def gcontramap[B](implicit F: Contravariant[F], twiddler: Twiddler.Aux[B, A]): F[B] = fa.contramap(twiddler.to)
  }

  implicit final class InvariantOps[F[_], A](val fa: F[A]) extends AnyVal {
    def gimap[B](implicit F: Invariant[F], twiddler: Twiddler.Aux[B, A]): F[B] = fa.imap(twiddler.from)(twiddler.to)
  }

  implicit final class ProfunctorOps[F[_, _], A, B](val fab: F[A, B]) extends AnyVal {
    def gdimap[C, D](implicit F: Profunctor[F], in: Twiddler.Aux[A, C], out: Twiddler.Aux[B, D]): F[C, D] =
      fab.dimap(in.from)(out.to)

    def grmap[C](implicit F: Profunctor[F], out: Twiddler.Aux[B, C]): F[A, C] = fab.rmap(out.to)

    def glmap[C](implicit F: Profunctor[F], out: Twiddler.Aux[A, C]): F[C, B] = fab.lmap(out.from)
  }
}
