package io.taig.taigless.validation

import cats.data.{Validated, ValidatedNel}
import cats.implicits._

trait Validation {
  def between[A](from: A, to: A)(value: A)(implicit numeric: Numeric[A]): ValidatedNel[String, Unit] =
    Validated.condNel(numeric.gteq(value, from) && numeric.lteq(value, to), (), s"Must be between $from and $to")

  def bigDecimal(value: String): ValidatedNel[String, BigDecimal] = numeric(BigDecimal(_))(value)

  def bigInt(value: String): ValidatedNel[String, BigInt] = numeric(BigInt(_))(value)

  def compare[A](operator: String, reference: A, f: (A, A) => Boolean)(value: A): ValidatedNel[String, Unit] =
    Validated.condNel(f(value, reference), (), s"Must be $operator $reference")

  def contains(reference: String)(value: String): ValidatedNel[String, Unit] =
    Validated.condNel(value contains reference, (), s"Must contain '$reference'")

  def double(value: String): ValidatedNel[String, Double] = numeric(_.toDouble)(value)

  def float(value: String): ValidatedNel[String, Float] = numeric(_.toFloat)(value)

  def gt[A](reference: A)(value: A)(implicit numeric: Numeric[A]): ValidatedNel[String, Unit] =
    compare[A](">", reference, numeric.gt)(value)

  def gteq[A](reference: A)(value: A)(implicit numeric: Numeric[A]): ValidatedNel[String, Unit] =
    compare[A](">=", reference, numeric.gteq)(value)

  def int(value: String): ValidatedNel[String, Int] = numeric(_.toInt)(value)

  def isDefined[A](value: Option[A]): ValidatedNel[String, A] = value.toValidNel("Required")

  def length(reference: Int)(value: String): ValidatedNel[String, Unit] =
    Validated.condNel(value.length == reference, (), s"Exactly $reference characters")

  def long(value: String): ValidatedNel[String, Long] = numeric(_.toLong)(value)

  def lt[A](reference: A)(value: A)(implicit numeric: Numeric[A]): ValidatedNel[String, Unit] =
    compare[A]("<", reference, numeric.lt)(value)

  def lteq[A](reference: A)(value: A)(implicit numeric: Numeric[A]): ValidatedNel[String, Unit] =
    compare[A]("<=", reference, numeric.lteq)(value)

  def maxLength(reference: Int)(value: String): ValidatedNel[String, Unit] =
    Validated.condNel(value.length <= reference, (), s"Max $reference characters")

  def minLength(reference: Int)(value: String): ValidatedNel[String, Unit] =
    Validated.condNel(value.length >= reference, (), s"At least $reference characters")

  def nonEmpty(value: String): ValidatedNel[String, Unit] = Validated.condNel(value.nonEmpty, (), "Required")

  def numeric[A](unsafeParse: String => A)(value: String): ValidatedNel[String, A] =
    try Validated.valid(unsafeParse(value))
    catch {
      case _: NumberFormatException => Validated.invalidNel("Invalid number format")
    }

  def required(value: String): ValidatedNel[String, String] = {
    val trimmed = value.trim
    nonEmpty(trimmed).as(trimmed)
  }

  def short(value: String): ValidatedNel[String, Short] = numeric(_.toShort)(value)
}

object Validation extends Validation
