package io.taig.taigless.validation

import cats.syntax.all._
import shapeless.{:+:, ::, CNil, Coproduct, Generic, HNil, Inl, Inr, Lazy}

import scala.reflect.ClassTag

trait DerivedAdtFields[A] extends Fields[A]

object DerivedAdtFields {
  implicit val cnil: DerivedAdtFields[CNil] = new DerivedAdtFields[CNil] {
    override def decode(value: String): Either[String, CNil] = Left(s"Unknown name: '$value''")

    override def encode(value: CNil): String = value.impossible
  }

  implicit def cons[H, T <: Coproduct](implicit
      head: DerivedAdtFields[H],
      tail: DerivedAdtFields[T]
  ): DerivedAdtFields[H :+: T] = new DerivedAdtFields[H :+: T] {
    override def decode(value: String): Either[String, H :+: T] =
      head.decode(value).map(Inl.apply).orElse(tail.decode(value).map(Inr.apply))

    override def encode(value: H :+: T): String = value match {
      case Inl(value) => head.encode(value)
      case Inr(value) => tail.encode(value)
    }
  }

  implicit def coproduct[A, B <: Coproduct](implicit
      generic: Generic.Aux[A, B],
      fields: DerivedAdtFields[B]
  ): DerivedAdtFields[A] = new DerivedAdtFields[A] {
    override def decode(value: String): Either[String, A] = fields.decode(value).map(generic.from)

    override def encode(value: A): String = fields.encode(generic.to(value))
  }

  implicit def nested[A, B](implicit
      generic: Lazy[Generic.Aux[A, B :: HNil]],
      fields: Fields[B],
      tag: ClassTag[A]
  ): DerivedAdtFields[A] = new DerivedAdtFields[A] {
    override def decode(value: String): Either[String, A] = {
      (value.indexOf("["), value.indexOf(']')) match {
        case (-1, _) => "Invalid format: missing opening index bracket '['".asLeft
        case (_, -1) => "Invalid format: missing closing index bracket ']'".asLeft
        case (open, close) =>
          val key = value.substring(0, open)
          val parent = toCamelCase(tag.runtimeClass.getSuperclass.getSimpleName)

          if (key != parent) s"Name mismatch: expected '$parent', got '$key'".asLeft
          else if (value.length <= close + 1) s"Invalid format: missing field after adt identifier [$value]".asLeft
          else {
            val expected = value.substring(open + 1, close)
            val field = value.substring(close + 2)

            fields.decode(field).map(b => generic.value.from(b :: HNil)).flatMap { a =>
              val actual = name(a)
              Either.cond(actual == expected, a, s"Name mismatch: expected '$expected', got '$actual'")
            }
          }
      }
    }

    override def encode(a: A): String = {
      val parent = toCamelCase(tag.runtimeClass.getSuperclass.getSimpleName)
      val field :: HNil = generic.value.to(a)
      s"$parent[${name(a)}].${fields.encode(field)}"
    }
  }
}
