package io.taig.taigless.validation

import cats.syntax.all._
import shapeless.{:+:, ::, CNil, Coproduct, Generic, HNil, Inl, Inr, Lazy}

trait DerivedRecordFields[A] extends Fields[A]

object DerivedRecordFields {
  implicit val cnil: DerivedRecordFields[CNil] = new DerivedRecordFields[CNil] {
    override def decode(value: String): Either[String, CNil] = Left(s"Unknown name: '$value''")

    override def encode(value: CNil): String = value.impossible
  }

  implicit def cons[H, T <: Coproduct](implicit
      head: DerivedRecordFields[H],
      tail: DerivedRecordFields[T]
  ): DerivedRecordFields[H :+: T] = new DerivedRecordFields[H :+: T] {
    override def decode(value: String): Either[String, H :+: T] =
      head.decode(value).map(Inl.apply).orElse(tail.decode(value).map(Inr.apply))

    override def encode(value: H :+: T): String = value match {
      case Inl(value) => head.encode(value)
      case Inr(value) => tail.encode(value)
    }
  }

  implicit def coproduct[A, B <: Coproduct](implicit
      generic: Generic.Aux[A, B],
      fields: DerivedRecordFields[B]
  ): DerivedRecordFields[A] = new DerivedRecordFields[A] {
    override def decode(value: String): Either[String, A] = fields.decode(value).map(generic.from)

    override def encode(value: A): String = fields.encode(generic.to(value))
  }

  implicit def field[A](implicit generic: Generic.Aux[A, HNil]): DerivedRecordFields[A] = new DerivedRecordFields[A] {
    override def decode(value: String): Either[String, A] = {
      val a = generic.from(HNil)
      val actual = name(a)
      if (actual == value) Right(a) else Left(s"Name mismatch: expected '$value', got '$actual'")
    }

    override def encode(value: A): String = name(generic.from(HNil))
  }

  implicit def collection[A](implicit generic: Generic.Aux[A, Option[Int] :: HNil]): DerivedRecordFields[A] =
    new DerivedRecordFields[A] {
      override def decode(value: String): Either[String, A] =
        value.indexOf("[") match {
          case -1 =>
            val a = generic.from(none[Int] :: HNil)
            val actual = name(a)
            if (actual == value) Right(a) else Left(s"Name mismatch: expected '$value', got '$actual'")
          case offset =>
            val (key, index) = value.splitAt(offset)
            if (index.endsWith("]"))
              index
                .substring(1, index.length - 1)
                .toIntOption
                .toRight("Invalid format: index is not a valid Int")
                .map(index => generic.from(index.some :: HNil))
                .flatMap { a =>
                  val actual = name(a)
                  if (actual == key) Right(a) else Left(s"Name mismatch: expected '$value', got '$actual'")
                }
            else "Invalid format: index must be enclosed in brackets".asLeft
        }

      override def encode(value: A): String = name(value) + generic.to(value).head.map(index => s"[$index]").orEmpty
    }

  implicit def nested[A, B](implicit
      generic: Lazy[Generic.Aux[A, B :: HNil]],
      fields: Fields[B]
  ): DerivedRecordFields[A] = new DerivedRecordFields[A] {
    override def decode(value: String): Either[String, A] = value.indexOf('.') match {
      case -1 => "Invalid format".asLeft
      case index =>
        val current = value.substring(0, index)
        val field = value.substring(index + 1)
        fields.decode(field).map(b => generic.value.from(b :: HNil)).flatMap { a =>
          val actual = name(a)
          Either.cond(actual == current, a, s"Name mismatch: expected '$current', got '$actual'")
        }
    }

    override def encode(a: A): String = {
      val field :: HNil = generic.value.to(a)
      name(a) + "." + fields.encode(field)
    }
  }

  implicit def nestedCollection[A, B](implicit
      generic: Lazy[Generic.Aux[A, Int :: B :: HNil]],
      fields: Fields[B]
  ): DerivedRecordFields[A] = new DerivedRecordFields[A] {
    override def decode(value: String): Either[String, A] = (value.indexOf("["), value.indexOf(']')) match {
      case (-1, _) => "Invalid format: missing opening index bracket '['".asLeft
      case (_, -1) => "Invalid format: missing closing index bracket ']'".asLeft
      case (open, close) =>
        if (value.length <= close + 1) s"Invalid format: missing field after collection index".asLeft
        else {
          value.substring(open + 1, close).toIntOption.toRight("Invalid format: index is not a valid Int").flatMap {
            index =>
              fields.decode(value.substring(close + 2)).map(b => generic.value.from(index :: b :: HNil))
          }
        }
    }

    override def encode(value: A): String = {
      val index :: field :: HNil = generic.value.to(value)
      s"${name(value)}[$index]." + fields.encode(field)
    }
  }
}
