package io.taig.taigless.validation

import scala.util.Try

import cats.data.{Validated, ValidatedNel}
import cats.implicits._

trait Validation[A] {
  protected def message: Validation.Message[A]

  def between[B](from: B, to: B)(value: B)(implicit numeric: Numeric[B]): ValidatedNel[A, Unit] =
    Validated.condNel(numeric.gteq(value, from) && numeric.lteq(value, to), (), message.between(from, to, value))

  def bigDecimal(value: String): ValidatedNel[A, BigDecimal] =
    unsafeNumber(BigDecimal(_), message.bigDecimal(value))(value)

  def bigInt(value: String): ValidatedNel[A, BigInt] = unsafeNumber(BigInt(_), message.bigInt(value))(value)

  def compare[B, C](reference: C, f: (C, C) => Boolean, message: => B)(value: C): ValidatedNel[B, Unit] =
    Validated.condNel(f(value, reference), (), message)

  def contains(reference: String)(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value contains reference, (), message.contains(reference, value))

  def defined[B](value: Option[B]): ValidatedNel[A, B] = value.toValidNel(message.defined)

  def double(value: String): ValidatedNel[A, Double] = unsafeNumber(_.toDouble, message.double(value))(value)

  private val ValidEmail = """^.+@.+\..+$""".r.pattern

  def email(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(ValidEmail.matcher(value).matches(), (), message.email(value))

  def endsWith(reference: String)(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value.endsWith(reference), (), message.endsWith(reference, value))

  def float(value: String): ValidatedNel[A, Float] = unsafeNumber(_.toFloat, message.float(value))(value)

  def gt[B](reference: B)(value: B)(implicit numeric: Numeric[B]): ValidatedNel[A, Unit] =
    compare(reference, numeric.gt, message.gt(reference, value))(value)

  def gteq[B](reference: B)(value: B)(implicit numeric: Numeric[B]): ValidatedNel[A, Unit] =
    compare(reference, numeric.gteq, message.gteq(reference, value))(value)

  def int(value: String): ValidatedNel[A, Int] = unsafeNumber(_.toInt, message.int(value))(value)

  def length(reference: Int)(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value.length == reference, (), message.length(reference, value))

  def long(value: String): ValidatedNel[A, Long] = unsafeNumber(_.toLong, message.long(value))(value)

  def lt[B](reference: B)(value: B)(implicit numeric: Numeric[B]): ValidatedNel[A, Unit] =
    compare(reference, numeric.lt, message.lt(reference, value))(value)

  def lteq[B](reference: B)(value: B)(implicit numeric: Numeric[B]): ValidatedNel[A, Unit] =
    compare(reference, numeric.lteq, message.lteq(reference, value))(value)

  def matches(regex: String)(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value.matches(regex), (), message.matches(regex, value))

  def maxLength(reference: Int)(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value.length <= reference, (), message.maxLength(reference, value))

  def minLength(reference: Int)(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value.length >= reference, (), message.minLength(reference, value))

  def nonEmpty(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value.nonEmpty, (), message.nonEmpty(value))

  private val Whitespace = "\\s".r

  def noSpaces(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(Whitespace.findFirstIn(value).isEmpty, (), message.noSpaces(value))

  private def unsafeNumber[B](parse: String => B, message: => A)(value: String): ValidatedNel[A, B] =
    number(value => Try(parse(value)).toOption, message)(value)

  def number[B](parse: String => Option[B], message: => A)(value: String): ValidatedNel[A, B] =
    parse(value).toValidNel(message)

  def optional[B, C](value: Option[B])(f: B => ValidatedNel[A, C]): ValidatedNel[A, Option[C]] =
    value.fold[ValidatedNel[A, Option[C]]](Validated.validNel(None))(f(_).map(_.some))

  def required(value: String): ValidatedNel[A, String] = {
    val trimmed = value.trim
    nonEmpty(trimmed).as(trimmed)
  }

  def short(value: String): ValidatedNel[A, Short] = unsafeNumber(_.toShort, message.short(value))(value)

  def startsWith(reference: String)(value: String): ValidatedNel[A, Unit] =
    Validated.condNel(value.startsWith(reference), (), message.startsWith(reference, value))
}

object Validation extends Validation[String] {
  trait Message[A] {
    def between[B: Numeric](from: B, to: B, value: B): A

    def bigDecimal(value: String): A

    def bigInt(value: String): A

    def contains(reference: String, value: String): A

    def defined: A

    def double(value: String): A

    def email(value: String): A

    def endsWith(reference: String, value: String): A

    def float(value: String): A

    def gt[B: Numeric](reference: B, value: B): A

    def gteq[B: Numeric](reference: B, value: B): A

    def int(value: String): A

    def length(reference: Int, value: String): A

    def long(value: String): A

    def lt[B: Numeric](reference: B, value: B): A

    def lteq[B: Numeric](reference: B, value: B): A

    def matches(regex: String, value: String): A

    def maxLength(reference: Int, value: String): A

    def minLength(reference: Int, value: String): A

    def nonEmpty(value: String): A

    def noSpaces(value: String): A

    def short(value: String): A

    def startsWith(reference: String, value: String): A
  }

  override protected val message: Message[String] = new Message[String] {
    val InvalidNumberFormat = "Invalid number format"

    val Required = "Required"

    def compare[A](operator: String, reference: A) = s"Must be $operator $reference"

    def length(operator: String, reference: Int) = s"$operator $reference characters"

    override def between[B: Numeric](from: B, to: B, value: B): String = s"Must be between $from and $to"

    override def bigDecimal(value: String): String = InvalidNumberFormat

    override def bigInt(value: String): String = InvalidNumberFormat

    override def contains(reference: String, value: String): String = s"Must contain '$reference'"

    override def defined: String = Required

    override def double(value: String): String = InvalidNumberFormat

    override def email(value: String): String = "Not a valid email address"

    override def endsWith(reference: String, value: String): String = s"Must end with '$reference'"

    override def float(value: String): String = InvalidNumberFormat

    override def gt[B: Numeric](reference: B, value: B): String = compare(">", reference)

    override def gteq[B: Numeric](reference: B, value: B): String = compare(">=", reference)

    override def int(value: String): String = InvalidNumberFormat

    override def length(reference: Int, value: String): String = length("Exactly", reference)

    override def long(value: String): String = InvalidNumberFormat

    override def lt[B: Numeric](reference: B, value: B): String = compare("<", reference)

    override def lteq[B: Numeric](reference: B, value: B): String = compare("<=", reference)

    override def matches(regex: String, value: String): String = "Invalid format"

    override def maxLength(reference: Int, value: String): String = length("At most", reference)

    override def minLength(reference: Int, value: String): String = length("At least", reference)

    override def nonEmpty(value: String): String = Required

    override def noSpaces(value: String): String = "May not contain whitespace"

    override def short(value: String): String = InvalidNumberFormat

    override def startsWith(reference: String, value: String): String = s"Must start with '$reference'"
  }
}
