/*
 * Copyright © 2015 Tapack, and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Research Projects is dual-licensed under the GNU General Public
 * License, version 2.0 (GPLv2) and the Tapack Commercial License.
 *
 * Solely for non-commercial purposes. A purpose is non-commercial only if
 * it is in no manner primarily intended for or directed toward commercial
 * advantage or private monetary compensation.
 *
 * This Tapack Software is supplied to you by Tapack in consideration of your
 * agreement to the following terms, and your use, installation, modification
 * or redistribution of this Tapack Software constitutes acceptance of these
 * terms. If you do not agree with these terms, please do not use, install,
 * modify or redistribute this Tapack Software.
 *
 * Neither the name, trademarks, service marks or logos of Tapack may be used
 * to endorse or promote products derived from the Tapack Software without
 * specific prior written permission from Tapack.
 *
 * The Tapack Software is provided by Tapack on an "AS IS" basis. TAPACK MAKES NO
 * WARRANTIES, EXPRESS  OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 * WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, REGARDING THE TAPACK SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
 * COMBINATION WITH YOUR PRODUCTS.
 *
 * IN NO EVENT SHALL TAPACK BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION
 * AND/OR DISTRIBUTION OF THE TAPACK SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER
 * THEORY OF CONTRACT, TORT (INCLUDING NEGLIGENCE), STRICT LIABILITY OR
 * OTHERWISE, EVEN IF TAPACK HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * A copy of the GNU General Public License is included in the distribution in
 * the file LICENSE and at
 *
 *     http://www.gnu.org/licenses/gpl-2.0.html
 *
 * If you are using the Research Projects for commercial purposes, we
 * encourage you to visit
 *
 *     http://products.tapack.io/license
 *
 * for more details.
 *
 * This software or hardware and documentation may provide access to
 * or information on content, products, and services from third parties.
 * Tapack and its affiliates are not responsible for and expressly disclaim
 * all warranties of any kind with respect to third-party content, products,
 * and services. Tapack and its affiliates will not be responsible for any loss,
 * costs, or damages incurred due to your access to or use of third-party
 * content, products, or services. If a third-party content exists, the
 * additional copyright notices and license terms applicable to portions of the
 * software are set forth in the THIRD_PARTY_LICENSE_README file.
 *
 * Please contact Tapack or visit www.tapack.io if you need additional
 * information or have any questions.
 */

package io.tapack.satisfy.pdf;

import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.parser.PdfTextExtractor;
import org.apache.commons.lang.StringUtils;
import org.junit.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;

public class PDFWords {

    private static final Logger LOGGER = LoggerFactory.getLogger(PDFWords.class);
    private Map<Integer, String> pdfData;

    public void pdfShouldEqualTo(PDFWords expectedPDFcontent) {
        assertThat(pdfData, equalTo(expectedPDFcontent.pdfData));
    }

    public void parsePdf(String filename) throws IOException {
        PdfReader reader = new PdfReader(filename);
        LOGGER.trace("Reading file " + filename);
        pdfData = new HashMap<Integer, String>();
        int numberOfPages = reader.getNumberOfPages();
        for (int page = 1; page <= numberOfPages; page++) {
            LOGGER.trace("Reading page " + page);
            String textFromPage = PdfTextExtractor.getTextFromPage(reader, page);
            pdfData.put(page, textFromPage);
        }
    }

    public void pdfShouldContainUnOrdered(PDFWords expectedPDFcontent) {
        Collection<String> values = pdfData.values();
        for (String value : values) {
            expectedPDFcontent.pdfShouldContain(value);
        }
    }

    public void pdfShouldContain(String expectedValue) {
        Collection<String> values = pdfData.values();
        collectionShouldContain(expectedValue, values);
    }

    public void pdfShouldNotContain(String expectedValue) {
        Collection<String> values = pdfData.values();
        collectionShouldNotContain(expectedValue, values);
    }

    public void pdfShouldContainOccurrencesOfString(String expectedValue, int expectedCount) {
        Collection<String> values = pdfData.values();
        Assert.assertEquals("THE MESSAGE '" + expectedValue + "' EXPECTED " + expectedCount + " BUT IN REALY = " + getCountMatchesInCollectiont(expectedValue, values), expectedCount, getCountMatchesInCollectiont(expectedValue, values));
    }

    public void pdfShouldContain(String expectedValue, String ignoreLinebreaks) {
        if (StringUtils.isEmpty(ignoreLinebreaks))
            pdfShouldContain(expectedValue);
        Collection<String> values = pdfData.values();
        values = removeLinebreaks(values);
        collectionShouldContain(expectedValue, values);
    }

    public void pdfShouldContain(String expectedValue, String ignoreLinebreaks, String ignoreCase) {
        if (StringUtils.isEmpty(ignoreCase))
            pdfShouldContain(expectedValue, ignoreLinebreaks);
        Collection<String> values = pdfData.values();
        values = removeLinebreaks(values);
        collectionShouldContain(expectedValue, values, StringUtils.isNotEmpty(ignoreCase));
    }

    protected Collection<String> removeLinebreaks(Collection<String> values) {
        List<String> contentWithoutLinebreacks = new ArrayList<String>();
        for (String string : values) {
            String reformattedString = string.replaceAll("\\n", " ");
            while (reformattedString.contains("  ")) {
                reformattedString = reformattedString.replaceAll("  ", " ");
            }
            contentWithoutLinebreacks.add(reformattedString);
        }
        return contentWithoutLinebreacks;
    }

    private boolean isContaining(String expectedValue, boolean ignoreCase, String content) {
        return ignoreCase ? StringUtils.containsIgnoreCase(content, expectedValue) : StringUtils.contains(content, expectedValue);
    }

    private void collectionShouldContain(String expectedValue, Collection<String> values) {
        collectionShouldContain(expectedValue, values, false);
    }

    private void collectionShouldContain(String expectedValue, Collection<String> values, boolean ignoreCase) {
        for (String content : values) {
            if (isContaining(expectedValue, ignoreCase, content)) {
                return;
            }
        }
        throw new AssertionError("COULD NOT FIND " + expectedValue + " IN " + pdfData);
    }

    private void collectionShouldNotContain(String expectedValue, Collection<String> values, boolean ignoreCase) {
        for (String content : values) {
            if (isContaining(expectedValue, ignoreCase, content)) {
                throw new AssertionError("TEXT '" + expectedValue + "' CONTAINS IN " + pdfData);
            }
        }
    }

    private int getCountMatchesInCollectiont(String expectedValue, Collection<String> values) {
        int count = 0;
        for (String content : values) {
            count += StringUtils.countMatches(content, expectedValue);
        }
        return count;
    }

    private void collectionShouldNotContain(String expectedValue, Collection<String> values) {
        collectionShouldNotContain(expectedValue, values, false);
    }
}
