/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2020 TileDB, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package io.tiledb.java.api;

import io.tiledb.libtiledb.*;
import java.math.BigInteger;

public class Group implements AutoCloseable {
  private Context ctx;
  private final String uri;
  private QueryType queryType;
  private SWIGTYPE_p_tiledb_group_t groupp;
  private SWIGTYPE_p_p_tiledb_group_t grouppp;

  public Group(Context ctx, String uri, QueryType queryType) throws TileDBError {
    SWIGTYPE_p_p_tiledb_group_t grouppp = tiledb.new_tiledb_group_tpp();
    try {
      ctx.handleError(tiledb.tiledb_group_alloc(ctx.getCtxp(), uri, grouppp));
    } catch (TileDBError err) {
      tiledb.delete_tiledb_group_tpp(grouppp);
      throw err;
    }
    this.ctx = ctx;
    this.groupp = tiledb.tiledb_group_tpp_value(grouppp);
    this.grouppp = grouppp;
    this.uri = uri;
    this.queryType = queryType;
    open(queryType);
  }

  /**
   * Gets group pointer.
   *
   * @return the group pointer
   */
  protected SWIGTYPE_p_tiledb_group_t getGroupp() {
    return this.groupp;
  }

  /**
   * Gets the Context.
   *
   * @return the context
   */
  protected Context getCtx() {
    return this.ctx;
  }

  /**
   * Sets the group config.
   *
   * @param config the configuration to set.
   * @throws TileDBError
   */
  public void setConfig(Config config) throws TileDBError {
    try {
      ctx.handleError(
          tiledb.tiledb_group_set_config(ctx.getCtxp(), getGroupp(), config.getConfigp()));
    } catch (TileDBError err) {
      throw err;
    }
  }

  /**
   * It deletes a metadata key-value item from an open group. The group must be opened in WRITE
   * mode, otherwise the function will error out.
   *
   * @param key the key to delete.
   * @throws TileDBError
   */
  public void deleteMetadata(String key) throws TileDBError {
    if (!isOpen()) throw new TileDBError("Group with URI: " + uri + " is closed");
    ctx.handleError(tiledb.tiledb_group_delete_metadata(ctx.getCtxp(), getGroupp(), key));
  }

  /**
   * It gets a metadata key-value item from an open group. The group must be opened in READ mode,
   * otherwise the function will error out.
   *
   * @param key the key
   * @param nativeType the datatype of the value.
   * @return
   * @throws TileDBError
   */
  public NativeArray getMetadata(String key, Datatype nativeType) throws TileDBError {
    if (!isOpen()) throw new TileDBError("Group with URI: " + uri + " is closed");
    SWIGTYPE_p_p_void resultArrpp = tiledb.new_voidpArray(0);
    SWIGTYPE_p_unsigned_int value_num = tiledb.new_uintp();
    SWIGTYPE_p_tiledb_datatype_t value_type =
        (nativeType == null)
            ? tiledb.new_tiledb_datatype_tp()
            : tiledb.copy_tiledb_datatype_tp(nativeType.toSwigEnum());

    ctx.handleError(
        tiledb.tiledb_group_get_metadata(
            ctx.getCtxp(), getGroupp(), key, value_type, value_num, resultArrpp));

    Datatype derivedNativeType = Datatype.fromSwigEnum(tiledb.tiledb_datatype_tp_value(value_type));

    long value = tiledb.uintp_value(value_num);
    NativeArray result = new NativeArray(ctx, derivedNativeType, resultArrpp, (int) value);

    tiledb.delete_uintp(value_num);
    tiledb.delete_tiledb_datatype_tp(value_type);

    return result;
  }

  /**
   * Gets the group config.
   *
   * @return the group config.
   * @throws TileDBError
   */
  public Config getConfig() throws TileDBError {
    SWIGTYPE_p_p_tiledb_config_t configpp = tiledb.new_tiledb_config_tpp();
    try {
      ctx.handleError(tiledb.tiledb_group_get_config(ctx.getCtxp(), getGroupp(), configpp));
    } catch (TileDBError err) {
      tiledb.delete_tiledb_config_tpp(configpp);
    }
    return new Config(configpp);
  }

  /**
   * Gets the number of metadata items in an open group. The array must be opened in READ mode,
   * otherwise the function will error out.
   *
   * @return the number of metadata items
   * @throws TileDBError A TileDB exception
   */
  public BigInteger getMetadataNum() throws TileDBError {
    if (!isOpen()) throw new TileDBError("Group with URI: " + uri + " is closed");
    SWIGTYPE_p_unsigned_long_long value_num = tiledb.new_ullp();
    ctx.handleError(tiledb.tiledb_group_get_metadata_num(ctx.getCtxp(), getGroupp(), value_num));
    BigInteger value = tiledb.ullp_value(value_num);
    tiledb.delete_ullp(value_num);

    return value;
  }

  /**
   * Get the count of members in a group.
   *
   * @return the number of members in a group.
   * @throws TileDBError
   */
  public long getMemberCount() throws TileDBError {
    SWIGTYPE_p_unsigned_long_long mc = tiledb.new_ullp();
    try {
      ctx.handleError(tiledb.tiledb_group_get_member_count(ctx.getCtxp(), getGroupp(), mc));
    } catch (TileDBError err) {
      tiledb.delete_ullp(mc);
      throw err;
    }
    long memberCount = tiledb.ullp_value(mc).longValue();
    tiledb.delete_ullp(mc);
    return memberCount;
  }

  /**
   * Get the URI of a member of a group by index and details of group
   *
   * @param index the index of the member.
   * @return the corresponding member in the group.
   * @throws TileDBError
   */
  public String getMemberURIByIndex(BigInteger index) throws TileDBError {
    Util.checkBigIntegerRange(index);
    SWIGTYPE_p_tiledb_object_t objtypep = tiledb.new_tiledb_object_tp();
    SWIGTYPE_p_p_char uripp = tiledb.new_charpp();
    SWIGTYPE_p_p_char namepp = tiledb.new_charpp(); // useless in this method
    ctx.handleError(
        tiledb.tiledb_group_get_member_by_index(
            ctx.getCtxp(), getGroupp(), index, uripp, objtypep, uripp));
    return tiledb.charpp_value(uripp);
  }

  /**
   * Get the URI of a member of a group by name
   *
   * @param name the name of the member
   * @return the URI of the member with the given name
   * @throws TileDBError
   */
  public String getMemberURIByName(String name) throws TileDBError {
    SWIGTYPE_p_tiledb_object_t objtypep = tiledb.new_tiledb_object_tp();
    SWIGTYPE_p_p_char uripp = tiledb.new_charpp();
    ctx.handleError(
        tiledb.tiledb_group_get_member_by_name(ctx.getCtxp(), getGroupp(), name, uripp, objtypep));
    return tiledb.charpp_value(uripp);
  }

  /**
   * Get the name of a member of a group by index and details of group
   *
   * @param index the index of the member.
   * @return the corresponding member in the group.
   * @throws TileDBError
   */
  public String getMemberNameByIndex(BigInteger index) throws TileDBError {
    Util.checkBigIntegerRange(index);
    SWIGTYPE_p_tiledb_object_t objtypep = tiledb.new_tiledb_object_tp();
    SWIGTYPE_p_p_char uripp = tiledb.new_charpp(); // useless in this method
    SWIGTYPE_p_p_char namepp = tiledb.new_charpp();
    ctx.handleError(
        tiledb.tiledb_group_get_member_by_index(
            ctx.getCtxp(), getGroupp(), index, uripp, objtypep, namepp));
    return tiledb.charpp_value(namepp);
  }

  /**
   * Add a member to a group.
   *
   * @param uri The uri of the member to add.
   * @param relative is the URI relative to the group.
   * @param name name of member, The caller takes ownership of the c-string. NULL if name was not
   *     set
   * @throws TileDBError
   */
  public void addMember(String uri, boolean relative, String name) throws TileDBError {
    ctx.handleError(
        tiledb.tiledb_group_add_member(
            ctx.getCtxp(), getGroupp(), uri, relative ? (short) 1 : (short) 0, name));
  }

  /**
   * Remove a member from a group.
   *
   * @param uri The URI of the member to remove.
   * @throws TileDBError
   */
  public void removeMember(String uri) throws TileDBError {
    ctx.handleError(tiledb.tiledb_group_remove_member(ctx.getCtxp(), getGroupp(), uri));
  }

  /**
   * Dump a string representation of a group
   *
   * @param string The string.
   * @param flag is recursive.
   * @throws TileDBError
   */
  public void dumpStr(String string, boolean flag) throws TileDBError {
    SWIGTYPE_p_p_char valuepp = tiledb.new_charpp();
    ctx.handleError(
        tiledb.tiledb_group_dump_str(
            ctx.getCtxp(), getGroupp(), valuepp, flag ? (short) 1 : (short) 0));
  }

  /**
   * Gets a metadata item from an open group using an index. The array must be opened in READ mode,
   * otherwise the function will error out.
   *
   * @param index index to retrieve metadata from
   * @return a pair, key and the metadata
   * @throws TileDBError A TileDB exception
   */
  public Pair<String, NativeArray> getMetadataFromIndex(BigInteger index) throws TileDBError {
    Util.checkBigIntegerRange(index);
    if (!isOpen()) throw new TileDBError("Group with URI: " + uri + " is closed");
    SWIGTYPE_p_p_char key = tiledb.new_charpp();
    SWIGTYPE_p_unsigned_int key_len = tiledb.new_uintp();
    SWIGTYPE_p_tiledb_datatype_t value_type = tiledb.new_tiledb_datatype_tp();
    SWIGTYPE_p_unsigned_int value_num = tiledb.new_uintp();
    SWIGTYPE_p_p_void value = tiledb.new_voidpArray(0);

    ctx.handleError(
        tiledb.tiledb_group_get_metadata_from_index(
            ctx.getCtxp(), getGroupp(), index, key, key_len, value_type, value_num, value));

    String keyString = tiledb.charpp_value(key);
    long valueLength = tiledb.uintp_value(value_num);
    Datatype nativeType = Datatype.fromSwigEnum(tiledb.tiledb_datatype_tp_value(value_type));

    NativeArray result = new NativeArray(ctx, nativeType, value, (int) valueLength);

    tiledb.delete_uintp(value_num);
    tiledb.delete_uintp(key_len);
    tiledb.delete_charpp(key);
    tiledb.delete_tiledb_datatype_tp(value_type);

    return new Pair<String, NativeArray>(keyString, result);
  }

  /**
   * Checks if the key is present in the group metadata. The array must be opened in READ mode,
   * otherwise the function will error out.
   *
   * @param key a key to retrieve from the metadata key-value
   * @return true if the key is present in the metadata, false if it is not
   * @throws TileDBError A TileDB exception
   */
  public Boolean hasMetadataKey(String key) throws TileDBError {
    if (!isOpen()) throw new TileDBError("Group with URI: " + uri + " is closed");
    SWIGTYPE_p_tiledb_datatype_t value_type = tiledb.new_tiledb_datatype_tp();
    SWIGTYPE_p_int has_key = tiledb.new_intp();

    ctx.handleError(
        tiledb.tiledb_group_has_metadata_key(ctx.getCtxp(), getGroupp(), key, value_type, has_key));

    Boolean result = tiledb.intp_value(has_key) > 0;

    tiledb.delete_intp(has_key);
    tiledb.delete_tiledb_datatype_tp(value_type);

    return result;
  }

  /**
   * Check if the group is open.
   *
   * @return True if the group is open.
   * @throws TileDBError
   */
  public boolean isOpen() throws TileDBError {
    boolean isOpen;
    SWIGTYPE_p_int ret = tiledb.new_intp();
    try {
      ctx.handleError(tiledb.tiledb_group_is_open(ctx.getCtxp(), getGroupp(), ret));
      isOpen = tiledb.intp_value(ret) != 0;
    } finally {
      tiledb.delete_intp(ret);
    }
    return isOpen;
  }

  /**
   * * It puts a metadata key-value item to an open group. The group must * be opened in WRITE mode,
   * otherwise the function will error out.
   *
   * @param key The key of the metadata item to be added.
   * @param value The metadata value.
   * @throws TileDBError
   */
  public void putMetadata(String key, NativeArray value) throws TileDBError {
    if (!isOpen()) throw new TileDBError("Group with URI: " + uri + " is closed");
    ctx.handleError(
        tiledb.tiledb_group_put_metadata(
            ctx.getCtxp(),
            getGroupp(),
            key,
            value.getNativeType().toSwigEnum(),
            value.getSize(),
            value.toVoidPointer()));
  }

  /**
   * Get a member of a group by name and relative characteristic of that name
   *
   * @param name name of member to fetch
   * @return True if relative
   * @throws TileDBError
   */
  public boolean getIsRelativeURIByName(String name) throws TileDBError {
    try {
      NativeArray arr = new NativeArray(ctx, 1, Datatype.TILEDB_UINT8);
      SWIGTYPE_p_unsigned_char relative = arr.getUint8_tArray().cast();
      ctx.handleError(
          tiledb.tiledb_group_get_is_relative_uri_by_name(ctx.getCtxp(), groupp, name, relative));

      return ((short) arr.getItem(0) == 1);
    } catch (TileDBError err) {
      throw err;
    }
  }

  /**
   * Cleans up the group metadata Note that this will coarsen the granularity of time traveling (see
   * docs for more information).
   *
   * @param config Configuration parameters for the vacuuming. (`null` means default, which will use
   *     the config from `ctx`).
   * @throws TileDBError
   */
  public void vacuumMetadata(Config config) throws TileDBError {
    SWIGTYPE_p_tiledb_config_t configp = null;
    try {
      configp = config.getConfigp();
    } catch (NullPointerException e) {
      // using null/default
    }
    ctx.handleError(tiledb.tiledb_group_vacuum_metadata(ctx.getCtxp(), uri, configp));
  }

  /**
   * Consolidates the group metadata into a single group metadata file.
   *
   * @param config Configuration parameters for the vacuuming. (`null` means default, which will use
   *     the config from `ctx`).
   * @throws TileDBError
   */
  public void consolidateMetadata(Config config) throws TileDBError {
    SWIGTYPE_p_tiledb_config_t configp = null;
    try {
      configp = config.getConfigp();
    } catch (NullPointerException e) {
      // using null/default
    }
    ctx.handleError(tiledb.tiledb_group_consolidate_metadata(ctx.getCtxp(), uri, configp));
  }

  /** Close resources */
  public void close() {
    if (groupp != null && grouppp != null) {
      tiledb.tiledb_group_close(ctx.getCtxp(), groupp);
      tiledb.tiledb_group_free(grouppp);
      grouppp = null;
      groupp = null;
    }
  }

  /**
   * Returns the URI of the group
   *
   * @return The URI of the group.
   * @exception TileDBError A TileDB exception
   */
  public String getUri() throws TileDBError {
    return uri;
  }

  /**
   * Reopens a TileDB group (the group must be already open). This is useful when the group got
   * updated after it got opened or when the user wants to reopen with a different queryType.
   *
   * @param queryType the queryType that will be used when the group opens.
   * @throws TileDBError
   */
  public void reopen(Context ctx, QueryType queryType) throws TileDBError {
    if (!isOpen()) throw new TileDBError("Can not reopen group. Group is closed");
    ctx.handleError(tiledb.tiledb_group_close(ctx.getCtxp(), groupp));
    this.ctx = ctx;
    open(queryType);
  }

  /**
   * * Opens a TileDB group. The group is opened using a query type as input. * This is to indicate
   * that queries created for this `tiledb_group_t` * object will inherit the query type. In other
   * words, `tiledb_group_t` * objects are opened to receive only one type of queries. * They can
   * always be closed and be re-opened with another query type. * Also there may be many different
   * `tiledb_group_t` * objects created and opened with different query types.
   *
   * @param queryType the query type to open the group.
   * @throws TileDBError
   */
  private void open(QueryType queryType) throws TileDBError {
    try {
      ctx.handleError(tiledb.tiledb_group_open(ctx.getCtxp(), groupp, queryType.toSwigEnum()));
    } catch (TileDBError err) {
      tiledb.delete_tiledb_group_tpp(grouppp);
      throw err;
    }
  }

  /**
   * Creates a new group. A Group is a logical grouping of TileDB objects on the storage system with
   * the sample path prefix.
   *
   * @param ctx The TileDB context.
   * @param uri The group URI.
   * @exception TileDBError A TileDB exception
   */
  public static void create(Context ctx, String uri) throws TileDBError {
    ctx.handleError(tiledb.tiledb_group_create(ctx.getCtxp(), uri));
  }
}
