/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.FieldList;
import com.google.cloud.bigquery.StandardSQLTypeName;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.RowType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimeWithTimeZoneType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nullable;

public final class BigQueryType {
    private static final int[] NANO_FACTOR = new int[]{-1, 100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10, 1};
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("''HH:mm:ss.SSSSSS''");
    private static final DateTimeFormatter DATETIME_FORMATTER = DateTimeFormatter.ofPattern("''yyyy-MM-dd HH:mm:ss.SSSSSS''");

    private BigQueryType() {
    }

    private static RowType.Field toRawTypeField(String name, Field field) {
        Type trinoType = BigQueryType.convertToTrinoType(field).orElseThrow(() -> new IllegalArgumentException("Unsupported column " + field));
        return RowType.field((String)name, (Type)(field.getMode() == Field.Mode.REPEATED ? new ArrayType(trinoType) : trinoType));
    }

    @VisibleForTesting
    public static LocalDateTime toLocalDateTime(String datetime) {
        int dotPosition = datetime.indexOf(46);
        if (dotPosition == -1) {
            return LocalDateTime.from(DateTimeFormatter.ISO_LOCAL_DATE_TIME.parse(datetime));
        }
        LocalDateTime result = LocalDateTime.from(DateTimeFormatter.ISO_LOCAL_DATE_TIME.parse(datetime.substring(0, dotPosition)));
        String nanosStr = datetime.substring(dotPosition + 1);
        int nanoOfSecond = Integer.parseInt(nanosStr) * NANO_FACTOR[nanosStr.length()];
        return result.withNano(nanoOfSecond);
    }

    public static long toTrinoTimestamp(String datetime) {
        Instant instant = BigQueryType.toLocalDateTime(datetime).toInstant(ZoneOffset.UTC);
        return instant.getEpochSecond() * 1000000L + (long)(instant.getNano() / 1000);
    }

    private static String floatToStringConverter(Object value) {
        return String.format("CAST('%s' AS float64)", value);
    }

    private static String simpleToStringConverter(Object value) {
        return String.valueOf(value);
    }

    @VisibleForTesting
    public static String dateToStringConverter(Object value) {
        LocalDate date = LocalDate.ofEpochDay((Long)value);
        return "'" + date + "'";
    }

    private static String datetimeToStringConverter(Object value) {
        long epochMicros = (Long)value;
        long epochSeconds = Math.floorDiv(epochMicros, 1000000);
        int nanoAdjustment = Math.floorMod(epochMicros, 1000000) * 1000;
        return BigQueryType.formatTimestamp(epochSeconds, nanoAdjustment, ZoneOffset.UTC);
    }

    @VisibleForTesting
    public static String timeToStringConverter(Object value) {
        long time = (Long)value;
        Verify.verify((0L <= time ? 1 : 0) != 0, (String)"Invalid time value: %s", (long)time);
        long epochSeconds = time / 1000000000000L;
        long nanoAdjustment = time % 1000000000000L / 1000L;
        return TIME_FORMATTER.format(BigQueryType.toZonedDateTime(epochSeconds, nanoAdjustment, ZoneOffset.UTC));
    }

    @VisibleForTesting
    public static String timestampToStringConverter(Object value) {
        LongTimestampWithTimeZone timestamp = (LongTimestampWithTimeZone)value;
        long epochMillis = timestamp.getEpochMillis();
        long epochSeconds = Math.floorDiv(epochMillis, 1000);
        int nanoAdjustment = Math.floorMod(epochMillis, 1000) * 1000000 + timestamp.getPicosOfMilli() / 1000;
        ZoneId zoneId = TimeZoneKey.getTimeZoneKey((short)timestamp.getTimeZoneKey()).getZoneId();
        return BigQueryType.formatTimestamp(epochSeconds, nanoAdjustment, zoneId);
    }

    private static String formatTimestamp(long epochSeconds, long nanoAdjustment, ZoneId zoneId) {
        return DATETIME_FORMATTER.format(BigQueryType.toZonedDateTime(epochSeconds, nanoAdjustment, zoneId));
    }

    public static ZonedDateTime toZonedDateTime(long epochSeconds, long nanoAdjustment, ZoneId zoneId) {
        Instant instant = Instant.ofEpochSecond(epochSeconds, nanoAdjustment);
        return ZonedDateTime.ofInstant(instant, zoneId);
    }

    static String stringToStringConverter(Object value) {
        Slice slice = (Slice)value;
        return "'%s'".formatted(slice.toStringUtf8().replace("\\", "\\\\").replace("\n", "\\n").replace("'", "\\'"));
    }

    static String numericToStringConverter(Object value) {
        return Decimals.toString((Int128)((Int128)value), (int)9);
    }

    static String bytesToStringConverter(Object value) {
        Slice slice = (Slice)value;
        return String.format("FROM_BASE64('%s')", Base64.getEncoder().encodeToString(slice.getBytes()));
    }

    public static Field toField(String name, Type type, @Nullable String comment) {
        if (type instanceof ArrayType) {
            Type elementType = ((ArrayType)type).getElementType();
            return BigQueryType.toInnerField(name, elementType, true, comment);
        }
        return BigQueryType.toInnerField(name, type, false, comment);
    }

    private static Field toInnerField(String name, Type type, boolean repeated, @Nullable String comment) {
        Field.Builder builder = type instanceof RowType ? Field.newBuilder((String)name, (StandardSQLTypeName)StandardSQLTypeName.STRUCT, (FieldList)BigQueryType.toFieldList((RowType)type)).setDescription(comment) : Field.newBuilder((String)name, (StandardSQLTypeName)BigQueryType.toStandardSqlTypeName(type), (Field[])new Field[0]).setDescription(comment);
        if (repeated) {
            builder = builder.setMode(Field.Mode.REPEATED);
        }
        return builder.build();
    }

    private static FieldList toFieldList(RowType rowType) {
        ImmutableList.Builder fields = ImmutableList.builder();
        for (RowType.Field field : rowType.getFields()) {
            String fieldName = (String)field.getName().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "ROW type does not have field names declared: " + rowType));
            fields.add((Object)BigQueryType.toField(fieldName, field.getType(), null));
        }
        return FieldList.of((Iterable)fields.build());
    }

    private static StandardSQLTypeName toStandardSqlTypeName(Type type) {
        if (type == BooleanType.BOOLEAN) {
            return StandardSQLTypeName.BOOL;
        }
        if (type == TinyintType.TINYINT || type == SmallintType.SMALLINT || type == IntegerType.INTEGER || type == BigintType.BIGINT) {
            return StandardSQLTypeName.INT64;
        }
        if (type == DoubleType.DOUBLE) {
            return StandardSQLTypeName.FLOAT64;
        }
        if (type instanceof DecimalType) {
            return StandardSQLTypeName.NUMERIC;
        }
        if (type == DateType.DATE) {
            return StandardSQLTypeName.DATE;
        }
        if (type == TimeWithTimeZoneType.createTimeWithTimeZoneType((int)3)) {
            return StandardSQLTypeName.TIME;
        }
        if (type == TimestampType.TIMESTAMP_MICROS) {
            return StandardSQLTypeName.DATETIME;
        }
        if (type == TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS) {
            return StandardSQLTypeName.TIMESTAMP;
        }
        if (type instanceof VarcharType) {
            return StandardSQLTypeName.STRING;
        }
        if (type == VarbinaryType.VARBINARY) {
            return StandardSQLTypeName.BYTES;
        }
        if (type instanceof ArrayType) {
            return StandardSQLTypeName.ARRAY;
        }
        if (type instanceof RowType) {
            return StandardSQLTypeName.STRUCT;
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported column type: " + type.getDisplayName());
    }

    public static Optional<String> convertToString(Type type, StandardSQLTypeName bigqueryType, Object value) {
        if (type instanceof ArrayType) {
            return Optional.empty();
        }
        switch (bigqueryType) {
            case BOOL: {
                return Optional.of(BigQueryType.simpleToStringConverter(value));
            }
            case BYTES: {
                return Optional.of(BigQueryType.bytesToStringConverter(value));
            }
            case DATE: {
                return Optional.of(BigQueryType.dateToStringConverter(value));
            }
            case DATETIME: {
                return Optional.of(BigQueryType.datetimeToStringConverter(value));
            }
            case FLOAT64: {
                return Optional.of(BigQueryType.floatToStringConverter(value));
            }
            case INT64: {
                return Optional.of(BigQueryType.simpleToStringConverter(value));
            }
            case NUMERIC: 
            case BIGNUMERIC: {
                String bigqueryTypeName = bigqueryType.name();
                DecimalType decimalType = (DecimalType)type;
                if (decimalType.isShort()) {
                    return Optional.of(String.format("%s '%s'", bigqueryTypeName, Decimals.toString((long)((Long)value), (int)((DecimalType)type).getScale())));
                }
                return Optional.of(String.format("%s '%s'", bigqueryTypeName, Decimals.toString((Int128)((Int128)value), (int)((DecimalType)type).getScale())));
            }
            case ARRAY: 
            case STRUCT: 
            case GEOGRAPHY: {
                return Optional.empty();
            }
            case STRING: {
                return Optional.of(BigQueryType.stringToStringConverter(value));
            }
            case TIME: {
                return Optional.of(BigQueryType.timeToStringConverter(value));
            }
            case TIMESTAMP: {
                return Optional.of(BigQueryType.timestampToStringConverter(value));
            }
        }
        throw new IllegalArgumentException("Unsupported type: " + bigqueryType);
    }

    public static Optional<Type> toTrinoType(Field field) {
        return BigQueryType.convertToTrinoType(field).map(type -> field.getMode() == Field.Mode.REPEATED ? new ArrayType(type) : type);
    }

    private static Optional<Type> convertToTrinoType(Field field) {
        switch (field.getType().getStandardType()) {
            case BOOL: {
                return Optional.of(BooleanType.BOOLEAN);
            }
            case INT64: {
                return Optional.of(BigintType.BIGINT);
            }
            case FLOAT64: {
                return Optional.of(DoubleType.DOUBLE);
            }
            case NUMERIC: 
            case BIGNUMERIC: {
                Long precision = field.getPrecision();
                Long scale = field.getScale();
                if (precision != null && scale != null) {
                    return Optional.of(DecimalType.createDecimalType((int)Math.toIntExact(precision), (int)Math.toIntExact(scale)));
                }
                if (precision != null) {
                    return Optional.of(DecimalType.createDecimalType((int)Math.toIntExact(precision)));
                }
                return Optional.of(DecimalType.createDecimalType((int)38, (int)9));
            }
            case STRING: {
                return Optional.of(VarcharType.createUnboundedVarcharType());
            }
            case BYTES: {
                return Optional.of(VarbinaryType.VARBINARY);
            }
            case DATE: {
                return Optional.of(DateType.DATE);
            }
            case DATETIME: {
                return Optional.of(TimestampType.TIMESTAMP_MICROS);
            }
            case TIME: {
                return Optional.of(TimeType.TIME_MICROS);
            }
            case TIMESTAMP: {
                return Optional.of(TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS);
            }
            case GEOGRAPHY: {
                return Optional.of(VarcharType.VARCHAR);
            }
            case STRUCT: {
                FieldList subTypes = field.getSubFields();
                Preconditions.checkArgument((!subTypes.isEmpty() ? 1 : 0) != 0, (Object)"a record or struct must have sub-fields");
                List fields = subTypes.stream().map(subField -> BigQueryType.toRawTypeField(subField.getName(), subField)).collect(Collectors.toList());
                RowType rowType = RowType.from(fields);
                return Optional.of(rowType);
            }
        }
        return Optional.empty();
    }
}

