/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.common.base.Strings;
import io.airlift.units.Duration;
import io.trino.Session;
import io.trino.plugin.bigquery.BigQueryQueryRunner;
import io.trino.spi.QueryId;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import io.trino.sql.planner.plan.FilterNode;
import io.trino.sql.query.QueryAssertions;
import io.trino.testing.BaseConnectorTest;
import io.trino.testing.MaterializedResult;
import io.trino.testing.MaterializedResultWithQueryId;
import io.trino.testing.QueryRunner;
import io.trino.testing.TestingConnectorBehavior;
import io.trino.testing.TestingNames;
import io.trino.testing.assertions.Assert;
import io.trino.testing.sql.SqlExecutor;
import io.trino.testing.sql.TestTable;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import org.assertj.core.api.AssertProvider;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ObjectAssert;
import org.intellij.lang.annotations.Language;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
@Execution(value=ExecutionMode.CONCURRENT)
public abstract class BaseBigQueryConnectorTest
extends BaseConnectorTest {
    protected BigQueryQueryRunner.BigQuerySqlExecutor bigQuerySqlExecutor;
    private String gcpStorageBucket;

    @BeforeAll
    public void initBigQueryExecutor() {
        this.bigQuerySqlExecutor = new BigQueryQueryRunner.BigQuerySqlExecutor();
        this.gcpStorageBucket = System.getProperty("testing.gcp-storage-bucket");
    }

    protected boolean hasBehavior(TestingConnectorBehavior connectorBehavior) {
        return switch (connectorBehavior) {
            case TestingConnectorBehavior.SUPPORTS_TRUNCATE -> true;
            case TestingConnectorBehavior.SUPPORTS_ADD_COLUMN, TestingConnectorBehavior.SUPPORTS_CREATE_MATERIALIZED_VIEW, TestingConnectorBehavior.SUPPORTS_CREATE_VIEW, TestingConnectorBehavior.SUPPORTS_DEREFERENCE_PUSHDOWN, TestingConnectorBehavior.SUPPORTS_MERGE, TestingConnectorBehavior.SUPPORTS_NEGATIVE_DATE, TestingConnectorBehavior.SUPPORTS_NOT_NULL_CONSTRAINT, TestingConnectorBehavior.SUPPORTS_RENAME_COLUMN, TestingConnectorBehavior.SUPPORTS_RENAME_SCHEMA, TestingConnectorBehavior.SUPPORTS_RENAME_TABLE, TestingConnectorBehavior.SUPPORTS_SET_COLUMN_TYPE, TestingConnectorBehavior.SUPPORTS_TOPN_PUSHDOWN, TestingConnectorBehavior.SUPPORTS_UPDATE -> false;
            default -> super.hasBehavior(connectorBehavior);
        };
    }

    @Test
    public void testShowColumns() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SHOW COLUMNS FROM orders"))).matches(this.getDescribeOrdersResult());
    }

    protected MaterializedResult getDescribeOrdersResult() {
        return MaterializedResult.resultBuilder((Session)this.getSession(), (Type[])new Type[]{VarcharType.VARCHAR, VarcharType.VARCHAR, VarcharType.VARCHAR, VarcharType.VARCHAR}).row(new Object[]{"orderkey", "bigint", "", ""}).row(new Object[]{"custkey", "bigint", "", ""}).row(new Object[]{"orderstatus", "varchar", "", ""}).row(new Object[]{"totalprice", "double", "", ""}).row(new Object[]{"orderdate", "date", "", ""}).row(new Object[]{"orderpriority", "varchar", "", ""}).row(new Object[]{"clerk", "varchar", "", ""}).row(new Object[]{"shippriority", "bigint", "", ""}).row(new Object[]{"comment", "varchar", "", ""}).build();
    }

    @Test
    public void testPredicateReflectedInExplain() {
        this.assertExplain("EXPLAIN SELECT name FROM nation WHERE nationkey = 42", new String[]{"nationkey", "bigint", "42"});
    }

    @Test
    public void testPredicatePushdown() {
        this.testPredicatePushdown("true", "true", true);
        this.testPredicatePushdown("CAST(1 AS INT64)", "1", true);
        this.testPredicatePushdown("CAST(0.1 AS FLOAT64)", "0.1", true);
        this.testPredicatePushdown("NUMERIC '123'", "123", true);
        this.testPredicatePushdown("'string'", "'string'", true);
        this.testPredicatePushdown("b''", "x''", true);
        this.testPredicatePushdown("DATE '2017-01-01'", "DATE '2017-01-01'", true);
        this.testPredicatePushdown("TIME '12:34:56'", "TIME '12:34:56'", true);
        this.testPredicatePushdown("TIMESTAMP '2018-04-01 02:13:55.123456 UTC'", "TIMESTAMP '2018-04-01 02:13:55.123456 UTC'", true);
        this.testPredicatePushdown("DATETIME '2018-04-01 02:13:55.123'", "TIMESTAMP '2018-04-01 02:13:55.123'", true);
        this.testPredicatePushdown("ST_GeogPoint(0, 0)", "'POINT(0 0)'", false);
        this.testPredicatePushdown("JSON '{\"age\": 30}'", "JSON '{\"age\": 30}'", false);
        this.testPredicatePushdown("[true]", "ARRAY[true]", false);
        this.testPredicatePushdown("STRUCT('nested' AS x)", "ROW('nested')", false);
    }

    private void testPredicatePushdown(@Language(value="SQL") String inputLiteral, @Language(value="SQL") String predicateLiteral, boolean isPushdownSupported) {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.test_predicate_pushdown", "AS SELECT %s col".formatted(inputLiteral));){
            String query = "SELECT * FROM " + table.getName() + " WHERE col = " + predicateLiteral;
            if (isPushdownSupported) {
                ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query(query))).isFullyPushedDown();
            } else {
                ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query(query))).isNotFullyPushedDown(FilterNode.class, new Class[0]);
            }
        }
    }

    @Test
    public void testCreateTableUnsupportedType() {
        this.testCreateTableUnsupportedType("json");
        this.testCreateTableUnsupportedType("uuid");
        this.testCreateTableUnsupportedType("ipaddress");
    }

    private void testCreateTableUnsupportedType(String createType) {
        String tableName = String.format("test_create_table_unsupported_type_%s_%s", createType.replaceAll("[^a-zA-Z0-9]", ""), TestingNames.randomNameSuffix());
        this.assertQueryFails(String.format("CREATE TABLE %s (col1 %s)", tableName, createType), "Unsupported column type: " + createType);
        this.assertUpdate("DROP TABLE IF EXISTS " + tableName);
    }

    @Test
    public void testCreateTableWithRowTypeWithoutField() {
        String tableName = "test_row_type_table_" + TestingNames.randomNameSuffix();
        this.assertQueryFails("CREATE TABLE " + tableName + "(col1 row(int))", "\\QROW type does not have field names declared: row(integer)\\E");
    }

    @Test
    public void testCreateTableAlreadyExists() {
        try (TestTable table = new TestTable(arg_0 -> ((QueryRunner)this.getQueryRunner()).execute(arg_0), "test_create_table_already_exists", "(col1 int)");){
            this.assertQueryFails("CREATE TABLE " + table.getName() + "(col1 int)", "\\Qline 1:1: Table 'bigquery.tpch." + table.getName() + "' already exists\\E");
        }
    }

    @Test
    public void testDeleteWithComplexPredicate() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithComplexPredicate()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testDeleteWithLike() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithLike()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testDeleteWithSemiJoin() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithSemiJoin()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testDeleteWithSubquery() {
        Assertions.assertThatThrownBy(() -> super.testDeleteWithSubquery()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testExplainAnalyzeWithDeleteWithSubquery() {
        Assertions.assertThatThrownBy(() -> super.testExplainAnalyzeWithDeleteWithSubquery()).hasStackTraceContaining("TrinoException: This connector does not support modifying table rows");
    }

    @Test
    public void testEmptyProjectionTable() {
        this.testEmptyProjection(tableName -> this.onBigQuery("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.region"), tableName -> this.onBigQuery("DROP TABLE " + tableName));
    }

    @Test
    public void testEmptyProjectionView() {
        this.testEmptyProjection(viewName -> this.onBigQuery("CREATE VIEW " + viewName + " AS SELECT * FROM tpch.region"), viewName -> this.onBigQuery("DROP VIEW " + viewName));
    }

    @Test
    public void testEmptyProjectionMaterializedView() {
        this.testEmptyProjection(materializedViewName -> this.onBigQuery("CREATE MATERIALIZED VIEW " + materializedViewName + " AS SELECT * FROM tpch.region"), materializedViewName -> this.onBigQuery("DROP MATERIALIZED VIEW " + materializedViewName));
    }

    @Test
    public void testEmptyProjectionExternalTable() {
        this.testEmptyProjection(externalTableName -> this.onBigQuery("CREATE EXTERNAL TABLE " + externalTableName + " OPTIONS (format = 'CSV', uris = ['gs://" + this.gcpStorageBucket + "/tpch/tiny/region.csv'])"), externalTableName -> this.onBigQuery("DROP EXTERNAL TABLE " + externalTableName));
    }

    @Test
    public void testEmptyProjectionSnapshotTable() {
        String regionCopy = "test.region_" + TestingNames.randomNameSuffix();
        this.onBigQuery("CREATE TABLE " + regionCopy + " AS SELECT * FROM tpch.region");
        try {
            this.testEmptyProjection(snapshotTableName -> this.onBigQuery("CREATE SNAPSHOT TABLE " + snapshotTableName + " CLONE " + regionCopy), snapshotTableName -> this.onBigQuery("DROP SNAPSHOT TABLE " + snapshotTableName));
        }
        finally {
            this.onBigQuery("DROP TABLE " + regionCopy);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testEmptyProjection(Consumer<String> createTable, Consumer<String> dropTable) {
        String name = "test.test_empty_projection_" + TestingNames.randomNameSuffix();
        createTable.accept(name);
        try {
            this.assertQuery("SELECT count(*) FROM " + name, "VALUES 5");
            this.assertQuery("SELECT count(*) FROM " + name, "VALUES 5");
            this.assertQuery("SELECT count(*) FROM " + name + " WHERE regionkey = 1", "VALUES 1");
            this.assertQuery("SELECT count(name) FROM " + name + " WHERE regionkey = 1", "VALUES 1");
        }
        finally {
            dropTable.accept(name);
        }
    }

    protected Optional<BaseConnectorTest.DataMappingTestSetup> filterDataMappingSmokeTestData(BaseConnectorTest.DataMappingTestSetup dataMappingTestSetup) {
        switch (dataMappingTestSetup.getTrinoTypeName()) {
            case "real": 
            case "char(3)": 
            case "time": 
            case "time(3)": 
            case "time(6)": 
            case "timestamp": 
            case "timestamp(3)": 
            case "timestamp(3) with time zone": {
                return Optional.of(dataMappingTestSetup.asUnsupported());
            }
        }
        return Optional.of(dataMappingTestSetup);
    }

    protected Optional<BaseConnectorTest.DataMappingTestSetup> filterCaseSensitiveDataMappingTestData(BaseConnectorTest.DataMappingTestSetup dataMappingTestSetup) {
        String typeName = dataMappingTestSetup.getTrinoTypeName();
        if (typeName.equals("char(1)")) {
            return Optional.of(dataMappingTestSetup.asUnsupported());
        }
        return Optional.of(dataMappingTestSetup);
    }

    @Test
    public void testNoDataSystemTable() {
        Assertions.assertThatThrownBy(() -> super.testNoDataSystemTable()).hasMessageFindingMatch("\\QExpecting message:\n  \"Cannot read partition information from a table that is not partitioned: \\E\\S+\\Q:tpch.nation$data\"\nto match regex:\n  \"line 1:1: Table '\\w+.\\w+.\"nation\\$data\"' does not exist\"\nbut did not.");
        Assumptions.abort((String)"TODO");
    }

    protected boolean isColumnNameRejected(Exception exception, String columnName, boolean delimited) {
        return Strings.nullToEmpty((String)exception.getMessage()).matches(".*Invalid field name \"%s\". Fields must contain the allowed characters, and be at most 300 characters long..*".formatted(columnName.replace("\\", "\\\\")));
    }

    @Test
    public void testCommentColumn() {
        try (TestTable table = new TestTable(arg_0 -> ((QueryRunner)this.getQueryRunner()).execute(arg_0), "test_comment_column_", "(a integer)");){
            this.assertUpdate("COMMENT ON COLUMN " + table.getName() + ".a IS 'new comment'");
            Assertions.assertThat((String)((String)this.computeScalar("SHOW CREATE TABLE " + table.getName()))).contains(new CharSequence[]{"COMMENT 'new comment'"});
            Assertions.assertThat((String)this.getColumnComment(table.getName(), "a")).isEqualTo("new comment");
            this.assertUpdate("COMMENT ON COLUMN " + table.getName() + ".a IS NULL");
            Assertions.assertThat((String)this.getColumnComment(table.getName(), "a")).isEqualTo(null);
        }
        table = new TestTable(arg_0 -> ((QueryRunner)this.getQueryRunner()).execute(arg_0), "test_comment_column_", "(a integer COMMENT 'test comment')");
        try {
            Assertions.assertThat((String)this.getColumnComment(table.getName(), "a")).isEqualTo("test comment");
            this.assertUpdate("COMMENT ON COLUMN " + table.getName() + ".a IS 'updated comment'");
            Assertions.assertThat((String)this.getColumnComment(table.getName(), "a")).isEqualTo("updated comment");
            this.assertUpdate("COMMENT ON COLUMN " + table.getName() + ".a IS ''");
            Assertions.assertThat((String)this.getColumnComment(table.getName(), "a")).isEqualTo("");
        }
        finally {
            table.close();
        }
    }

    @Test
    public void testPartitionDateColumn() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.partition_date_column", "(value INT64) PARTITION BY _PARTITIONDATE");){
            this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('1960-01-01', 1)", table.getName()));
            this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('2159-12-31', 2)", table.getName()));
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT value, \"$partition_date\" FROM " + table.getName()))).matches("VALUES (BIGINT '1', DATE '1960-01-01'), (BIGINT '2', DATE '2159-12-31')");
            this.assertQuery(String.format("SELECT value FROM %s WHERE \"$partition_date\" = DATE '1960-01-01'", table.getName()), "VALUES 1");
            this.assertQuery(String.format("SELECT value FROM %s WHERE \"$partition_date\" = DATE '2159-12-31'", table.getName()), "VALUES 2");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("DESCRIBE " + table.getName()))).projected(new String[]{"Column"}).skippingTypesCheck().matches("VALUES 'value'");
        }
    }

    @Test
    public void testPartitionTimeColumn() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.partition_time_column", "(value INT64) PARTITION BY DATE_TRUNC(_PARTITIONTIME, HOUR)");){
            this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('1960-01-01 00:00:00', 1)", table.getName()));
            this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('2159-12-31 23:00:00', 2)", table.getName()));
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT value, \"$partition_time\" FROM " + table.getName()))).matches("VALUES (BIGINT '1', CAST('1960-01-01 00:00:00 UTC' AS TIMESTAMP(6) WITH TIME ZONE)), (BIGINT '2', CAST('2159-12-31 23:00:00 UTC' AS TIMESTAMP(6) WITH TIME ZONE))");
            this.assertQuery(String.format("SELECT value FROM %s WHERE \"$partition_time\" = CAST('1960-01-01 00:00:00 UTC' AS TIMESTAMP(6) WITH TIME ZONE)", table.getName()), "VALUES 1");
            this.assertQuery(String.format("SELECT value FROM %s WHERE \"$partition_time\" = CAST('2159-12-31 23:00:00 UTC' AS TIMESTAMP(6) WITH TIME ZONE)", table.getName()), "VALUES 2");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("DESCRIBE " + table.getName()))).projected(new String[]{"Column"}).skippingTypesCheck().matches("VALUES 'value'");
        }
    }

    @Test
    public void testIngestionTimePartitionedTableInvalidValue() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.invalid_ingestion_time", "(value INT64) PARTITION BY _PARTITIONDATE");){
            Assertions.assertThatThrownBy(() -> this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('0001-01-01', 1)", table.getName()))).hasMessageMatching("Cannot set pseudo column for automatic partitioned table.* Supported values are in the range \\[1960-01-01, 2159-12-31]");
            Assertions.assertThatThrownBy(() -> this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('1959-12-31', 1)", table.getName()))).hasMessageMatching("Cannot set pseudo column for automatic partitioned table.* Supported values are in the range \\[1960-01-01, 2159-12-31]");
            Assertions.assertThatThrownBy(() -> this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('2160-01-01', 1)", table.getName()))).hasMessageMatching("Cannot set pseudo column for automatic partitioned table.* Supported values are in the range \\[1960-01-01, 2159-12-31]");
            Assertions.assertThatThrownBy(() -> this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES ('9999-12-31', 1)", table.getName()))).hasMessageMatching("Cannot set pseudo column for automatic partitioned table.* Supported values are in the range \\[1960-01-01, 2159-12-31]");
            Assertions.assertThatThrownBy(() -> this.onBigQuery(String.format("INSERT INTO %s (_PARTITIONTIME, value) VALUES (NULL, 1)", table.getName()))).hasMessageContaining("Cannot set timestamp pseudo column for automatic partitioned table to NULL");
        }
    }

    @Test
    public void testPseudoColumnNotExist() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.non_partitioned_table", "(value INT64, ts TIMESTAMP)");){
            this.assertQueryFails("SELECT \"$partition_date\" FROM " + table.getName(), ".* Column '\\$partition_date' cannot be resolved");
            this.assertQueryFails("SELECT \"$partition_time\" FROM " + table.getName(), ".* Column '\\$partition_time' cannot be resolved");
        }
        table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.time_unit_partition", "(value INT64, dt DATE) PARTITION BY dt");
        try {
            this.assertQueryFails("SELECT \"$partition_date\" FROM " + table.getName(), ".* Column '\\$partition_date' cannot be resolved");
            this.assertQueryFails("SELECT \"$partition_time\" FROM " + table.getName(), ".* Column '\\$partition_time' cannot be resolved");
        }
        finally {
            table.close();
        }
        table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.integer_range_partition", "(value INT64, dt DATE) PARTITION BY RANGE_BUCKET(value, GENERATE_ARRAY(0, 100, 10))");
        try {
            this.assertQueryFails("SELECT \"$partition_date\" FROM " + table.getName(), ".* Column '\\$partition_date' cannot be resolved");
            this.assertQueryFails("SELECT \"$partition_time\" FROM " + table.getName(), ".* Column '\\$partition_time' cannot be resolved");
        }
        finally {
            table.close();
        }
    }

    @Test
    public void testSelectFromHourlyPartitionedTable() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.hourly_partitioned", "(value INT64, ts TIMESTAMP) PARTITION BY TIMESTAMP_TRUNC(ts, HOUR)", List.of("1000, '2018-01-01 10:00:00'"));){
            this.assertQuery("SELECT COUNT(1) FROM " + table.getName(), "VALUES 1");
        }
    }

    @Test
    public void testSelectFromYearlyPartitionedTable() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.yearly_partitioned", "(value INT64, ts TIMESTAMP) PARTITION BY TIMESTAMP_TRUNC(ts, YEAR)", List.of("1000, '2018-01-01 10:00:00'"));){
            this.assertQuery("SELECT COUNT(1) FROM " + table.getName(), "VALUES 1");
        }
    }

    @Test
    public void testSelectWithSingleQuoteInWhereClause() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.select_with_single_quote", "(col INT64, val STRING)", List.of("1, 'escape\\'single quote'"));){
            this.assertQuery("SELECT val FROM " + table.getName() + " WHERE val = 'escape''single quote'", "VALUES 'escape''single quote'");
        }
    }

    @Test
    public void testPredicatePushdownPrunnedColumns() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.predicate_pushdown_prunned_columns", "(a INT64, b INT64, c INT64)", List.of("1, 2, 3"));){
            this.assertQuery("SELECT 1 FROM " + table.getName() + " WHERE     ((NULL IS NULL) OR a = 100) AND     b = 2", "VALUES (1)");
        }
    }

    @Test
    public void testColumnPositionMismatch() {
        try (TestTable table = new TestTable(arg_0 -> ((QueryRunner)this.getQueryRunner()).execute(arg_0), "test.test_column_position_mismatch", "(c_varchar VARCHAR, c_int INT, c_date DATE)");){
            this.onBigQuery("INSERT INTO " + table.getName() + " VALUES ('a', 1, '2021-01-01')");
            this.assertQuery("SELECT c_varchar, CAST(c_int AS SMALLINT), c_date FROM " + table.getName(), "VALUES ('a', 1, '2021-01-01')");
        }
    }

    @Test
    public void testSelectTableWithRowAccessPolicyFilterAll() {
        String policyName = "test_policy" + TestingNames.randomNameSuffix();
        try (TestTable table = new TestTable(this::onBigQuery, "test.test_row_access_policy", "AS SELECT 1 col");){
            this.assertQuery("SELECT * FROM " + table.getName(), "VALUES 1");
            this.onBigQuery("CREATE ROW ACCESS POLICY " + policyName + " ON " + table.getName() + " FILTER USING (true)");
            Assert.assertEventually((Duration)new Duration(1.0, TimeUnit.MINUTES), () -> this.assertQueryReturnsEmptyResult("SELECT * FROM " + table.getName()));
            this.onBigQuery("DROP ALL ROW ACCESS POLICIES ON " + table.getName());
            Assert.assertEventually((Duration)new Duration(1.0, TimeUnit.MINUTES), () -> this.assertQuery("SELECT * FROM " + table.getName(), "VALUES 1"));
        }
    }

    @Test
    public void testSelectTableWithRowAccessPolicyFilterPartialRow() {
        String policyName = "test_policy" + TestingNames.randomNameSuffix();
        try (TestTable table = new TestTable(this::onBigQuery, "test.test_row_access_policy", "AS (SELECT 1 col UNION ALL SELECT 2 col)");){
            this.assertQuery("SELECT * FROM " + table.getName(), "VALUES (1), (2)");
            this.onBigQuery("CREATE ROW ACCESS POLICY " + policyName + " ON " + table.getName() + " GRANT TO (\"allAuthenticatedUsers\") FILTER USING (col = 1)");
            Assert.assertEventually((Duration)new Duration(1.0, TimeUnit.MINUTES), () -> this.assertQuery("SELECT * FROM " + table.getName(), "VALUES 1"));
            this.onBigQuery("DROP ALL ROW ACCESS POLICIES ON " + table.getName());
            Assert.assertEventually((Duration)new Duration(1.0, TimeUnit.MINUTES), () -> this.assertQuery("SELECT * FROM " + table.getName(), "VALUES (1), (2)"));
        }
    }

    @Test
    public void testViewDefinitionSystemTable() {
        String schemaName = "test";
        String tableName = "views_system_table_base_" + TestingNames.randomNameSuffix();
        String viewName = "views_system_table_view_" + TestingNames.randomNameSuffix();
        this.onBigQuery(String.format("CREATE TABLE %s.%s (a INT64, b INT64, c INT64)", schemaName, tableName));
        this.onBigQuery(String.format("CREATE VIEW %s.%s AS SELECT * FROM %s.%s", schemaName, viewName, schemaName, tableName));
        Assertions.assertThat((Object)this.computeScalar(String.format("SELECT * FROM %s.\"%s$view_definition\"", schemaName, viewName))).isEqualTo((Object)String.format("SELECT * FROM %s.%s", schemaName, tableName));
        this.assertQueryFails(String.format("SELECT * FROM %s.\"%s$view_definition\"", schemaName, tableName), String.format("Table '%s.%s\\$view_definition' not found", schemaName, tableName));
        this.onBigQuery(String.format("DROP TABLE %s.%s", schemaName, tableName));
        this.onBigQuery(String.format("DROP VIEW %s.%s", schemaName, viewName));
    }

    @Test
    public void testShowCreateTable() {
        Assertions.assertThat((String)((String)this.computeActual("SHOW CREATE TABLE orders").getOnlyValue())).isEqualTo("CREATE TABLE bigquery.tpch.orders (\n   orderkey bigint NOT NULL,\n   custkey bigint NOT NULL,\n   orderstatus varchar NOT NULL,\n   totalprice double NOT NULL,\n   orderdate date NOT NULL,\n   orderpriority varchar NOT NULL,\n   clerk varchar NOT NULL,\n   shippriority bigint NOT NULL,\n   comment varchar NOT NULL\n)");
    }

    @Test
    public void testSkipUnsupportedType() {
        try (TestTable table = new TestTable((SqlExecutor)this.bigQuerySqlExecutor, "test.test_skip_unsupported_type", "(a INT64, unsupported BIGNUMERIC, b INT64)", List.of("1, 999, 2"));){
            this.assertQuery("SELECT * FROM " + table.getName(), "VALUES (1, 2)");
            Assertions.assertThat((String)((String)this.computeActual("SHOW CREATE TABLE " + table.getName()).getOnlyValue())).isEqualTo("CREATE TABLE bigquery." + table.getName() + " (\n   a bigint,\n   b bigint\n)");
        }
    }

    @Test
    public void testDateYearOfEraPredicate() {
        this.assertQuery("SELECT orderdate FROM orders WHERE orderdate = DATE '1997-09-14'", "VALUES DATE '1997-09-14'");
        Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM orders WHERE orderdate = DATE '-1996-09-14'")).hasMessageMatching(".*Could not cast literal \"-1996-09-14\" to type DATE.*");
    }

    @Test
    public void testBigQueryMaterializedView() {
        String materializedView = "test_materialized_view" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE MATERIALIZED VIEW test." + materializedView + " AS SELECT count(1) AS cnt FROM tpch.region");
            this.assertQuery("SELECT table_type FROM information_schema.tables WHERE table_schema = 'test' AND table_name = '" + materializedView + "'", "VALUES 'BASE TABLE'");
            this.assertQuery("DESCRIBE test." + materializedView, "VALUES ('cnt', 'bigint', '', '')");
            this.assertQuery("SELECT * FROM test." + materializedView, "VALUES 5");
            this.assertUpdate("DROP TABLE test." + materializedView);
            this.assertQueryReturnsEmptyResult("SELECT * FROM information_schema.tables WHERE table_schema = 'test' AND table_name = '" + materializedView + "'");
        }
        finally {
            this.onBigQuery("DROP MATERIALIZED VIEW IF EXISTS test." + materializedView);
        }
    }

    @Test
    public void testBigQuerySnapshotTable() {
        String regionCopy = "region_" + TestingNames.randomNameSuffix();
        String snapshotTable = "test_snapshot" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE TABLE test." + regionCopy + " AS SELECT * FROM tpch.region");
            this.onBigQuery("CREATE SNAPSHOT TABLE test." + snapshotTable + " CLONE test." + regionCopy);
            this.assertQuery("SELECT table_type FROM information_schema.tables WHERE table_schema = 'test' AND table_name = '" + snapshotTable + "'", "VALUES 'BASE TABLE'");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("DESCRIBE test." + snapshotTable))).matches("DESCRIBE tpch.region");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + snapshotTable))).matches("SELECT * FROM tpch.region");
            this.assertUpdate("DROP TABLE test." + snapshotTable);
            this.assertQueryReturnsEmptyResult("SELECT * FROM information_schema.tables WHERE table_schema = 'test' AND table_name = '" + snapshotTable + "'");
        }
        finally {
            this.onBigQuery("DROP SNAPSHOT TABLE IF EXISTS test." + snapshotTable);
            this.onBigQuery("DROP TABLE test." + regionCopy);
        }
    }

    @Test
    public void testBigQueryExternalTable() {
        String externalTable = "test_external" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE EXTERNAL TABLE test." + externalTable + " OPTIONS (format = 'CSV', uris = ['gs://" + this.gcpStorageBucket + "/tpch/tiny/region.csv'])");
            this.assertQuery("SELECT table_type FROM information_schema.tables WHERE table_schema = 'test' AND table_name = '" + externalTable + "'", "VALUES 'BASE TABLE'");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("DESCRIBE test." + externalTable))).matches("DESCRIBE tpch.region");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM test." + externalTable))).matches("SELECT * FROM tpch.region");
            this.assertUpdate("DROP TABLE test." + externalTable);
            this.assertQueryReturnsEmptyResult("SELECT * FROM information_schema.tables WHERE table_schema = 'test' AND table_name = '" + externalTable + "'");
        }
        finally {
            this.onBigQuery("DROP EXTERNAL TABLE IF EXISTS test." + externalTable);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testQueryLabeling() {
        Function<String, Session> sessionWithToken = token -> Session.builder((Session)this.getSession()).setTraceToken(Optional.of(token)).build();
        String materializedView = "test_query_label" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE MATERIALIZED VIEW test." + materializedView + " AS SELECT count(1) AS cnt FROM tpch.region");
            String query = "SELECT * FROM test." + materializedView;
            MaterializedResultWithQueryId result = this.getDistributedQueryRunner().executeWithQueryId(sessionWithToken.apply("first_token"), query);
            this.assertLabelForTable(materializedView, result.getQueryId(), "first_token");
            MaterializedResultWithQueryId result2 = this.getDistributedQueryRunner().executeWithQueryId(sessionWithToken.apply("second_token"), query);
            this.assertLabelForTable(materializedView, result2.getQueryId(), "second_token");
            Assertions.assertThatThrownBy(() -> this.getDistributedQueryRunner().executeWithQueryId((Session)sessionWithToken.apply("InvalidToken"), query)).hasMessageContaining("BigQuery label value can contain only lowercase letters, numeric characters, underscores, and dashes");
        }
        finally {
            this.onBigQuery("DROP MATERIALIZED VIEW IF EXISTS test." + materializedView);
        }
    }

    private void assertLabelForTable(String expectedView, QueryId queryId, String traceToken) {
        String expectedLabel = "q_" + queryId.toString() + "__t_" + traceToken;
        String checkForLabelQuery = "SELECT * FROM region-us.INFORMATION_SCHEMA.JOBS_BY_USER WHERE EXISTS(\n    SELECT * FROM UNNEST(labels) AS label WHERE label.key = 'trino_query' AND label.value = '%s'\n)".formatted(expectedLabel);
        Assert.assertEventually(() -> ((ObjectAssert)Assertions.assertThat((Iterable)this.bigQuerySqlExecutor.executeQuery(checkForLabelQuery).getValues()).extracting(values -> values.get("query").getStringValue()).singleElement()).matches(statement -> statement.contains(expectedView)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testQueryCache() {
        Session queryResultsCacheSession = Session.builder((Session)this.getSession()).setCatalogSessionProperty("bigquery", "query_results_cache_enabled", "true").build();
        Session createNeverDisposition = Session.builder((Session)this.getSession()).setCatalogSessionProperty("bigquery", "query_results_cache_enabled", "true").setCatalogSessionProperty("bigquery", "create_disposition_type", "create_never").build();
        String materializedView = "test_materialized_view" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE MATERIALIZED VIEW test." + materializedView + " AS SELECT count(1) AS cnt FROM tpch.region");
            Assertions.assertThatThrownBy(() -> this.query(createNeverDisposition, "SELECT * FROM test." + materializedView)).hasMessageContaining("Not found");
            this.assertQuery(queryResultsCacheSession, "SELECT * FROM test." + materializedView, "VALUES 5");
            this.assertQuery(createNeverDisposition, "SELECT * FROM test." + materializedView, "VALUES 5");
            this.assertUpdate("DROP TABLE test." + materializedView);
        }
        finally {
            this.onBigQuery("DROP MATERIALIZED VIEW IF EXISTS test." + materializedView);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testWildcardTable() {
        String suffix = TestingNames.randomNameSuffix();
        String firstTable = String.format("test_wildcard_%s_1", suffix);
        String secondTable = String.format("test_wildcard_%s_2", suffix);
        String wildcardTable = String.format("test_wildcard_%s_*", suffix);
        try {
            this.onBigQuery("CREATE TABLE test." + firstTable + " AS SELECT 1 AS value");
            this.onBigQuery("CREATE TABLE test." + secondTable + " AS SELECT 2 AS value");
            this.assertQuery("DESCRIBE test.\"" + wildcardTable + "\"", "VALUES ('value', 'bigint', '', '')");
            this.assertQuery("SELECT * FROM test.\"" + wildcardTable + "\"", "VALUES (1), (2)");
            this.assertQueryFails("DROP TABLE test.\"" + wildcardTable + "\"", "This connector does not support dropping wildcard tables");
            this.assertQueryFails("INSERT INTO test.\"" + wildcardTable + "\" VALUES (1)", "This connector does not support inserting into wildcard tables");
            this.assertQueryFails("ALTER TABLE test.\"" + wildcardTable + "\" ADD COLUMN new_column INT", "This connector does not support adding columns");
            this.assertQueryFails("ALTER TABLE test.\"" + wildcardTable + "\" RENAME TO test.new_wildcard_table", "This connector does not support renaming tables");
        }
        finally {
            this.onBigQuery("DROP TABLE IF EXISTS test." + firstTable);
            this.onBigQuery("DROP TABLE IF EXISTS test." + secondTable);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testWildcardTableWithDifferentColumnDefinition() {
        String suffix = TestingNames.randomNameSuffix();
        String firstTable = String.format("test_invalid_wildcard_%s_1", suffix);
        String secondTable = String.format("test_invalid_wildcard_%s_2", suffix);
        String wildcardTable = String.format("test_invalid_wildcard_%s_*", suffix);
        try {
            this.onBigQuery("CREATE TABLE test." + firstTable + " AS SELECT 1 AS value");
            this.onBigQuery("CREATE TABLE test." + secondTable + " AS SELECT 'string' AS value");
            this.assertQuery("DESCRIBE test.\"" + wildcardTable + "\"", "VALUES ('value', 'varchar', '', '')");
            Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM test.\"" + wildcardTable + "\"")).hasMessageContaining("Cannot read field of type INT64 as STRING Field: value");
        }
        finally {
            this.onBigQuery("DROP TABLE IF EXISTS test." + firstTable);
            this.onBigQuery("DROP TABLE IF EXISTS test." + secondTable);
        }
    }

    @Test
    public void testMissingWildcardTable() {
        Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM test.\"test_missing_wildcard_table_*\"")).hasMessageEndingWith("does not match any table.");
    }

    protected OptionalInt maxSchemaNameLength() {
        return OptionalInt.of(1024);
    }

    @Test
    public void testCreateSchemaWithLongName() {
        Assumptions.abort((String)"Dropping schema with long name causes BigQuery to return code 500");
    }

    protected void verifySchemaNameLengthFailurePermissible(Throwable e) {
        Assertions.assertThat((Throwable)e).hasMessageContaining("Invalid dataset ID");
    }

    protected OptionalInt maxTableNameLength() {
        return OptionalInt.of(1024);
    }

    protected void verifyTableNameLengthFailurePermissible(Throwable e) {
        Assertions.assertThat((Throwable)e).hasMessageContaining("Invalid table ID");
    }

    @Test
    public void testNativeQuerySimple() {
        this.assertQuery("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT 1'))", "VALUES 1");
    }

    @Test
    public void testNativeQuerySimpleWithProjectedColumns() {
        this.assertQuery("SELECT z, y, x FROM (SELECT y, z, x FROM TABLE(bigquery.system.query(query => 'SELECT 1 x, 2 y, 3 z')))", "VALUES (3, 2, 1)");
        this.assertQuery("SELECT z FROM (SELECT x, y, z FROM TABLE(bigquery.system.query(query => 'SELECT 1 x, 2 y, 3 z')))", "VALUES 3");
    }

    @Test
    public void testNativeQuerySelectForCaseSensitiveColumnNames() {
        Assertions.assertThat((List)this.computeActual("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT 1 AS lower, 2 AS UPPER, 3 AS miXED'))").getColumnNames()).containsExactly((Object[])new String[]{"lower", "UPPER", "miXED"});
        Assertions.assertThat((List)this.computeActual("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT 1 AS duplicated, 2 AS duplicated'))").getColumnNames()).containsExactly((Object[])new String[]{"duplicated", "duplicated_1"});
        String tableName = "test.test_non_lowercase" + TestingNames.randomNameSuffix();
        this.onBigQuery("CREATE TABLE " + tableName + " AS SELECT 1 AS lower, 2 AS UPPER, 3 AS miXED");
        try {
            this.assertQuery("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT * FROM " + tableName + "'))", "VALUES (1, 2, 3)");
            this.assertQuery("SELECT \"lower\", \"UPPER\", \"miXED\" FROM TABLE(bigquery.system.query(query => 'SELECT * FROM " + tableName + "'))", "VALUES (1, 2, 3)");
            this.assertQuery("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT * FROM " + tableName + "')) WHERE \"UPPER\" = 2", "VALUES (1, 2, 3)");
            this.assertQueryReturnsEmptyResult("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT * FROM " + tableName + "')) WHERE \"UPPER\" = 100");
            this.assertQueryReturnsEmptyResult("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT * FROM " + tableName + "')) WHERE upper = 100");
        }
        finally {
            this.onBigQuery("DROP TABLE " + tableName);
        }
    }

    @Test
    public void testNativeQuerySelectFromNation() {
        this.assertQuery("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT name FROM tpch.nation WHERE nationkey = 0'))", "VALUES 'ALGERIA'");
        this.assertQuery("SELECT name FROM TABLE(bigquery.system.query(query => 'SELECT * FROM tpch.nation')) WHERE nationkey = 0", "VALUES 'ALGERIA'");
    }

    @Test
    public void testNativeQueryColumnAlias() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT region_name FROM TABLE(system.query(query => 'SELECT name AS region_name FROM tpch.region WHERE regionkey = 0'))"))).matches("VALUES CAST('AFRICA' AS VARCHAR)");
    }

    @Test
    public void testNativeQueryColumnAliasNotFound() {
        this.assertQueryFails("SELECT name FROM TABLE(system.query(query => 'SELECT name AS region_name FROM tpch.region'))", ".* Column 'name' cannot be resolved");
        this.assertQueryFails("SELECT column_not_found FROM TABLE(system.query(query => 'SELECT name AS region_name FROM tpch.region'))", ".* Column 'column_not_found' cannot be resolved");
    }

    @Test
    public void testNativeQuerySelectFromTestTable() {
        String tableName = "test.test_select" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE TABLE " + tableName + "(col BIGINT)");
            this.onBigQuery("INSERT INTO " + tableName + " VALUES (1), (2)");
            this.assertQuery("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT * FROM " + tableName + "'))", "VALUES 1, 2");
        }
        finally {
            this.onBigQuery("DROP TABLE IF EXISTS " + tableName);
        }
    }

    @Test
    public void testNativeQuerySelectUnsupportedType() {
        String tableName = "test_unsupported" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE TABLE test." + tableName + "(one BIGINT, two BIGNUMERIC(40,2), three STRING)");
            this.assertQuery("SELECT column_name FROM information_schema.columns WHERE table_schema = 'test' AND table_name = '" + tableName + "'", "VALUES 'one', 'three'");
            Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM TABLE(bigquery.system.query(query => 'SELECT * FROM test." + tableName + "'))")).hasMessageContaining("Unsupported type");
        }
        finally {
            this.onBigQuery("DROP TABLE IF EXISTS test." + tableName);
        }
    }

    @Test
    public void testNativeQueryCreateStatement() {
        String tableName = "test_create" + TestingNames.randomNameSuffix();
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
        Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM TABLE(bigquery.system.query(query => 'CREATE TABLE test." + tableName + "(n INTEGER)'))")).hasMessage("Unsupported statement type: CREATE_TABLE");
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
    }

    @Test
    public void testNativeQueryInsertStatementTableDoesNotExist() {
        String tableName = "test_insert" + TestingNames.randomNameSuffix();
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
        Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM TABLE(bigquery.system.query(query => 'INSERT INTO test." + tableName + " VALUES (1)'))")).hasMessageContaining("Failed to get schema for query").hasStackTraceContaining("%s was not found", new Object[]{tableName});
    }

    @Test
    public void testNativeQueryInsertStatementTableExists() {
        String tableName = "test_insert" + TestingNames.randomNameSuffix();
        try {
            this.onBigQuery("CREATE TABLE test." + tableName + "(col BIGINT)");
            Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM TABLE(bigquery.system.query(query => 'INSERT INTO test." + tableName + " VALUES (3)'))")).hasMessage("Unsupported statement type: INSERT");
        }
        finally {
            this.onBigQuery("DROP TABLE IF EXISTS test." + tableName);
        }
    }

    @Test
    public void testNativeQueryIncorrectSyntax() {
        Assertions.assertThatThrownBy(() -> this.query("SELECT * FROM TABLE(system.query(query => 'some wrong syntax'))")).hasMessageContaining("Failed to get schema for query");
    }

    @Test
    public void testInsertArray() {
        try (TestTable table = new TestTable(arg_0 -> ((QueryRunner)this.getQueryRunner()).execute(arg_0), "test_insert_array_", "(a ARRAY<DOUBLE>, b ARRAY<BIGINT>)");){
            this.assertUpdate("INSERT INTO " + table.getName() + " (a, b) VALUES (ARRAY[1.23E1], ARRAY[1.23E1])", 1L);
            this.assertQuery("SELECT a[1], b[1] FROM " + table.getName(), "VALUES (12.3, 12)");
        }
    }

    @Test
    public void testInsertRowConcurrently() {
        Assumptions.abort((String)"Test fails with a timeout sometimes and is flaky");
    }

    protected String errorMessageForCreateTableAsSelectNegativeDate(String date) {
        return "BigQuery supports dates between 0001-01-01 and 9999-12-31 but got " + date;
    }

    protected String errorMessageForInsertNegativeDate(String date) {
        return "BigQuery supports dates between 0001-01-01 and 9999-12-31 but got " + date;
    }

    protected TestTable createTableWithDefaultColumns() {
        return new TestTable(this::onBigQuery, "test.test_table", "(col_required INT64 NOT NULL,col_nullable INT64,col_default INT64 DEFAULT 43,col_nonnull_default INT64 DEFAULT 42 NOT NULL,col_required2 INT64 NOT NULL)");
    }

    @Test
    public void testCharVarcharComparison() {
        Assertions.assertThatThrownBy(() -> super.testCharVarcharComparison()).hasMessage("Unsupported column type: char(3)");
    }

    protected OptionalInt maxColumnNameLength() {
        return OptionalInt.of(300);
    }

    protected void verifyColumnNameLengthFailurePermissible(Throwable e) {
        Assertions.assertThat((Throwable)e).hasMessageContaining("Fields must contain only letters, numbers, and underscores, start with a letter or underscore, and be at most 300 characters long.");
    }

    private void onBigQuery(@Language(value="SQL") String sql) {
        this.bigQuerySqlExecutor.execute(sql);
    }
}

