/*
 * Decompiled with CFR 0.152.
 */
package io.trino.client.auth.external;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.concurrent.Threads;
import io.trino.client.ClientException;
import io.trino.client.auth.external.ExternalAuthentication;
import io.trino.client.auth.external.ExternalAuthenticator;
import io.trino.client.auth.external.KnownToken;
import io.trino.client.auth.external.MockRedirectHandler;
import io.trino.client.auth.external.MockTokenPoller;
import io.trino.client.auth.external.RedirectHandler;
import io.trino.client.auth.external.Token;
import io.trino.client.auth.external.TokenPollResult;
import io.trino.client.auth.external.TokenPoller;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Stream;
import okhttp3.HttpUrl;
import okhttp3.Protocol;
import okhttp3.Request;
import okhttp3.Response;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.assertj.core.api.ThrowableAssert;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.Timeout;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
public class TestExternalAuthenticator {
    private static final ExecutorService executor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)(TestExternalAuthenticator.class.getName() + "-%d")));

    @AfterAll
    public void shutDownThreadPool() {
        executor.shutdownNow();
    }

    @Test
    public void testChallengeWithOnlyTokenServerUri() {
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("Bearer x_token_server=\"http://token.uri\"")).hasValueSatisfying(authentication -> {
            Assertions.assertThat((Optional)authentication.getRedirectUri()).isEmpty();
            Assertions.assertThat((URI)authentication.getTokenUri()).isEqualTo((Object)URI.create("http://token.uri"));
        });
    }

    @Test
    public void testChallengeWithBothUri() {
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("Bearer x_redirect_server=\"http://redirect.uri\", x_token_server=\"http://token.uri\"")).hasValueSatisfying(authentication -> {
            Assertions.assertThat((Optional)authentication.getRedirectUri()).hasValue((Object)URI.create("http://redirect.uri"));
            Assertions.assertThat((URI)authentication.getTokenUri()).isEqualTo((Object)URI.create("http://token.uri"));
        });
    }

    @Test
    public void testChallengeWithValuesWithoutQuotes() {
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("Bearer x_redirect_server=http://redirect.uri, x_token_server=http://token.uri")).hasValueSatisfying(authentication -> {
            Assertions.assertThat((Optional)authentication.getRedirectUri()).hasValue((Object)URI.create("http://redirect.uri"));
            Assertions.assertThat((URI)authentication.getTokenUri()).isEqualTo((Object)URI.create("http://token.uri"));
        });
    }

    @Test
    public void testChallengeWithAdditionalFields() {
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("Bearer type=\"token\", x_redirect_server=\"http://redirect.uri\", x_token_server=\"http://token.uri\", description=\"oauth challenge\"")).hasValueSatisfying(authentication -> {
            Assertions.assertThat((Optional)authentication.getRedirectUri()).hasValue((Object)URI.create("http://redirect.uri"));
            Assertions.assertThat((URI)authentication.getTokenUri()).isEqualTo((Object)URI.create("http://token.uri"));
        });
    }

    @Test
    public void testInvalidChallenges() {
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("Bearer")).isEmpty();
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("x_redirect_server=\"http://redirect.uri\", x_token_server=\"http://token.uri\"")).isEmpty();
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("Bearer x_redirect_server=\"http://redirect.uri\" x_token_server=\"http://token.uri\"")).isEmpty();
        Assertions.assertThat(TestExternalAuthenticator.buildAuthentication("Bearer x_redirect_server=\"http://redirect.uri\"=x_token_server=\"http://token.uri\"")).isEmpty();
    }

    @Test
    public void testChallengeWithMalformedUri() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> TestExternalAuthenticator.buildAuthentication("Bearer x_token_server=\"http://[1.1.1.1]\"")).isInstanceOf(ClientException.class)).hasMessageContaining("Failed to parse URI for field '%s'", new Object[]{"x_token_server"}).hasRootCauseInstanceOf(URISyntaxException.class).hasRootCauseMessage("Malformed IPv6 address at index 8: http://[1.1.1.1]");
    }

    @Test
    public void testAuthentication() {
        MockTokenPoller tokenPoller = new MockTokenPoller().withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("valid-token")));
        ExternalAuthenticator authenticator = new ExternalAuthenticator(uri -> {}, (TokenPoller)tokenPoller, KnownToken.local(), Duration.ofSeconds(1L));
        Request authenticated = authenticator.authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\""));
        Assertions.assertThat((String)authenticated.headers().get("Authorization")).isEqualTo("Bearer valid-token");
    }

    @Test
    public void testReAuthenticationAfterRejectingToken() {
        MockTokenPoller tokenPoller = new MockTokenPoller().withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("first-token"))).withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("second-token")));
        ExternalAuthenticator authenticator = new ExternalAuthenticator(uri -> {}, (TokenPoller)tokenPoller, KnownToken.local(), Duration.ofSeconds(1L));
        Request request = authenticator.authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\""));
        Request reAuthenticated = authenticator.authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\"", request));
        Assertions.assertThat((List)reAuthenticated.headers("Authorization")).containsExactly((Object[])new String[]{"Bearer second-token"});
    }

    @Test
    @Timeout(value=2L)
    public void testAuthenticationFromMultipleThreadsWithLocallyStoredToken() {
        MockTokenPoller tokenPoller = new MockTokenPoller().withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("valid-token-1"))).withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("valid-token-2"))).withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("valid-token-3"))).withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("valid-token-4")));
        MockRedirectHandler redirectHandler = new MockRedirectHandler();
        List requests = (List)TestExternalAuthenticator.times(4, () -> new ExternalAuthenticator((RedirectHandler)redirectHandler, (TokenPoller)tokenPoller, KnownToken.local(), Duration.ofSeconds(1L)).authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\", x_redirect_server=\"http://redirect.uri\""))).map(executor::submit).collect(ImmutableList.toImmutableList());
        ConcurrentRequestAssertion assertion = new ConcurrentRequestAssertion(requests);
        assertion.requests().extracting(Request::headers).extracting(headers -> headers.get("Authorization")).contains((Object[])new String[]{"Bearer valid-token-1", "Bearer valid-token-2", "Bearer valid-token-3", "Bearer valid-token-4"});
        assertion.assertThatNoExceptionsHasBeenThrown();
        Assertions.assertThat((int)redirectHandler.getRedirectionCount()).isEqualTo(4);
    }

    @Test
    @Timeout(value=2L)
    public void testAuthenticationFromMultipleThreadsWithCachedToken() {
        MockTokenPoller tokenPoller = new MockTokenPoller().withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("valid-token")));
        MockRedirectHandler redirectHandler = new MockRedirectHandler().sleepOnRedirect(Duration.ofSeconds(1L));
        List requests = (List)TestExternalAuthenticator.times(2, () -> new ExternalAuthenticator((RedirectHandler)redirectHandler, (TokenPoller)tokenPoller, KnownToken.memoryCached(), Duration.ofSeconds(1L)).authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\", x_redirect_server=\"http://redirect.uri\""))).map(executor::submit).collect(ImmutableList.toImmutableList());
        ConcurrentRequestAssertion assertion = new ConcurrentRequestAssertion(requests);
        assertion.requests().extracting(Request::headers).extracting(headers -> headers.get("Authorization")).containsOnly((Object[])new String[]{"Bearer valid-token"});
        assertion.assertThatNoExceptionsHasBeenThrown();
        Assertions.assertThat((int)redirectHandler.getRedirectionCount()).isEqualTo(1);
    }

    @Test
    @Timeout(value=2L)
    public void testAuthenticationFromMultipleThreadsWithCachedTokenAfterAuthenticateFails() {
        MockTokenPoller tokenPoller = new MockTokenPoller().withResult(URI.create("http://token.uri"), TokenPollResult.successful((Token)new Token("first-token"))).withResult(URI.create("http://token.uri"), TokenPollResult.failed((String)"external authentication error"));
        MockRedirectHandler redirectHandler = new MockRedirectHandler().sleepOnRedirect(Duration.ofMillis(500L));
        ExternalAuthenticator authenticator = new ExternalAuthenticator((RedirectHandler)redirectHandler, (TokenPoller)tokenPoller, KnownToken.memoryCached(), Duration.ofSeconds(1L));
        Request firstRequest = authenticator.authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\", x_redirect_server=\"http://redirect.uri\""));
        List requests = (List)TestExternalAuthenticator.times(4, () -> new ExternalAuthenticator((RedirectHandler)redirectHandler, (TokenPoller)tokenPoller, KnownToken.memoryCached(), Duration.ofSeconds(1L)).authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\", x_redirect_server=\"http://redirect.uri\"", firstRequest))).map(executor::submit).collect(ImmutableList.toImmutableList());
        ConcurrentRequestAssertion assertion = new ConcurrentRequestAssertion(requests);
        assertion.requests().containsOnlyNulls();
        ((ThrowableAssert)assertion.firstException().hasMessage("external authentication error")).isInstanceOf(ClientException.class);
        Assertions.assertThat((int)redirectHandler.getRedirectionCount()).isEqualTo(2);
    }

    @Test
    @Timeout(value=2L)
    public void testAuthenticationFromMultipleThreadsWithCachedTokenAfterAuthenticateTimesOut() {
        MockRedirectHandler redirectHandler = new MockRedirectHandler().sleepOnRedirect(Duration.ofSeconds(1L));
        List requests = (List)TestExternalAuthenticator.times(2, () -> new ExternalAuthenticator((RedirectHandler)redirectHandler, MockTokenPoller.onPoll(TokenPollResult::pending), KnownToken.memoryCached(), Duration.ofMillis(1L)).authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\", x_redirect_server=\"http://redirect.uri\""))).map(executor::submit).collect(ImmutableList.toImmutableList());
        ConcurrentRequestAssertion assertion = new ConcurrentRequestAssertion(requests);
        assertion.requests().containsExactly((Object[])new Request[]{null, null});
        assertion.assertThatNoExceptionsHasBeenThrown();
        Assertions.assertThat((int)redirectHandler.getRedirectionCount()).isEqualTo(1);
    }

    @Test
    @Timeout(value=2L)
    public void testAuthenticationFromMultipleThreadsWithCachedTokenAfterAuthenticateIsInterrupted() throws Exception {
        ExecutorService interruptableThreadPool = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)(this.getClass().getName() + "-interruptable-%d")));
        MockRedirectHandler redirectHandler = new MockRedirectHandler().sleepOnRedirect(Duration.ofMinutes(1L));
        ExternalAuthenticator authenticator = new ExternalAuthenticator((RedirectHandler)redirectHandler, MockTokenPoller.onPoll(TokenPollResult::pending), KnownToken.memoryCached(), Duration.ofMillis(1L));
        Future<Request> interruptedAuthentication = interruptableThreadPool.submit(() -> authenticator.authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\", x_redirect_server=\"http://redirect.uri\"")));
        Thread.sleep(100L);
        List requests = (List)TestExternalAuthenticator.times(2, () -> new ExternalAuthenticator((RedirectHandler)redirectHandler, MockTokenPoller.onPoll(TokenPollResult::pending), KnownToken.memoryCached(), Duration.ofMillis(1L)).authenticate(null, TestExternalAuthenticator.getUnauthorizedResponse("Bearer x_token_server=\"http://token.uri\", x_redirect_server=\"http://redirect.uri\""))).map(executor::submit).collect(ImmutableList.toImmutableList());
        Thread.sleep(100L);
        interruptableThreadPool.shutdownNow();
        ConcurrentRequestAssertion assertion = new ConcurrentRequestAssertion((List<Future<Request>>)ImmutableList.builder().addAll((Iterable)requests).add(interruptedAuthentication).build());
        assertion.requests().containsExactly((Object[])new Request[]{null, null});
        assertion.firstException().hasRootCauseInstanceOf(InterruptedException.class);
        Assertions.assertThat((int)redirectHandler.getRedirectionCount()).isEqualTo(1);
    }

    private static Stream<Callable<Request>> times(int times, Callable<Request> request) {
        return Stream.generate(() -> request).limit(times);
    }

    private static Optional<ExternalAuthentication> buildAuthentication(String challengeHeader) {
        return ExternalAuthenticator.toAuthentication((Response)TestExternalAuthenticator.getUnauthorizedResponse(challengeHeader));
    }

    private static Response getUnauthorizedResponse(String challengeHeader) {
        return TestExternalAuthenticator.getUnauthorizedResponse(challengeHeader, new Request.Builder().url(HttpUrl.get((String)"http://example.com")).build());
    }

    private static Response getUnauthorizedResponse(String challengeHeader, Request request) {
        return new Response.Builder().request(request).protocol(Protocol.HTTP_1_1).code(401).message("Unauthorized").header("WWW-Authenticate", challengeHeader).build();
    }

    static class ConcurrentRequestAssertion {
        private final List<Throwable> exceptions = new ArrayList<Throwable>();
        private final List<Request> requests = new ArrayList<Request>();

        public ConcurrentRequestAssertion(List<Future<Request>> requests) {
            for (Future<Request> request : requests) {
                try {
                    this.requests.add(request.get());
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException(e);
                }
                catch (CancellationException ex) {
                    this.exceptions.add(ex);
                }
                catch (ExecutionException ex) {
                    Preconditions.checkState((ex.getCause() != null ? 1 : 0) != 0, (Object)("Missing cause on ExecutionException " + ex.getMessage()));
                    this.exceptions.add(ex.getCause());
                }
            }
        }

        ThrowableAssert<Throwable> firstException() {
            return this.exceptions.stream().findFirst().map(ThrowableAssert::new).orElseGet(() -> new ThrowableAssert(() -> null));
        }

        void assertThatNoExceptionsHasBeenThrown() {
            if (!this.exceptions.isEmpty()) {
                Throwable firstException = this.exceptions.get(0);
                AssertionError assertionError = new AssertionError("Expected no exceptions, but some exceptions has been thrown", firstException);
                for (int i = 1; i < this.exceptions.size(); ++i) {
                    ((Throwable)((Object)assertionError)).addSuppressed(this.exceptions.get(i));
                }
                throw assertionError;
            }
        }

        ListAssert<Request> requests() {
            return Assertions.assertThat(this.requests);
        }
    }
}

