/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.concurrent;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.base.Verify;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.google.errorprone.annotations.ThreadSafe;
import jakarta.annotation.Nullable;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.ToLongFunction;

@ThreadSafe
public class DynamicSizeBoundQueue<T> {
    private final AtomicLong size = new AtomicLong();
    private final Queue<ElementAndSize<T>> queue = new ConcurrentLinkedQueue<ElementAndSize<T>>();
    private final AtomicReference<SettableFuture<Void>> enqueueFuture = new AtomicReference();
    private final AtomicReference<SettableFuture<Void>> dequeueFuture = new AtomicReference();
    private final long maxSize;
    private final ToLongFunction<T> elementSizeFunction;
    private final Ticker ticker;

    public DynamicSizeBoundQueue(long maxSize, ToLongFunction<T> elementSizeFunction) {
        this(maxSize, elementSizeFunction, Ticker.systemTicker());
    }

    public DynamicSizeBoundQueue(long maxSize, ToLongFunction<T> elementSizeFunction, Ticker ticker) {
        Preconditions.checkArgument((maxSize > 0L ? 1 : 0) != 0, (Object)"maxSize must be positive");
        this.maxSize = maxSize;
        this.elementSizeFunction = Objects.requireNonNull(elementSizeFunction, "elementSizeFunction is null");
        this.ticker = Objects.requireNonNull(ticker, "ticker is null");
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    public long getSize() {
        return this.size.get();
    }

    public boolean offer(T element) {
        long elementSize = this.elementSizeFunction.applyAsLong(element);
        return this.offer(element, elementSize);
    }

    private boolean offer(T element, long elementSize) {
        Objects.requireNonNull(element, "element is null");
        Preconditions.checkArgument((elementSize > 0L ? 1 : 0) != 0, (Object)"element size must be positive");
        if (!this.tryAcquireSizeReservation(elementSize)) {
            return false;
        }
        this.queue.add(new ElementAndSize<T>(element, elementSize));
        DynamicSizeBoundQueue.notifyIfNecessary(this.enqueueFuture);
        return true;
    }

    private boolean tryAcquireSizeReservation(long elementSize) {
        long newSize;
        if (this.size.get() >= this.maxSize) {
            return false;
        }
        try {
            newSize = DynamicSizeBoundQueue.getAndAddOverflowChecked(this.size, elementSize);
        }
        catch (ArithmeticException e) {
            return false;
        }
        if (newSize >= this.maxSize) {
            Verify.verify((this.size.addAndGet(-elementSize) >= 0L ? 1 : 0) != 0);
            return false;
        }
        return true;
    }

    private static long getAndAddOverflowChecked(AtomicLong atomicLong, long delta) {
        return atomicLong.getAndAccumulate(delta, Math::addExact);
    }

    public boolean offer(T element, long timeout, TimeUnit unit) throws InterruptedException {
        long elementSize = this.elementSizeFunction.applyAsLong(element);
        long remainingTimeoutNs = unit.toNanos(timeout);
        while (!this.offer(element, elementSize)) {
            ListenableFuture<Void> future = DynamicSizeBoundQueue.getOrCreateFuture(this.dequeueFuture);
            if (this.offer(element, elementSize)) break;
            long startTimeNs = this.ticker.read();
            if (remainingTimeoutNs <= 0L || !this.awaitDequeueFuture((Future<?>)future, remainingTimeoutNs, TimeUnit.NANOSECONDS)) {
                return false;
            }
            remainingTimeoutNs -= this.ticker.read() - startTimeNs;
        }
        return true;
    }

    public void put(T element) throws InterruptedException {
        long elementSize = this.elementSizeFunction.applyAsLong(element);
        while (!this.offer(element, elementSize)) {
            ListenableFuture<Void> future = DynamicSizeBoundQueue.getOrCreateFuture(this.dequeueFuture);
            if (this.offer(element, elementSize)) break;
            this.awaitDequeueFuture((Future<?>)future);
        }
    }

    public Optional<ListenableFuture<Void>> offerWithBackoff(T element) {
        long elementSize = this.elementSizeFunction.applyAsLong(element);
        if (this.offer(element, elementSize)) {
            return Optional.empty();
        }
        ListenableFuture<Void> future = DynamicSizeBoundQueue.getOrCreateFuture(this.dequeueFuture);
        if (this.offer(element, elementSize)) {
            return Optional.empty();
        }
        return Optional.of(Futures.nonCancellationPropagating(future));
    }

    public void forcePut(T element) {
        long elementSize = this.elementSizeFunction.applyAsLong(element);
        Preconditions.checkArgument((elementSize > 0L ? 1 : 0) != 0, (Object)"element size must be positive");
        try {
            DynamicSizeBoundQueue.getAndAddOverflowChecked(this.size, elementSize);
        }
        catch (ArithmeticException e) {
            throw new IllegalStateException("Forced element triggered queue size numeric overflow");
        }
        this.queue.add(new ElementAndSize<T>(element, elementSize));
        DynamicSizeBoundQueue.notifyIfNecessary(this.enqueueFuture);
    }

    @Nullable
    public T poll() {
        ElementAndSize<T> elementAndSize = this.queue.poll();
        if (elementAndSize == null) {
            return null;
        }
        Verify.verify((this.size.addAndGet(-elementAndSize.size()) >= 0L ? 1 : 0) != 0);
        DynamicSizeBoundQueue.notifyIfNecessary(this.dequeueFuture);
        return elementAndSize.element();
    }

    public T poll(long timeout, TimeUnit unit) throws InterruptedException {
        long remainingTimeoutNs = unit.toNanos(timeout);
        T element;
        while ((element = this.poll()) == null) {
            ListenableFuture<Void> future = DynamicSizeBoundQueue.getOrCreateFuture(this.enqueueFuture);
            element = this.poll();
            if (element != null) {
                return element;
            }
            long startTimeNs = this.ticker.read();
            if (remainingTimeoutNs <= 0L || !this.awaitEnqueueFuture((Future<?>)future, remainingTimeoutNs, TimeUnit.NANOSECONDS)) {
                return null;
            }
            remainingTimeoutNs -= this.ticker.read() - startTimeNs;
        }
        return element;
    }

    public T take() throws InterruptedException {
        T element;
        while ((element = this.poll()) == null) {
            ListenableFuture<Void> future = DynamicSizeBoundQueue.getOrCreateFuture(this.enqueueFuture);
            element = this.poll();
            if (element != null) {
                return element;
            }
            this.awaitEnqueueFuture((Future<?>)future);
        }
        return element;
    }

    private static ListenableFuture<Void> getOrCreateFuture(AtomicReference<SettableFuture<Void>> reference) {
        return (ListenableFuture)reference.updateAndGet(current -> Objects.requireNonNullElseGet(current, SettableFuture::create));
    }

    private static void notifyIfNecessary(AtomicReference<SettableFuture<Void>> reference) {
        SettableFuture future = reference.getAndSet(null);
        if (future != null) {
            future.set(null);
        }
    }

    @VisibleForTesting
    void preEnqueueAwaitHook() {
    }

    @VisibleForTesting
    void preDequeueAwaitHook() {
    }

    private void awaitDequeueFuture(Future<?> future) throws InterruptedException {
        this.preDequeueAwaitHook();
        DynamicSizeBoundQueue.awaitFutureUnchecked(future);
    }

    private boolean awaitDequeueFuture(Future<?> future, long timeout, TimeUnit timeUnit) throws InterruptedException {
        this.preDequeueAwaitHook();
        return DynamicSizeBoundQueue.awaitFutureUnchecked(future, timeout, timeUnit);
    }

    private void awaitEnqueueFuture(Future<?> future) throws InterruptedException {
        this.preEnqueueAwaitHook();
        DynamicSizeBoundQueue.awaitFutureUnchecked(future);
    }

    private boolean awaitEnqueueFuture(Future<?> future, long timeout, TimeUnit timeUnit) throws InterruptedException {
        this.preEnqueueAwaitHook();
        return DynamicSizeBoundQueue.awaitFutureUnchecked(future, timeout, timeUnit);
    }

    private static void awaitFutureUnchecked(Future<?> future) throws InterruptedException {
        try {
            future.get();
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private static boolean awaitFutureUnchecked(Future<?> future, long timeout, TimeUnit timeUnit) throws InterruptedException {
        try {
            future.get(timeout, timeUnit);
            return true;
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        catch (TimeoutException e) {
            return false;
        }
    }

    private record ElementAndSize<T>(T element, long size) {
        private ElementAndSize {
            Objects.requireNonNull(element, "element is null");
            Preconditions.checkArgument((size > 0L ? 1 : 0) != 0, (Object)"size must be positive");
        }
    }
}

