/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.stats;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterators;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Ordering;
import com.google.common.collect.PeekingIterator;
import com.google.common.util.concurrent.AtomicDouble;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.stats.ExponentialDecay;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
public class QuantileDigest {
    private static final int MAX_BITS = 64;
    private static final int QUANTILE_DIGEST_SIZE = SizeOf.instanceSize(QuantileDigest.class);
    static final long RESCALE_THRESHOLD_SECONDS = 50L;
    static final double ZERO_WEIGHT_THRESHOLD = 1.0E-5;
    private static final int INITIAL_CAPACITY = 1;
    private final double maxError;
    private final Ticker ticker;
    private final double alpha;
    private long landmarkInSeconds;
    private double weightedCount;
    private long max = Long.MIN_VALUE;
    private long min = Long.MAX_VALUE;
    private int root = -1;
    private int nextNode;
    private double[] counts;
    private byte[] levels;
    private long[] values;
    private int[] lefts;
    private int[] rights;
    private int freeCount;
    private int firstFree = -1;

    public QuantileDigest(double maxError) {
        this(maxError, 0.0);
    }

    public QuantileDigest(double maxError, double alpha) {
        this(maxError, alpha, alpha == 0.0 ? QuantileDigest.noOpTicker() : Ticker.systemTicker());
    }

    @VisibleForTesting
    QuantileDigest(double maxError, double alpha, Ticker ticker) {
        Preconditions.checkArgument((maxError >= 0.0 && maxError <= 1.0 ? 1 : 0) != 0, (Object)"maxError must be in range [0, 1]");
        Preconditions.checkArgument((alpha >= 0.0 && alpha < 1.0 ? 1 : 0) != 0, (Object)"alpha must be in range [0, 1)");
        this.maxError = maxError;
        this.alpha = alpha;
        this.ticker = ticker;
        this.landmarkInSeconds = TimeUnit.NANOSECONDS.toSeconds(ticker.read());
        this.counts = new double[1];
        this.levels = new byte[1];
        this.values = new long[1];
        this.lefts = new int[1];
        this.rights = new int[1];
        Arrays.fill(this.lefts, -1);
        Arrays.fill(this.rights, -1);
    }

    public QuantileDigest(QuantileDigest other) {
        this.maxError = other.maxError;
        this.alpha = other.alpha;
        this.ticker = this.alpha == 0.0 ? QuantileDigest.noOpTicker() : Ticker.systemTicker();
        this.landmarkInSeconds = other.landmarkInSeconds;
        this.weightedCount = other.weightedCount;
        this.max = other.max;
        this.min = other.min;
        this.root = other.root;
        this.nextNode = other.nextNode;
        this.counts = (double[])other.counts.clone();
        this.levels = (byte[])other.levels.clone();
        this.values = (long[])other.values.clone();
        this.lefts = (int[])other.lefts.clone();
        this.rights = (int[])other.rights.clone();
        this.freeCount = other.freeCount;
        this.firstFree = other.firstFree;
    }

    public QuantileDigest(Slice serialized) {
        BasicSliceInput input = new BasicSliceInput(serialized);
        byte format = input.readByte();
        Preconditions.checkArgument((format == 0 ? 1 : 0) != 0, (Object)"Invalid format");
        this.maxError = input.readDouble();
        this.alpha = input.readDouble();
        this.ticker = this.alpha == 0.0 ? QuantileDigest.noOpTicker() : Ticker.systemTicker();
        this.landmarkInSeconds = input.readLong();
        this.min = input.readLong();
        this.max = input.readLong();
        int nodeCount = input.readInt();
        int height = 64 - Long.numberOfLeadingZeros(this.min ^ this.max) + 1;
        Preconditions.checkArgument((height >= 64 || (long)nodeCount <= (1L << height) - 1L ? 1 : 0) != 0, (Object)"Too many nodes in deserialized tree. Possible corruption");
        this.counts = new double[nodeCount];
        this.levels = new byte[nodeCount];
        this.values = new long[nodeCount];
        int[] stack = new int[(Integer.highestOneBit(nodeCount - 1) << 1) + 1];
        int top = -1;
        this.lefts = new int[nodeCount];
        this.rights = new int[nodeCount];
        for (int node = 0; node < nodeCount; ++node) {
            byte nodeStructure = input.readByte();
            boolean hasRight = (nodeStructure & 2) != 0;
            boolean hasLeft = (nodeStructure & 1) != 0;
            byte level = (byte)(nodeStructure >>> 2 & 0x3F);
            if (hasLeft || hasRight) {
                level = (byte)(level + 1);
            }
            this.levels[node] = level;
            this.rights[node] = hasRight ? stack[top--] : -1;
            this.lefts[node] = hasLeft ? stack[top--] : -1;
            stack[++top] = node;
            double count = input.readDouble();
            this.weightedCount += count;
            this.counts[node] = count;
            this.values[node] = input.readLong();
        }
        Preconditions.checkArgument((nodeCount == 0 || top == 0 ? 1 : 0) != 0, (Object)"Tree is corrupted. Expected a single root node");
        this.root = nodeCount - 1;
        this.nextNode = nodeCount;
    }

    public double getMaxError() {
        return this.maxError;
    }

    public double getAlpha() {
        return this.alpha;
    }

    public void add(long value) {
        this.add(value, 1L);
    }

    public void add(long value, double weight) {
        Preconditions.checkArgument((weight > 0.0 ? 1 : 0) != 0, (Object)"weight must be > 0");
        boolean needsCompression = false;
        if (this.alpha > 0.0) {
            long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this.ticker.read());
            if (nowInSeconds - this.landmarkInSeconds >= 50L) {
                this.rescale(nowInSeconds);
                needsCompression = true;
            }
            weight *= ExponentialDecay.weight(this.alpha, nowInSeconds, this.landmarkInSeconds);
        }
        this.max = Math.max(this.max, value);
        this.min = Math.min(this.min, value);
        double previousCount = this.weightedCount;
        this.insert(QuantileDigest.longToBits(value), weight);
        int compressionFactor = this.calculateCompressionFactor();
        if (needsCompression || (long)previousCount / (long)compressionFactor != (long)this.weightedCount / (long)compressionFactor) {
            this.compress();
        }
    }

    public void add(long value, long weight) {
        this.add(value, (double)weight);
    }

    public void merge(QuantileDigest other) {
        this.rescaleToCommonLandmark(this, other);
        this.root = this.merge(this.root, other, other.root);
        this.max = Math.max(this.max, other.max);
        this.min = Math.min(this.min, other.min);
        this.compress();
    }

    public List<Long> getQuantilesLowerBound(List<Double> quantiles) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(quantiles), (Object)"quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument((quantile >= 0.0 && quantile <= 1.0 ? 1 : 0) != 0, (Object)"quantile must be between [0,1]");
        }
        ImmutableList reversedQuantiles = ImmutableList.copyOf(quantiles).reverse();
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(reversedQuantiles.iterator());
        this.postOrderTraversal(this.root, new Callback(){
            private double sum;

            @Override
            public boolean process(int node) {
                this.sum += QuantileDigest.this.counts[node];
                while (iterator.hasNext() && this.sum > (1.0 - (Double)iterator.peek()) * QuantileDigest.this.weightedCount) {
                    iterator.next();
                    long value = Math.max(QuantileDigest.this.lowerBound(node), QuantileDigest.this.min);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        }, TraversalOrder.REVERSE);
        while (iterator.hasNext()) {
            builder.add((Object)this.min);
            iterator.next();
        }
        return builder.build().reverse();
    }

    public List<Long> getQuantilesUpperBound(List<Double> quantiles) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(quantiles), (Object)"quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument((quantile >= 0.0 && quantile <= 1.0 ? 1 : 0) != 0, (Object)"quantile must be between [0,1]");
        }
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(quantiles.iterator());
        this.postOrderTraversal(this.root, new Callback(){
            private double sum;

            @Override
            public boolean process(int node) {
                this.sum += QuantileDigest.this.counts[node];
                while (iterator.hasNext() && this.sum > (Double)iterator.peek() * QuantileDigest.this.weightedCount) {
                    iterator.next();
                    long value = Math.min(QuantileDigest.this.upperBound(node), QuantileDigest.this.max);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            builder.add((Object)this.max);
            iterator.next();
        }
        return builder.build();
    }

    public List<Long> getQuantiles(List<Double> quantiles) {
        return this.getQuantilesUpperBound(quantiles);
    }

    public long getQuantile(double quantile) {
        return this.getQuantiles((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public long getQuantileLowerBound(double quantile) {
        return this.getQuantilesLowerBound((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public long getQuantileUpperBound(double quantile) {
        return this.getQuantilesUpperBound((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public double getCount() {
        return this.weightedCount / ExponentialDecay.weight(this.alpha, TimeUnit.NANOSECONDS.toSeconds(this.ticker.read()), this.landmarkInSeconds);
    }

    public List<Bucket> getHistogram(List<Long> bucketUpperBounds) {
        return this.getHistogram(bucketUpperBounds, MiddleFunction.DEFAULT);
    }

    public List<Bucket> getHistogram(List<Long> bucketUpperBounds, MiddleFunction middleFunction) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(bucketUpperBounds), (Object)"buckets must be sorted in increasing order");
        ImmutableList.Builder builder = ImmutableList.builder();
        PeekingIterator iterator = Iterators.peekingIterator(bucketUpperBounds.iterator());
        HistogramBuilderStateHolder holder = new HistogramBuilderStateHolder();
        double normalizationFactor = ExponentialDecay.weight(this.alpha, TimeUnit.NANOSECONDS.toSeconds(this.ticker.read()), this.landmarkInSeconds);
        this.postOrderTraversal(this.root, node -> {
            while (iterator.hasNext() && (Long)iterator.peek() <= this.upperBound(node)) {
                double bucketCount = holder.sum - holder.lastSum;
                Bucket bucket = new Bucket(bucketCount / normalizationFactor, holder.bucketWeightedSum / bucketCount);
                builder.add((Object)bucket);
                holder.lastSum = holder.sum;
                holder.bucketWeightedSum = 0.0;
                iterator.next();
            }
            holder.bucketWeightedSum += middleFunction.middle(this.lowerBound(node), this.upperBound(node)) * this.counts[node];
            holder.sum += this.counts[node];
            return iterator.hasNext();
        });
        while (iterator.hasNext()) {
            double bucketCount = holder.sum - holder.lastSum;
            Bucket bucket = new Bucket(bucketCount / normalizationFactor, holder.bucketWeightedSum / bucketCount);
            builder.add((Object)bucket);
            iterator.next();
        }
        return builder.build();
    }

    public long getMin() {
        AtomicLong chosen = new AtomicLong(this.min);
        this.postOrderTraversal(this.root, node -> {
            if (this.counts[node] >= 1.0E-5) {
                chosen.set(this.lowerBound(node));
                return false;
            }
            return true;
        }, TraversalOrder.FORWARD);
        return Math.max(this.min, chosen.get());
    }

    public long getMax() {
        AtomicLong chosen = new AtomicLong(this.max);
        this.postOrderTraversal(this.root, node -> {
            if (this.counts[node] >= 1.0E-5) {
                chosen.set(this.upperBound(node));
                return false;
            }
            return true;
        }, TraversalOrder.REVERSE);
        return Math.min(this.max, chosen.get());
    }

    public int estimatedInMemorySizeInBytes() {
        return (int)((long)QUANTILE_DIGEST_SIZE + SizeOf.sizeOf((double[])this.counts) + SizeOf.sizeOf((byte[])this.levels) + SizeOf.sizeOf((long[])this.values) + SizeOf.sizeOf((int[])this.lefts) + SizeOf.sizeOf((int[])this.rights));
    }

    public int estimatedSerializedSizeInBytes() {
        int nodeSize = 17;
        return 45 + this.getNodeCount() * nodeSize;
    }

    public Slice serialize() {
        this.compress();
        DynamicSliceOutput output = new DynamicSliceOutput(this.estimatedSerializedSizeInBytes());
        output.writeByte(0);
        output.writeDouble(this.maxError);
        output.writeDouble(this.alpha);
        output.writeLong(this.landmarkInSeconds);
        output.writeLong(this.min);
        output.writeLong(this.max);
        output.writeInt(this.getNodeCount());
        final int[] nodes = new int[this.getNodeCount()];
        this.postOrderTraversal(this.root, new Callback(){
            int index;

            @Override
            public boolean process(int node) {
                nodes[this.index++] = node;
                return true;
            }
        });
        for (int node : nodes) {
            byte nodeStructure = (byte)(Math.max(this.levels[node] - 1, 0) << 2);
            if (this.lefts[node] != -1) {
                nodeStructure = (byte)(nodeStructure | 1);
            }
            if (this.rights[node] != -1) {
                nodeStructure = (byte)(nodeStructure | 2);
            }
            output.writeByte((int)nodeStructure);
            output.writeDouble(this.counts[node]);
            output.writeLong(this.values[node]);
        }
        return output.slice();
    }

    @VisibleForTesting
    int getNodeCount() {
        return this.nextNode - this.freeCount;
    }

    @VisibleForTesting
    void compress() {
        double bound = Math.floor(this.weightedCount / (double)this.calculateCompressionFactor());
        this.postOrderTraversal(this.root, node -> {
            boolean shouldCompress;
            int left = this.lefts[node];
            int right = this.rights[node];
            if (left == -1 && right == -1) {
                return true;
            }
            double leftCount = left == -1 ? 0.0 : this.counts[left];
            double rightCount = right == -1 ? 0.0 : this.counts[right];
            boolean bl = shouldCompress = this.counts[node] + leftCount + rightCount < bound;
            if (left != -1 && (shouldCompress || leftCount < 1.0E-5)) {
                this.lefts[node] = this.tryRemove(left);
                int n = node;
                this.counts[n] = this.counts[n] + leftCount;
            }
            if (right != -1 && (shouldCompress || rightCount < 1.0E-5)) {
                this.rights[node] = this.tryRemove(right);
                int n = node;
                this.counts[n] = this.counts[n] + rightCount;
            }
            return true;
        });
        if (this.root != -1 && this.counts[this.root] < 1.0E-5) {
            this.root = this.tryRemove(this.root);
        }
    }

    private void rescale(long newLandmarkInSeconds) {
        double factor = ExponentialDecay.weight(this.alpha, newLandmarkInSeconds, this.landmarkInSeconds);
        this.weightedCount /= factor;
        int i = 0;
        while (i < this.nextNode) {
            int n = i++;
            this.counts[n] = this.counts[n] / factor;
        }
        this.landmarkInSeconds = newLandmarkInSeconds;
    }

    private int calculateCompressionFactor() {
        if (this.root == -1) {
            return 1;
        }
        return Math.max((int)((double)(this.levels[this.root] + 1) / this.maxError), 1);
    }

    private void insert(long value, double count) {
        if (count < 1.0E-5) {
            return;
        }
        long lastBranch = 0L;
        int parent = -1;
        int current = this.root;
        while (true) {
            if (current == -1) {
                this.setChild(parent, lastBranch, this.createLeaf(value, count));
                return;
            }
            long currentValue = this.values[current];
            byte currentLevel = this.levels[current];
            if (!QuantileDigest.inSameSubtree(value, currentValue, currentLevel)) {
                this.setChild(parent, lastBranch, this.makeSiblings(current, this.createLeaf(value, count)));
                return;
            }
            if (currentLevel == 0 && currentValue == value) {
                int n = current;
                this.counts[n] = this.counts[n] + count;
                this.weightedCount += count;
                return;
            }
            long branch = value & this.getBranchMask(currentLevel);
            parent = current;
            lastBranch = branch;
            if (branch == 0L) {
                current = this.lefts[current];
                continue;
            }
            current = this.rights[current];
        }
    }

    private void setChild(int parent, long branch, int child) {
        if (parent == -1) {
            this.root = child;
        } else if (branch == 0L) {
            this.lefts[parent] = child;
        } else {
            this.rights[parent] = child;
        }
    }

    private int makeSiblings(int first, int second) {
        long firstValue = this.values[first];
        long secondValue = this.values[second];
        int parentLevel = 64 - Long.numberOfLeadingZeros(firstValue ^ secondValue);
        int parent = this.createNode(firstValue, parentLevel, 0.0);
        long branch = firstValue & this.getBranchMask(this.levels[parent]);
        if (branch == 0L) {
            this.lefts[parent] = first;
            this.rights[parent] = second;
        } else {
            this.lefts[parent] = second;
            this.rights[parent] = first;
        }
        return parent;
    }

    private int createLeaf(long value, double count) {
        return this.createNode(value, 0, count);
    }

    private int createNode(long value, int level, double count) {
        int node = this.popFree();
        if (node == -1) {
            if (this.nextNode == this.counts.length) {
                int newSize = this.counts.length + Math.min(this.counts.length, this.calculateCompressionFactor() / 5 + 1);
                this.counts = Arrays.copyOf(this.counts, newSize);
                this.levels = Arrays.copyOf(this.levels, newSize);
                this.values = Arrays.copyOf(this.values, newSize);
                this.lefts = Arrays.copyOf(this.lefts, newSize);
                this.rights = Arrays.copyOf(this.rights, newSize);
            }
            node = this.nextNode++;
        }
        this.weightedCount += count;
        this.values[node] = value;
        this.levels[node] = (byte)level;
        this.counts[node] = count;
        this.lefts[node] = -1;
        this.rights[node] = -1;
        return node;
    }

    private int merge(int node, QuantileDigest other, int otherNode) {
        if (otherNode == -1) {
            return node;
        }
        if (node == -1) {
            return this.copyRecursive(other, otherNode);
        }
        if (!QuantileDigest.inSameSubtree(this.values[node], other.values[otherNode], Math.max(this.levels[node], other.levels[otherNode]))) {
            return this.makeSiblings(node, this.copyRecursive(other, otherNode));
        }
        if (this.levels[node] > other.levels[otherNode]) {
            long branch = other.values[otherNode] & this.getBranchMask(this.levels[node]);
            if (branch == 0L) {
                int left;
                this.lefts[node] = left = this.merge(this.lefts[node], other, otherNode);
            } else {
                int right;
                this.rights[node] = right = this.merge(this.rights[node], other, otherNode);
            }
            return node;
        }
        if (this.levels[node] < other.levels[otherNode]) {
            int right;
            int left;
            long branch = this.values[node] & this.getBranchMask(other.levels[otherNode]);
            if (branch == 0L) {
                left = this.merge(node, other, other.lefts[otherNode]);
                right = this.copyRecursive(other, other.rights[otherNode]);
            } else {
                left = this.copyRecursive(other, other.lefts[otherNode]);
                right = this.merge(node, other, other.rights[otherNode]);
            }
            int result = this.createNode(other.values[otherNode], other.levels[otherNode], other.counts[otherNode]);
            this.lefts[result] = left;
            this.rights[result] = right;
            return result;
        }
        this.weightedCount += other.counts[otherNode];
        int n = node;
        this.counts[n] = this.counts[n] + other.counts[otherNode];
        int left = this.merge(this.lefts[node], other, other.lefts[otherNode]);
        int right = this.merge(this.rights[node], other, other.rights[otherNode]);
        this.lefts[node] = left;
        this.rights[node] = right;
        return node;
    }

    private static boolean inSameSubtree(long bitsA, long bitsB, int level) {
        return level == 64 || bitsA >>> level == bitsB >>> level;
    }

    private int copyRecursive(QuantileDigest other, int otherNode) {
        if (otherNode == -1) {
            return otherNode;
        }
        int node = this.createNode(other.values[otherNode], other.levels[otherNode], other.counts[otherNode]);
        if (other.lefts[otherNode] != -1) {
            int left;
            this.lefts[node] = left = this.copyRecursive(other, other.lefts[otherNode]);
        }
        if (other.rights[otherNode] != -1) {
            int right;
            this.rights[node] = right = this.copyRecursive(other, other.rights[otherNode]);
        }
        return node;
    }

    private int tryRemove(int node) {
        Preconditions.checkArgument((node != -1 ? 1 : 0) != 0, (Object)"node is -1");
        int left = this.lefts[node];
        int right = this.rights[node];
        if (left == -1 && right == -1) {
            this.remove(node);
            return -1;
        }
        if (left != -1 && right != -1) {
            this.counts[node] = 0.0;
            return node;
        }
        this.remove(node);
        if (left != -1) {
            return left;
        }
        return right;
    }

    private void remove(int node) {
        if (node == this.nextNode - 1) {
            --this.nextNode;
        } else {
            this.pushFree(node);
        }
        if (node == this.root) {
            this.root = -1;
        }
    }

    private void pushFree(int node) {
        this.lefts[node] = this.firstFree;
        this.firstFree = node;
        ++this.freeCount;
    }

    private int popFree() {
        int node = this.firstFree;
        if (node == -1) {
            return node;
        }
        this.firstFree = this.lefts[this.firstFree];
        --this.freeCount;
        return node;
    }

    private void postOrderTraversal(int node, Callback callback) {
        this.postOrderTraversal(node, callback, TraversalOrder.FORWARD);
    }

    private void postOrderTraversal(int node, Callback callback, TraversalOrder order) {
        if (order == TraversalOrder.FORWARD) {
            this.postOrderTraversal(node, callback, this.lefts, this.rights);
        } else {
            this.postOrderTraversal(node, callback, this.rights, this.lefts);
        }
    }

    private boolean postOrderTraversal(int node, Callback callback, int[] lefts, int[] rights) {
        if (node == -1) {
            return false;
        }
        int first = lefts[node];
        int second = rights[node];
        if (first != -1 && !this.postOrderTraversal(first, callback, lefts, rights)) {
            return false;
        }
        if (second != -1 && !this.postOrderTraversal(second, callback, lefts, rights)) {
            return false;
        }
        return callback.process(node);
    }

    public double getConfidenceFactor() {
        return this.computeMaxPathWeight(this.root) * 1.0 / this.weightedCount;
    }

    @VisibleForTesting
    boolean equivalent(QuantileDigest other) {
        return this.getNodeCount() == other.getNodeCount() && this.min == other.min && this.max == other.max && this.weightedCount == other.weightedCount && this.alpha == other.alpha;
    }

    private void rescaleToCommonLandmark(QuantileDigest one, QuantileDigest two) {
        long targetLandmark;
        long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this.ticker.read());
        if (nowInSeconds - (targetLandmark = Math.max(one.landmarkInSeconds, two.landmarkInSeconds)) >= 50L) {
            targetLandmark = nowInSeconds;
        }
        if (targetLandmark != one.landmarkInSeconds) {
            one.rescale(targetLandmark);
        }
        if (targetLandmark != two.landmarkInSeconds) {
            two.rescale(targetLandmark);
        }
    }

    private double computeMaxPathWeight(int node) {
        if (node == -1 || this.levels[node] == 0) {
            return 0.0;
        }
        double leftMaxWeight = this.computeMaxPathWeight(this.lefts[node]);
        double rightMaxWeight = this.computeMaxPathWeight(this.rights[node]);
        return Math.max(leftMaxWeight, rightMaxWeight) + this.counts[node];
    }

    @VisibleForTesting
    void validate() {
        AtomicDouble sum = new AtomicDouble();
        AtomicInteger nodeCount = new AtomicInteger();
        Set<Integer> freeSlots = this.computeFreeList();
        Preconditions.checkState((freeSlots.size() == this.freeCount ? 1 : 0) != 0, (String)"Free count (%s) doesn't match actual free slots: %s", (int)this.freeCount, (int)freeSlots.size());
        if (this.root != -1) {
            this.validateStructure(this.root, freeSlots);
            this.postOrderTraversal(this.root, node -> {
                sum.addAndGet(this.counts[node]);
                nodeCount.incrementAndGet();
                return true;
            });
        }
        Preconditions.checkState((Math.abs(sum.get() - this.weightedCount) < 1.0E-5 ? 1 : 0) != 0, (String)"Computed weight (%s) doesn't match summary (%s)", (Object)sum.get(), (Object)this.weightedCount);
        Preconditions.checkState((nodeCount.get() == this.getNodeCount() ? 1 : 0) != 0, (String)"Actual node count (%s) doesn't match summary (%s)", (int)nodeCount.get(), (int)this.getNodeCount());
    }

    private void validateStructure(int node, Set<Integer> freeNodes) {
        Preconditions.checkState((this.levels[node] >= 0 ? 1 : 0) != 0);
        Preconditions.checkState((!freeNodes.contains(node) ? 1 : 0) != 0, (String)"Node is in list of free slots: %s", (int)node);
        if (this.lefts[node] != -1) {
            this.validateBranchStructure(node, this.lefts[node], this.rights[node], true);
            this.validateStructure(this.lefts[node], freeNodes);
        }
        if (this.rights[node] != -1) {
            this.validateBranchStructure(node, this.rights[node], this.lefts[node], false);
            this.validateStructure(this.rights[node], freeNodes);
        }
    }

    private void validateBranchStructure(int parent, int child, int otherChild, boolean isLeft) {
        Preconditions.checkState((this.levels[child] < this.levels[parent] ? 1 : 0) != 0, (String)"Child level (%s) should be smaller than parent level (%s)", (int)this.levels[child], (int)this.levels[parent]);
        long branch = this.values[child] & 1L << this.levels[parent] - 1;
        Preconditions.checkState((branch == 0L && isLeft || branch != 0L && !isLeft ? 1 : 0) != 0, (Object)"Value of child node is inconsistent with its branch");
        Preconditions.checkState((this.counts[parent] > 0.0 || this.counts[child] > 0.0 || otherChild != -1 ? 1 : 0) != 0, (Object)"Found a linear chain of zero-weight nodes");
    }

    private Set<Integer> computeFreeList() {
        HashSet<Integer> freeSlots = new HashSet<Integer>();
        int index = this.firstFree;
        while (index != -1) {
            freeSlots.add(index);
            index = this.lefts[index];
        }
        return freeSlots;
    }

    public String toGraphviz() {
        StringBuilder builder = new StringBuilder();
        builder.append("digraph QuantileDigest {\n").append("\tgraph [ordering=\"out\"];");
        ArrayList nodes = new ArrayList();
        this.postOrderTraversal(this.root, node -> {
            nodes.add(node);
            return true;
        });
        ImmutableListMultimap nodesByLevel = Multimaps.index(nodes, input -> this.levels[input]);
        for (Map.Entry entry : nodesByLevel.asMap().entrySet()) {
            builder.append("\tsubgraph level_" + entry.getKey() + " {\n").append("\t\trank = same;\n");
            Iterator iterator = ((Collection)entry.getValue()).iterator();
            while (iterator.hasNext()) {
                int node2 = (Integer)iterator.next();
                if (this.levels[node2] == 0) {
                    builder.append(String.format("\t\t%s [label=\"%s:[%s]@%s\\n%s\", shape=rect, style=filled,color=%s];\n", QuantileDigest.idFor(node2), node2, this.lowerBound(node2), this.levels[node2], this.counts[node2], this.counts[node2] > 0.0 ? "salmon2" : "white"));
                    continue;
                }
                builder.append(String.format("\t\t%s [label=\"%s:[%s..%s]@%s\\n%s\", shape=rect, style=filled,color=%s];\n", QuantileDigest.idFor(node2), node2, this.lowerBound(node2), this.upperBound(node2), this.levels[node2], this.counts[node2], this.counts[node2] > 0.0 ? "salmon2" : "white"));
            }
            builder.append("\t}\n");
        }
        Iterator iterator = nodes.iterator();
        while (iterator.hasNext()) {
            int node3 = (Integer)((Object)iterator.next());
            if (this.lefts[node3] != -1) {
                builder.append(String.format("\t%s -> %s [style=\"%s\"];\n", QuantileDigest.idFor(node3), QuantileDigest.idFor(this.lefts[node3]), this.levels[node3] - this.levels[this.lefts[node3]] == 1 ? "solid" : "dotted"));
            }
            if (this.rights[node3] == -1) continue;
            builder.append(String.format("\t%s -> %s [style=\"%s\"];\n", QuantileDigest.idFor(node3), QuantileDigest.idFor(this.rights[node3]), this.levels[node3] - this.levels[this.rights[node3]] == 1 ? "solid" : "dotted"));
        }
        builder.append("}\n");
        return builder.toString();
    }

    private static String idFor(int node) {
        return String.format("node_%x", node);
    }

    private static long longToBits(long value) {
        return value ^ Long.MIN_VALUE;
    }

    private static long bitsToLong(long bits) {
        return bits ^ Long.MIN_VALUE;
    }

    private long getBranchMask(byte level) {
        return 1L << level - 1;
    }

    private long upperBound(int node) {
        long mask = 0L;
        if (this.levels[node] > 0) {
            mask = -1L >>> 64 - this.levels[node];
        }
        return QuantileDigest.bitsToLong(this.values[node] | mask);
    }

    private long lowerBound(int node) {
        long mask = 0L;
        if (this.levels[node] > 0) {
            mask = -1L >>> 64 - this.levels[node];
        }
        return QuantileDigest.bitsToLong(this.values[node] & (mask ^ 0xFFFFFFFFFFFFFFFFL));
    }

    private long middle(int node) {
        long lower = this.lowerBound(node);
        long upper = this.upperBound(node);
        return lower + (upper - lower) / 2L;
    }

    private static Ticker noOpTicker() {
        return new Ticker(){

            public long read() {
                return 0L;
            }
        };
    }

    private static class Flags {
        public static final int HAS_LEFT = 1;
        public static final int HAS_RIGHT = 2;
        public static final byte FORMAT = 0;

        private Flags() {
        }
    }

    private static enum TraversalOrder {
        FORWARD,
        REVERSE;

    }

    private static interface Callback {
        public boolean process(int var1);
    }

    public static interface MiddleFunction {
        public static final MiddleFunction DEFAULT = (lowerBound, upperBound) -> (double)lowerBound + (double)(upperBound - lowerBound) / 2.0;

        public double middle(long var1, long var3);
    }

    private static final class HistogramBuilderStateHolder {
        double sum;
        double lastSum;
        double bucketWeightedSum;

        private HistogramBuilderStateHolder() {
        }
    }

    public static class Bucket {
        private double count;
        private double mean;

        public Bucket(double count, double mean) {
            this.count = count;
            this.mean = mean;
        }

        public double getCount() {
            return this.count;
        }

        public double getMean() {
            return this.mean;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Bucket bucket = (Bucket)o;
            if (Double.compare(bucket.count, this.count) != 0) {
                return false;
            }
            return Double.compare(bucket.mean, this.mean) == 0;
        }

        public int hashCode() {
            long temp = this.count != 0.0 ? Double.doubleToLongBits(this.count) : 0L;
            int result = (int)(temp ^ temp >>> 32);
            temp = this.mean != 0.0 ? Double.doubleToLongBits(this.mean) : 0L;
            result = 31 * result + (int)(temp ^ temp >>> 32);
            return result;
        }

        public String toString() {
            return String.format("[count: %f, mean: %f]", this.count, this.mean);
        }
    }
}

