/*
 * Decompiled with CFR 0.152.
 */
package io.airlift.stats;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Doubles;
import com.google.common.primitives.Ints;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.SizeOf;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.airlift.stats.DoubleArrays;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import javax.annotation.concurrent.NotThreadSafe;
import org.openjdk.jol.info.ClassLayout;

@NotThreadSafe
public class TDigest {
    public static final double DEFAULT_COMPRESSION = 100.0;
    private static final int FORMAT_TAG = 0;
    private static final int T_DIGEST_SIZE = Math.toIntExact(ClassLayout.parseClass(TDigest.class).instanceSize());
    private static final int INITIAL_CAPACITY = 1;
    private static final int FUDGE_FACTOR = 10;
    private final int maxSize;
    private final double compression;
    double[] means;
    double[] weights;
    int centroidCount;
    double totalWeight;
    double min;
    double max;
    private boolean backwards;
    private boolean needsMerge;
    private int[] indexes;
    private double[] tempMeans;
    private double[] tempWeights;

    public TDigest() {
        this(100.0);
    }

    public TDigest(double compression) {
        this(compression, Double.POSITIVE_INFINITY, Double.NEGATIVE_INFINITY, 0.0, 0, new double[1], new double[1], false, false);
    }

    private TDigest(double compression, double min, double max, double totalWeight, int centroidCount, double[] means, double[] weights, boolean needsMerge, boolean backwards) {
        Preconditions.checkArgument((compression >= 10.0 ? 1 : 0) != 0, (Object)"compression factor too small (< 10)");
        this.compression = compression;
        this.maxSize = (int)(6.0 * (TDigest.internalCompressionFactor(compression) + 10.0));
        this.totalWeight = totalWeight;
        this.min = min;
        this.max = max;
        this.centroidCount = centroidCount;
        this.means = Objects.requireNonNull(means, "means is null");
        this.weights = Objects.requireNonNull(weights, "weights is null");
        this.needsMerge = needsMerge;
        this.backwards = backwards;
    }

    public static TDigest copyOf(TDigest other) {
        return new TDigest(other.compression, other.min, other.max, other.totalWeight, other.centroidCount, Arrays.copyOf(other.means, other.centroidCount), Arrays.copyOf(other.weights, other.centroidCount), other.needsMerge, other.backwards);
    }

    public static TDigest deserialize(Slice serialized) {
        BasicSliceInput input = serialized.getInput();
        byte format = input.readByte();
        Preconditions.checkArgument((format == 0 ? 1 : 0) != 0, (Object)"Invalid format");
        double min = input.readDouble();
        double max = input.readDouble();
        double compression = input.readDouble();
        double totalWeight = input.readDouble();
        int centroidCount = input.readInt();
        double[] means = new double[centroidCount];
        for (int i = 0; i < centroidCount; ++i) {
            means[i] = input.readDouble();
        }
        double[] weights = new double[centroidCount];
        for (int i = 0; i < centroidCount; ++i) {
            weights[i] = input.readDouble();
        }
        return new TDigest(compression, min, max, totalWeight, centroidCount, means, weights, false, false);
    }

    public double getMin() {
        if (this.totalWeight == 0.0) {
            return Double.NaN;
        }
        return this.min;
    }

    public double getMax() {
        if (this.totalWeight == 0.0) {
            return Double.NaN;
        }
        return this.max;
    }

    public double getCount() {
        return this.totalWeight;
    }

    public void add(double value) {
        this.add(value, 1.0);
    }

    public void add(double value, double weight) {
        Preconditions.checkArgument((!Double.isNaN(value) ? 1 : 0) != 0, (Object)"value is NaN");
        Preconditions.checkArgument((!Double.isNaN(weight) ? 1 : 0) != 0, (Object)"weight is NaN");
        Preconditions.checkArgument((!Double.isInfinite(value) ? 1 : 0) != 0, (Object)"value must be finite");
        Preconditions.checkArgument((!Double.isInfinite(weight) ? 1 : 0) != 0, (Object)"weight must be finite");
        if (this.centroidCount == this.means.length) {
            if (this.means.length < this.maxSize) {
                this.ensureCapacity(Math.min(Math.max(this.means.length * 2, 1), this.maxSize));
            } else {
                this.merge(TDigest.internalCompressionFactor(this.compression));
                if (this.centroidCount >= this.means.length) {
                    throw new AssertionError((Object)("Invalid size estimation for T-Digest: " + Base64.getEncoder().encodeToString(this.serializeInternal().getBytes())));
                }
            }
        }
        this.means[this.centroidCount] = value;
        this.weights[this.centroidCount] = weight;
        ++this.centroidCount;
        this.totalWeight += weight;
        this.min = Math.min(value, this.min);
        this.max = Math.max(value, this.max);
        this.needsMerge = true;
    }

    public void mergeWith(TDigest other) {
        if (this.centroidCount + other.centroidCount > this.means.length) {
            this.merge(TDigest.internalCompressionFactor(this.compression));
            other.merge(TDigest.internalCompressionFactor(this.compression));
            this.ensureCapacity(this.centroidCount + other.centroidCount);
        }
        System.arraycopy(other.means, 0, this.means, this.centroidCount, other.centroidCount);
        System.arraycopy(other.weights, 0, this.weights, this.centroidCount, other.centroidCount);
        this.centroidCount += other.centroidCount;
        this.totalWeight += other.totalWeight;
        this.min = Math.min(this.min, other.min);
        this.max = Math.max(this.max, other.max);
        this.needsMerge = true;
    }

    public double valueAt(double quantile) {
        return this.valuesAt(quantile)[0];
    }

    public List<Double> valuesAt(List<Double> quantiles) {
        return Doubles.asList((double[])this.valuesAt(Doubles.toArray(quantiles)));
    }

    public double[] valuesAt(double ... quantiles) {
        int index;
        if (quantiles.length == 0) {
            return new double[0];
        }
        TDigest.validateQuantilesArgument(quantiles);
        double[] result = new double[quantiles.length];
        if (this.centroidCount == 0) {
            Arrays.fill(result, Double.NaN);
            return result;
        }
        this.mergeIfNeeded(TDigest.internalCompressionFactor(this.compression));
        if (this.centroidCount == 1) {
            Arrays.fill(result, this.means[0]);
            return result;
        }
        for (int i = 0; i < result.length; ++i) {
            result[i] = quantiles[i] * this.totalWeight;
        }
        for (index = 0; index < result.length && result[index] < 1.0; ++index) {
            result[index] = this.min;
        }
        while (index < result.length && result[index] < this.weights[0] / 2.0) {
            result[index] = this.min + TDigest.interpolate(result[index], 1.0, this.min, this.weights[0] / 2.0, this.means[0]);
            ++index;
        }
        while (index < result.length && result[index] <= this.totalWeight - 1.0 && this.totalWeight - result[index] <= this.weights[this.centroidCount - 1] / 2.0 && this.weights[this.centroidCount - 1] / 2.0 > 1.0) {
            result[index] = this.max + TDigest.interpolate(this.totalWeight - result[index], 1.0, this.max, this.weights[this.centroidCount - 1] / 2.0, this.means[this.centroidCount - 1]);
            ++index;
        }
        if (index < result.length && result[index] >= this.totalWeight - 1.0) {
            Arrays.fill(result, index, result.length, this.max);
            return result;
        }
        double weightSoFar = this.weights[0] / 2.0;
        int currentCentroid = 0;
        while (index < result.length) {
            double delta = (this.weights[currentCentroid] + this.weights[currentCentroid + 1]) / 2.0;
            while (currentCentroid < this.centroidCount - 1 && weightSoFar + delta <= result[index]) {
                weightSoFar += delta;
                if (++currentCentroid >= this.centroidCount - 1) continue;
                delta = (this.weights[currentCentroid] + this.weights[currentCentroid + 1]) / 2.0;
            }
            if (currentCentroid == this.centroidCount - 1) {
                while (index < result.length && result[index] <= this.totalWeight - 1.0 && this.weights[this.centroidCount - 1] / 2.0 > 1.0) {
                    result[index] = this.max + TDigest.interpolate(this.totalWeight - result[index], 1.0, this.max, this.weights[this.centroidCount - 1] / 2.0, this.means[this.centroidCount - 1]);
                    ++index;
                }
                if (index < result.length) {
                    Arrays.fill(result, index, result.length, this.max);
                }
                return result;
            }
            if (this.weights[currentCentroid] == 1.0 && result[index] - weightSoFar < this.weights[currentCentroid] / 2.0) {
                result[index] = this.means[currentCentroid];
            } else if (this.weights[currentCentroid + 1] == 1.0 && result[index] - weightSoFar >= this.weights[currentCentroid] / 2.0) {
                result[index] = this.means[currentCentroid + 1];
            } else {
                double interpolationOffset = result[index] - weightSoFar;
                double interpolationSectionLength = delta;
                if (this.weights[currentCentroid] == 1.0) {
                    interpolationOffset -= this.weights[currentCentroid] / 2.0;
                    interpolationSectionLength = this.weights[currentCentroid + 1] / 2.0;
                } else if (this.weights[currentCentroid + 1] == 1.0) {
                    interpolationSectionLength = this.weights[currentCentroid] / 2.0;
                }
                result[index] = this.means[currentCentroid] + TDigest.interpolate(interpolationOffset, 0.0, this.means[currentCentroid], interpolationSectionLength, this.means[currentCentroid + 1]);
            }
            ++index;
        }
        return result;
    }

    private static void validateQuantilesArgument(double[] quantiles) {
        for (int i = 0; i < quantiles.length; ++i) {
            double quantile = quantiles[i];
            if (i > 0 && quantile < quantiles[i - 1]) {
                throw new IllegalArgumentException("quantiles must be sorted in increasing order");
            }
            if (!(quantile < 0.0) && !(quantile > 1.0)) continue;
            throw new IllegalArgumentException("quantiles should be in [0, 1] range");
        }
    }

    public Slice serialize() {
        this.merge(this.compression);
        return this.serializeInternal();
    }

    private Slice serializeInternal() {
        int i;
        Slice result = Slices.allocate((int)this.serializedSizeInBytes());
        SliceOutput output = result.getOutput();
        output.writeByte(0);
        output.writeDouble(this.min);
        output.writeDouble(this.max);
        output.writeDouble(this.compression);
        output.writeDouble(this.totalWeight);
        output.writeInt(this.centroidCount);
        for (i = 0; i < this.centroidCount; ++i) {
            output.writeDouble(this.means[i]);
        }
        for (i = 0; i < this.centroidCount; ++i) {
            output.writeDouble(this.weights[i]);
        }
        Preconditions.checkState((!output.isWritable() ? 1 : 0) != 0, (Object)"Expected serialized size doesn't match actual written size");
        return result;
    }

    public int serializedSizeInBytes() {
        return 37 + 8 * this.centroidCount + 8 * this.centroidCount;
    }

    public int estimatedInMemorySizeInBytes() {
        return (int)((long)T_DIGEST_SIZE + SizeOf.sizeOf((double[])this.means) + SizeOf.sizeOf((double[])this.weights) + SizeOf.sizeOf((double[])this.tempMeans) + SizeOf.sizeOf((double[])this.tempWeights) + SizeOf.sizeOf((int[])this.indexes));
    }

    private void merge(double compression) {
        if (this.centroidCount == 0) {
            return;
        }
        this.initializeIndexes();
        DoubleArrays.quickSortIndirect(this.indexes, this.means, 0, this.centroidCount);
        if (this.backwards) {
            Ints.reverse((int[])this.indexes, (int)0, (int)this.centroidCount);
        }
        double centroidMean = this.means[this.indexes[0]];
        double centroidWeight = this.weights[this.indexes[0]];
        if (this.tempMeans == null) {
            this.tempMeans = new double[1];
            this.tempWeights = new double[1];
        }
        int lastCentroid = 0;
        this.tempMeans[lastCentroid] = centroidMean;
        this.tempWeights[lastCentroid] = centroidWeight;
        double weightSoFar = 0.0;
        double normalizer = TDigest.normalizer(compression, this.totalWeight);
        double currentQuantile = 0.0;
        double currentQuantileMaxClusterSize = TDigest.maxRelativeClusterSize(currentQuantile, normalizer);
        for (int i = 1; i < this.centroidCount; ++i) {
            int index = this.indexes[i];
            double entryWeight = this.weights[index];
            double entryMean = this.means[index];
            double tentativeWeight = centroidWeight + entryWeight;
            double tentativeQuantile = Math.min((weightSoFar + tentativeWeight) / this.totalWeight, 1.0);
            double maxClusterWeight = this.totalWeight * Math.min(currentQuantileMaxClusterSize, TDigest.maxRelativeClusterSize(tentativeQuantile, normalizer));
            if (tentativeWeight <= maxClusterWeight) {
                centroidMean += (entryMean - centroidMean) * entryWeight / tentativeWeight;
                centroidWeight = tentativeWeight;
            } else {
                ++lastCentroid;
                currentQuantile = (weightSoFar += centroidWeight) / this.totalWeight;
                currentQuantileMaxClusterSize = TDigest.maxRelativeClusterSize(currentQuantile, normalizer);
                centroidWeight = entryWeight;
                centroidMean = entryMean;
            }
            this.ensureTempCapacity(lastCentroid);
            this.tempMeans[lastCentroid] = centroidMean;
            this.tempWeights[lastCentroid] = centroidWeight;
        }
        this.centroidCount = lastCentroid + 1;
        if (this.backwards) {
            Doubles.reverse((double[])this.tempMeans, (int)0, (int)this.centroidCount);
            Doubles.reverse((double[])this.tempWeights, (int)0, (int)this.centroidCount);
        }
        this.backwards = !this.backwards;
        System.arraycopy(this.tempMeans, 0, this.means, 0, this.centroidCount);
        System.arraycopy(this.tempWeights, 0, this.weights, 0, this.centroidCount);
    }

    @VisibleForTesting
    void forceMerge() {
        this.merge(TDigest.internalCompressionFactor(this.compression));
    }

    @VisibleForTesting
    int getCentroidCount() {
        return this.centroidCount;
    }

    private void mergeIfNeeded(double compression) {
        if (this.needsMerge) {
            this.merge(compression);
        }
    }

    private void ensureCapacity(int newSize) {
        if (this.means.length < newSize) {
            this.means = Arrays.copyOf(this.means, newSize);
            this.weights = Arrays.copyOf(this.weights, newSize);
        }
    }

    private void ensureTempCapacity(int capacity) {
        if (this.tempMeans.length <= capacity) {
            int newSize = capacity + (int)Math.ceil((double)capacity * 0.5);
            this.tempMeans = Arrays.copyOf(this.tempMeans, newSize);
            this.tempWeights = Arrays.copyOf(this.tempWeights, newSize);
        }
    }

    private void initializeIndexes() {
        if (this.indexes == null || this.indexes.length != this.means.length) {
            this.indexes = new int[this.means.length];
        }
        for (int i = 0; i < this.centroidCount; ++i) {
            this.indexes[i] = i;
        }
    }

    private static double interpolate(double x, double x0, double y0, double x1, double y1) {
        return (x - x0) / (x1 - x0) * (y1 - y0);
    }

    private static double maxRelativeClusterSize(double quantile, double normalizer) {
        return quantile * (1.0 - quantile) / normalizer;
    }

    private static double normalizer(double compression, double weight) {
        return compression / (4.0 * Math.log(weight / compression) + 24.0);
    }

    private static double internalCompressionFactor(double compression) {
        return 2.0 * compression;
    }
}

