/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.buffer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import com.google.common.base.Ticker;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.stats.TDigest;
import io.trino.memory.context.LocalMemoryContext;
import jakarta.annotation.Nullable;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

@ThreadSafe
class OutputBufferMemoryManager {
    private static final ListenableFuture<Void> NOT_BLOCKED = Futures.immediateVoidFuture();
    private final long maxBufferedBytes;
    private final AtomicLong bufferedBytes = new AtomicLong();
    private final AtomicLong peakMemoryUsage = new AtomicLong();
    @GuardedBy(value="this")
    private boolean closed;
    @Nullable
    private volatile SettableFuture<Void> bufferBlockedFuture;
    private volatile ListenableFuture<Void> blockedOnMemory = NOT_BLOCKED;
    private final Ticker ticker = Ticker.systemTicker();
    private final AtomicBoolean blockOnFull = new AtomicBoolean(true);
    private final Supplier<LocalMemoryContext> memoryContextSupplier;
    private final Executor notificationExecutor;
    @GuardedBy(value="this")
    private final TDigest bufferUtilization = new TDigest();
    @GuardedBy(value="this")
    private long lastBufferUtilizationRecordTime = -1L;
    @GuardedBy(value="this")
    private double lastBufferUtilization;

    public OutputBufferMemoryManager(long maxBufferedBytes, Supplier<LocalMemoryContext> memoryContextSupplier, Executor notificationExecutor) {
        Objects.requireNonNull(memoryContextSupplier, "memoryContextSupplier is null");
        Preconditions.checkArgument((maxBufferedBytes > 0L ? 1 : 0) != 0, (Object)"maxBufferedBytes must be > 0");
        this.maxBufferedBytes = maxBufferedBytes;
        this.memoryContextSupplier = Suppliers.memoize(memoryContextSupplier::get);
        this.notificationExecutor = Objects.requireNonNull(notificationExecutor, "notificationExecutor is null");
        this.lastBufferUtilization = 0.0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateMemoryUsage(long bytesAdded) {
        long currentBufferedBytes;
        LocalMemoryContext memoryContext = this.getMemoryContextOrNull();
        if (memoryContext == null) {
            return;
        }
        ListenableFuture waitForMemory = null;
        SettableFuture<Void> notifyUnblocked = null;
        OutputBufferMemoryManager outputBufferMemoryManager = this;
        synchronized (outputBufferMemoryManager) {
            if (this.closed) {
                return;
            }
            currentBufferedBytes = this.bufferedBytes.updateAndGet(bytes -> {
                long result = bytes + bytesAdded;
                Preconditions.checkArgument((result >= 0L ? 1 : 0) != 0, (String)"bufferedBytes (%s) plus delta (%s) would be negative", (long)bytes, (long)bytesAdded);
                return result;
            });
            ListenableFuture blockedOnMemory = memoryContext.setBytes(currentBufferedBytes);
            if (!blockedOnMemory.isDone()) {
                if (this.blockedOnMemory != blockedOnMemory) {
                    this.blockedOnMemory = blockedOnMemory;
                    waitForMemory = blockedOnMemory;
                }
            } else {
                this.blockedOnMemory = NOT_BLOCKED;
                if (currentBufferedBytes <= this.maxBufferedBytes || !this.blockOnFull.get()) {
                    notifyUnblocked = this.bufferBlockedFuture;
                    this.bufferBlockedFuture = null;
                }
            }
            this.recordBufferUtilization();
        }
        this.peakMemoryUsage.accumulateAndGet(currentBufferedBytes, Math::max);
        this.notifyListener(notifyUnblocked);
        if (waitForMemory != null) {
            waitForMemory.addListener(this::onMemoryAvailable, this.notificationExecutor);
        }
    }

    private synchronized void recordBufferUtilization() {
        long recordTime = this.ticker.read();
        if (this.lastBufferUtilizationRecordTime != -1L) {
            this.bufferUtilization.add(this.lastBufferUtilization, (double)recordTime - (double)this.lastBufferUtilizationRecordTime);
        }
        double utilization = this.getUtilization();
        if (this.lastBufferUtilizationRecordTime != -1L || utilization != 0.0) {
            this.lastBufferUtilizationRecordTime = recordTime;
            this.lastBufferUtilization = utilization;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ListenableFuture<Void> getBufferBlockedFuture() {
        SettableFuture<Void> bufferBlockedFuture = this.bufferBlockedFuture;
        if (bufferBlockedFuture == null) {
            if (this.blockedOnMemory.isDone() && !this.isBufferFull()) {
                return NOT_BLOCKED;
            }
            OutputBufferMemoryManager outputBufferMemoryManager = this;
            synchronized (outputBufferMemoryManager) {
                if (this.bufferBlockedFuture == null) {
                    if (this.blockedOnMemory.isDone() && !this.isBufferFull()) {
                        return NOT_BLOCKED;
                    }
                    this.bufferBlockedFuture = SettableFuture.create();
                }
                return this.bufferBlockedFuture;
            }
        }
        return bufferBlockedFuture;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setNoBlockOnFull() {
        SettableFuture<Void> future = null;
        OutputBufferMemoryManager outputBufferMemoryManager = this;
        synchronized (outputBufferMemoryManager) {
            this.blockOnFull.set(false);
            if (this.blockedOnMemory.isDone()) {
                future = this.bufferBlockedFuture;
                this.bufferBlockedFuture = null;
            }
        }
        this.notifyListener(future);
    }

    public long getBufferedBytes() {
        return this.bufferedBytes.get();
    }

    public double getUtilization() {
        return (double)this.bufferedBytes.get() / (double)this.maxBufferedBytes;
    }

    public synchronized TDigest getUtilizationHistogram() {
        this.recordBufferUtilization();
        return TDigest.copyOf((TDigest)this.bufferUtilization);
    }

    public boolean isOverutilized() {
        return this.isBufferFull();
    }

    private boolean isBufferFull() {
        return this.bufferedBytes.get() > this.maxBufferedBytes && this.blockOnFull.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void onMemoryAvailable() {
        SettableFuture<Void> future;
        if (this.isBufferFull()) {
            return;
        }
        OutputBufferMemoryManager outputBufferMemoryManager = this;
        synchronized (outputBufferMemoryManager) {
            if (this.isBufferFull() || !this.blockedOnMemory.isDone()) {
                return;
            }
            future = this.bufferBlockedFuture;
            this.bufferBlockedFuture = null;
        }
        this.notifyListener(future);
    }

    public long getPeakMemoryUsage() {
        return this.peakMemoryUsage.get();
    }

    public synchronized void close() {
        this.updateMemoryUsage(-this.bufferedBytes.get());
        LocalMemoryContext memoryContext = this.getMemoryContextOrNull();
        if (memoryContext != null) {
            memoryContext.close();
        }
        this.closed = true;
    }

    private void notifyListener(@Nullable SettableFuture<Void> future) {
        if (future != null) {
            this.notificationExecutor.execute(() -> future.set(null));
        }
    }

    @Nullable
    private LocalMemoryContext getMemoryContextOrNull() {
        try {
            return this.memoryContextSupplier.get();
        }
        catch (RuntimeException runtimeException) {
            return null;
        }
    }
}

