/*
 * Decompiled with CFR 0.152.
 */
package io.trino.tpch;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.trino.tpch.Distribution;
import io.trino.tpch.Distributions;
import io.trino.tpch.RandomInt;
import io.trino.tpch.SharingTextPoolSupplier;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Logger;

public class TextPool {
    private static final Logger log = Logger.getLogger(TextPool.class.getName());
    private static final int DEFAULT_TEXT_POOL_SIZE = 314572800;
    private static final int MAX_SENTENCE_LENGTH = 256;
    private static final ScheduledExecutorService EXECUTOR = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat(TextPool.class.getName() + "-%s").build());
    private static final Supplier<TextPool> DEFAULT_TEXT_POOL = new SharingTextPoolSupplier(EXECUTOR, TextPool::createDefaultTextPool);
    private final byte[] textPool;
    private final int textPoolSize;

    public static Supplier<TextPool> getDefaultTextPool() {
        return DEFAULT_TEXT_POOL;
    }

    private static TextPool createDefaultTextPool() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        TextPool pool = new TextPool(314572800, Distributions.getDefaultDistributions());
        log.fine(() -> String.format("Created new TextPool in %s ms", stopwatch.elapsed(TimeUnit.MILLISECONDS)));
        return pool;
    }

    public TextPool(int size, Distributions distributions) {
        this(size, distributions, progress -> {});
    }

    public TextPool(int size, Distributions distributions, TextGenerationProgressMonitor monitor) {
        Objects.requireNonNull(distributions, "distributions is null");
        Objects.requireNonNull(monitor, "monitor is null");
        ByteArrayBuilder output = new ByteArrayBuilder(size + 256);
        RandomInt randomInt = new RandomInt(933588178L, Integer.MAX_VALUE);
        while (output.getLength() < size) {
            TextPool.generateSentence(distributions, output, randomInt);
            monitor.updateProgress(Math.min(1.0 * (double)output.getLength() / (double)size, 1.0));
        }
        output.erase(output.getLength() - size);
        this.textPool = output.getBytes();
        this.textPoolSize = output.getLength();
    }

    public int size() {
        return this.textPoolSize;
    }

    public String getText(int begin, int end) {
        if (end > this.textPoolSize) {
            throw new IndexOutOfBoundsException(String.format("Index %d is beyond end of text pool (size = %d)", end, this.textPoolSize));
        }
        return new String(this.textPool, begin, end - begin, StandardCharsets.US_ASCII);
    }

    private static void generateSentence(Distributions distributions, ByteArrayBuilder builder, RandomInt random) {
        String syntax = distributions.getGrammars().randomValue(random);
        int maxLength = syntax.length();
        for (int i = 0; i < maxLength; i += 2) {
            switch (syntax.charAt(i)) {
                case 'V': {
                    TextPool.generateVerbPhrase(distributions, builder, random);
                    break;
                }
                case 'N': {
                    TextPool.generateNounPhrase(distributions, builder, random);
                    break;
                }
                case 'P': {
                    String preposition = distributions.getPrepositions().randomValue(random);
                    builder.append(preposition);
                    builder.append(" the ");
                    TextPool.generateNounPhrase(distributions, builder, random);
                    break;
                }
                case 'T': {
                    builder.erase(1);
                    String terminator = distributions.getTerminators().randomValue(random);
                    builder.append(terminator);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unknown token '" + syntax.charAt(i) + "'");
                }
            }
            if (builder.getLastChar() == ' ') continue;
            builder.append(" ");
        }
    }

    private static void generateVerbPhrase(Distributions distributions, ByteArrayBuilder builder, RandomInt random) {
        String syntax = distributions.getVerbPhrase().randomValue(random);
        int maxLength = syntax.length();
        for (int i = 0; i < maxLength; i += 2) {
            Distribution source = switch (syntax.charAt(i)) {
                case 'D' -> distributions.getAdverbs();
                case 'V' -> distributions.getVerbs();
                case 'X' -> distributions.getAuxiliaries();
                default -> throw new IllegalArgumentException("Unknown token '" + syntax.charAt(i) + "'");
            };
            String word = source.randomValue(random);
            builder.append(word);
            builder.append(" ");
        }
    }

    private static void generateNounPhrase(Distributions distributions, ByteArrayBuilder builder, RandomInt random) {
        String syntax = distributions.getNounPhrase().randomValue(random);
        int maxLength = syntax.length();
        for (int i = 0; i < maxLength; ++i) {
            Distribution source = switch (syntax.charAt(i)) {
                case 'A' -> distributions.getArticles();
                case 'J' -> distributions.getAdjectives();
                case 'D' -> distributions.getAdverbs();
                case 'N' -> distributions.getNouns();
                case ',' -> {
                    builder.erase(1);
                    builder.append(", ");
                    yield null;
                }
                case ' ' -> null;
                default -> throw new IllegalArgumentException("Unknown token '" + syntax.charAt(i) + "'");
            };
            if (source == null) continue;
            builder.append(source.randomValue(random));
            builder.append(" ");
        }
    }

    public static interface TextGenerationProgressMonitor {
        public void updateProgress(double var1);
    }

    private static class ByteArrayBuilder {
        private int length;
        private final byte[] bytes;

        public ByteArrayBuilder(int size) {
            this.bytes = new byte[size];
        }

        public void append(String string) {
            string.getBytes(0, string.length(), this.bytes, this.length);
            this.length += string.length();
        }

        public void erase(int count) {
            Preconditions.checkState((this.length >= count ? 1 : 0) != 0, (Object)"not enough bytes to erase");
            this.length -= count;
        }

        public int getLength() {
            return this.length;
        }

        public byte[] getBytes() {
            return this.bytes;
        }

        public char getLastChar() {
            return (char)this.bytes[this.length - 1];
        }
    }
}

