/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.store;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.RandomAccessInput;
import org.apache.lucene.util.GroupVIntUtil;

public abstract class BufferedIndexInput
extends IndexInput
implements RandomAccessInput {
    private static final ByteBuffer EMPTY_BYTEBUFFER = ByteBuffer.allocate(0).order(ByteOrder.LITTLE_ENDIAN);
    public static final int BUFFER_SIZE = 1024;
    public static final int MIN_BUFFER_SIZE = 8;
    public static final int MERGE_BUFFER_SIZE = 4096;
    private final int bufferSize;
    private ByteBuffer buffer = EMPTY_BYTEBUFFER;
    private long bufferStart = 0L;

    @Override
    public final byte readByte() throws IOException {
        if (!this.buffer.hasRemaining()) {
            this.refill();
        }
        return this.buffer.get();
    }

    public BufferedIndexInput(String resourceDesc) {
        this(resourceDesc, 1024);
    }

    public BufferedIndexInput(String resourceDesc, IOContext context) {
        this(resourceDesc, BufferedIndexInput.bufferSize(context));
    }

    public BufferedIndexInput(String resourceDesc, int bufferSize) {
        super(resourceDesc);
        this.checkBufferSize(bufferSize);
        this.bufferSize = bufferSize;
    }

    public final int getBufferSize() {
        return this.bufferSize;
    }

    private void checkBufferSize(int bufferSize) {
        if (bufferSize < 8) {
            throw new IllegalArgumentException("bufferSize must be at least MIN_BUFFER_SIZE (got " + bufferSize + ")");
        }
    }

    @Override
    public final void readBytes(byte[] b, int offset, int len) throws IOException {
        this.readBytes(b, offset, len, true);
    }

    @Override
    public final void readBytes(byte[] b, int offset, int len, boolean useBuffer) throws IOException {
        int available = this.buffer.remaining();
        if (len <= available) {
            if (len > 0) {
                this.buffer.get(b, offset, len);
            }
        } else {
            if (available > 0) {
                this.buffer.get(b, offset, available);
                offset += available;
                len -= available;
            }
            if (useBuffer && len < this.bufferSize) {
                this.refill();
                if (this.buffer.remaining() < len) {
                    this.buffer.get(b, offset, this.buffer.remaining());
                    throw new EOFException("read past EOF: " + String.valueOf(this));
                }
                this.buffer.get(b, offset, len);
            } else {
                long after = this.bufferStart + (long)this.buffer.position() + (long)len;
                if (after > this.length()) {
                    throw new EOFException("read past EOF: " + String.valueOf(this));
                }
                this.readInternal(ByteBuffer.wrap(b, offset, len));
                this.bufferStart = after;
                this.buffer.limit(0);
            }
        }
    }

    @Override
    public final short readShort() throws IOException {
        if (2 <= this.buffer.remaining()) {
            return this.buffer.getShort();
        }
        return super.readShort();
    }

    @Override
    public final int readInt() throws IOException {
        if (4 <= this.buffer.remaining()) {
            return this.buffer.getInt();
        }
        return super.readInt();
    }

    @Override
    public void readGroupVInt(long[] dst, int offset) throws IOException {
        int len = GroupVIntUtil.readGroupVInt(this, this.buffer.remaining(), p -> this.buffer.getInt((int)p), this.buffer.position(), dst, offset);
        if (len > 0) {
            this.buffer.position(this.buffer.position() + len);
        }
    }

    @Override
    public final long readLong() throws IOException {
        if (8 <= this.buffer.remaining()) {
            return this.buffer.getLong();
        }
        return super.readLong();
    }

    @Override
    public void readFloats(float[] dst, int offset, int len) throws IOException {
        int remainingDst = len;
        while (remainingDst > 0) {
            int cnt = Math.min(this.buffer.remaining() / 4, remainingDst);
            this.buffer.asFloatBuffer().get(dst, offset + len - remainingDst, cnt);
            this.buffer.position(this.buffer.position() + 4 * cnt);
            if ((remainingDst -= cnt) <= 0) continue;
            if (this.buffer.hasRemaining()) {
                dst[offset + len - remainingDst] = Float.intBitsToFloat(this.readInt());
                --remainingDst;
                continue;
            }
            this.refill();
        }
    }

    @Override
    public void readLongs(long[] dst, int offset, int len) throws IOException {
        int remainingDst = len;
        while (remainingDst > 0) {
            int cnt = Math.min(this.buffer.remaining() / 8, remainingDst);
            this.buffer.asLongBuffer().get(dst, offset + len - remainingDst, cnt);
            this.buffer.position(this.buffer.position() + 8 * cnt);
            if ((remainingDst -= cnt) <= 0) continue;
            if (this.buffer.hasRemaining()) {
                dst[offset + len - remainingDst] = this.readLong();
                --remainingDst;
                continue;
            }
            this.refill();
        }
    }

    @Override
    public void readInts(int[] dst, int offset, int len) throws IOException {
        int remainingDst = len;
        while (remainingDst > 0) {
            int cnt = Math.min(this.buffer.remaining() / 4, remainingDst);
            this.buffer.asIntBuffer().get(dst, offset + len - remainingDst, cnt);
            this.buffer.position(this.buffer.position() + 4 * cnt);
            if ((remainingDst -= cnt) <= 0) continue;
            if (this.buffer.hasRemaining()) {
                dst[offset + len - remainingDst] = this.readInt();
                --remainingDst;
                continue;
            }
            this.refill();
        }
    }

    private long resolvePositionInBuffer(long pos, int width) throws IOException {
        long index = pos - this.bufferStart;
        if (index >= 0L && index <= (long)(this.buffer.limit() - width)) {
            return index;
        }
        if (index < 0L) {
            this.bufferStart = Math.max(this.bufferStart - (long)this.bufferSize, pos + (long)width - (long)this.bufferSize);
            this.bufferStart = Math.max(this.bufferStart, 0L);
            this.bufferStart = Math.min(this.bufferStart, pos);
        } else {
            this.bufferStart = pos;
        }
        this.buffer.limit(0);
        this.seekInternal(this.bufferStart);
        this.refill();
        return pos - this.bufferStart;
    }

    @Override
    public final byte readByte(long pos) throws IOException {
        long index = this.resolvePositionInBuffer(pos, 1);
        return this.buffer.get((int)index);
    }

    @Override
    public void readBytes(long pos, byte[] bytes, int offset, int len) throws IOException {
        if (len <= this.bufferSize) {
            if (len > 0) {
                long index = this.resolvePositionInBuffer(pos, len);
                this.buffer.get((int)index, bytes, offset, len);
            }
        } else {
            long index;
            while (len > this.bufferSize) {
                index = this.resolvePositionInBuffer(pos, this.bufferSize);
                this.buffer.get((int)index, bytes, offset, this.bufferSize);
                len -= this.bufferSize;
                offset += this.bufferSize;
                pos += (long)this.bufferSize;
            }
            index = this.resolvePositionInBuffer(pos, len);
            this.buffer.get((int)index, bytes, offset, len);
        }
    }

    @Override
    public final short readShort(long pos) throws IOException {
        long index = this.resolvePositionInBuffer(pos, 2);
        return this.buffer.getShort((int)index);
    }

    @Override
    public final int readInt(long pos) throws IOException {
        long index = this.resolvePositionInBuffer(pos, 4);
        return this.buffer.getInt((int)index);
    }

    @Override
    public final long readLong(long pos) throws IOException {
        long index = this.resolvePositionInBuffer(pos, 8);
        return this.buffer.getLong((int)index);
    }

    private void refill() throws IOException {
        int newLength;
        long start = this.bufferStart + (long)this.buffer.position();
        long end = start + (long)this.bufferSize;
        if (end > this.length()) {
            end = this.length();
        }
        if ((newLength = (int)(end - start)) <= 0) {
            throw new EOFException("read past EOF: " + String.valueOf(this));
        }
        if (this.buffer == EMPTY_BYTEBUFFER) {
            this.buffer = ByteBuffer.allocate(this.bufferSize).order(ByteOrder.LITTLE_ENDIAN);
            this.seekInternal(this.bufferStart);
        }
        this.buffer.position(0);
        this.buffer.limit(newLength);
        this.bufferStart = start;
        this.readInternal(this.buffer);
        assert (this.buffer.order() == ByteOrder.LITTLE_ENDIAN) : this.buffer.order();
        assert (this.buffer.remaining() == 0) : "should have thrown EOFException";
        assert (this.buffer.position() == newLength);
        this.buffer.flip();
    }

    protected abstract void readInternal(ByteBuffer var1) throws IOException;

    @Override
    public final long getFilePointer() {
        return this.bufferStart + (long)this.buffer.position();
    }

    @Override
    public final void seek(long pos) throws IOException {
        if (pos >= this.bufferStart && pos < this.bufferStart + (long)this.buffer.limit()) {
            this.buffer.position((int)(pos - this.bufferStart));
        } else {
            this.bufferStart = pos;
            this.buffer.limit(0);
            this.seekInternal(pos);
        }
    }

    protected abstract void seekInternal(long var1) throws IOException;

    @Override
    public BufferedIndexInput clone() {
        BufferedIndexInput clone = (BufferedIndexInput)super.clone();
        clone.buffer = EMPTY_BYTEBUFFER;
        clone.bufferStart = this.getFilePointer();
        return clone;
    }

    @Override
    public IndexInput slice(String sliceDescription, long offset, long length) throws IOException {
        return BufferedIndexInput.wrap(sliceDescription, this, offset, length);
    }

    public static int bufferSize(IOContext context) {
        switch (context.context()) {
            case MERGE: {
                return 4096;
            }
        }
        return 1024;
    }

    public static BufferedIndexInput wrap(String sliceDescription, IndexInput other, long offset, long length) {
        return new SlicedIndexInput(sliceDescription, other, offset, length);
    }

    private static final class SlicedIndexInput
    extends BufferedIndexInput {
        IndexInput base;
        long fileOffset;
        long length;

        SlicedIndexInput(String sliceDescription, IndexInput base, long offset, long length) {
            super((String)(sliceDescription == null ? base.toString() : base.toString() + " [slice=" + sliceDescription + "]"), 1024);
            if (offset < 0L || length < 0L || offset + length > base.length()) {
                throw new IllegalArgumentException("slice() " + sliceDescription + " out of bounds: " + String.valueOf(base));
            }
            this.base = base.clone();
            this.fileOffset = offset;
            this.length = length;
        }

        @Override
        public SlicedIndexInput clone() {
            SlicedIndexInput clone = (SlicedIndexInput)super.clone();
            clone.base = this.base.clone();
            clone.fileOffset = this.fileOffset;
            clone.length = this.length;
            return clone;
        }

        @Override
        protected void readInternal(ByteBuffer b) throws IOException {
            long start = this.getFilePointer();
            if (start + (long)b.remaining() > this.length) {
                throw new EOFException("read past EOF: " + String.valueOf(this));
            }
            this.base.seek(this.fileOffset + start);
            this.base.readBytes(b.array(), b.position(), b.remaining());
            b.position(b.position() + b.remaining());
        }

        @Override
        protected void seekInternal(long pos) {
        }

        @Override
        public void close() throws IOException {
            this.base.close();
        }

        @Override
        public long length() {
            return this.length;
        }
    }
}

